/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 *
 */

/*
 * Created on Sep 7, 2004
 */
package org.apache.jmeter.save.converters;

import java.util.ArrayList;
import java.util.Collection;

import org.apache.jmeter.reporters.ResultCollectorHelper;
import org.apache.jmeter.samplers.SampleResult;
import org.apache.jmeter.save.SaveService;
import org.apache.jmeter.save.TestResultWrapper;

import com.thoughtworks.xstream.converters.MarshallingContext;
import com.thoughtworks.xstream.converters.UnmarshallingContext;
import com.thoughtworks.xstream.converters.collections.AbstractCollectionConverter;
import com.thoughtworks.xstream.io.HierarchicalStreamReader;
import com.thoughtworks.xstream.io.HierarchicalStreamWriter;
import com.thoughtworks.xstream.mapper.Mapper;

/**
 * XStream Class to convert TestResultWrapper
 *
 */
public class TestResultWrapperConverter extends AbstractCollectionConverter {

    /**
     * Returns the converter version; used to check for possible
     * incompatibilities
     *
     * @return the version of this converter
     */
    public static String getVersion() {
        return "$Revision: 1697162 $";  //$NON-NLS-1$
    }

    /**
     * @param arg0 the {@link Mapper} to be used
     */
    public TestResultWrapperConverter(Mapper arg0) {
        super(arg0);
    }

    /** {@inheritDoc} */
    @Override
    public boolean canConvert(@SuppressWarnings("rawtypes") Class arg0) { // superclass does not use types
        return TestResultWrapper.class.equals(arg0);
    }

    /** {@inheritDoc} */
    @Override
    public void marshal(Object arg0, HierarchicalStreamWriter arg1, MarshallingContext arg2) {
        // Not used, as the <testResult> element is generated by the
        // ResultCollector class
    }

    /**
     * Read test results from JTL files and pass them to the visualiser directly.
     * If the ResultCollector helper object is defined, then pass the samples to that
     * rather than adding them to the test result wrapper.
     * 
     * @return the test result wrapper (may be empty)
     *
     * @see com.thoughtworks.xstream.converters.Converter#unmarshal(com.thoughtworks.xstream.io.HierarchicalStreamReader,
     *      com.thoughtworks.xstream.converters.UnmarshallingContext)
     */
    @Override
    public Object unmarshal(HierarchicalStreamReader reader, UnmarshallingContext context) {
        TestResultWrapper results = new TestResultWrapper();
        Collection<SampleResult> samples = new ArrayList<>();
        String ver = reader.getAttribute("version");  //$NON-NLS-1$
        if (ver == null || ver.length() == 0) {
            ver = "1.0";  //$NON-NLS-1$
        }
        results.setVersion(ver);
        ConversionHelp.setInVersion(ver);// Make sure decoding follows input file
        final ResultCollectorHelper resultCollectorHelper = (ResultCollectorHelper) context.get(SaveService.RESULTCOLLECTOR_HELPER_OBJECT);
        while (reader.hasMoreChildren()) {
            reader.moveDown();
            SampleResult sample = (SampleResult) readItem(reader, context, results);
            if (resultCollectorHelper != null) {
                resultCollectorHelper.add(sample);
            } else {
                samples.add(sample);
            }
            reader.moveUp();
        }
        results.setSampleResults(samples);
        return results;
    }
}
