/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.driver.internal.cluster;

import java.net.UnknownHostException;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.neo4j.driver.internal.BoltServerAddress;
import org.neo4j.driver.internal.cluster.ClusterComposition;
import org.neo4j.driver.internal.cluster.ClusterCompositionProvider;
import org.neo4j.driver.internal.cluster.ClusterCompositionResponse;
import org.neo4j.driver.internal.cluster.RoutingSettings;
import org.neo4j.driver.internal.cluster.RoutingTable;
import org.neo4j.driver.internal.shaded.io.netty.util.concurrent.EventExecutorGroup;
import org.neo4j.driver.internal.spi.Connection;
import org.neo4j.driver.internal.spi.ConnectionPool;
import org.neo4j.driver.internal.util.Futures;
import org.neo4j.driver.v1.Logger;
import org.neo4j.driver.v1.exceptions.SecurityException;
import org.neo4j.driver.v1.exceptions.ServiceUnavailableException;
import org.neo4j.driver.v1.net.ServerAddressResolver;

public class Rediscovery {
    private static final String NO_ROUTERS_AVAILABLE = "Could not perform discovery. No routing servers available.";
    private final BoltServerAddress initialRouter;
    private final RoutingSettings settings;
    private final Logger logger;
    private final ClusterCompositionProvider provider;
    private final ServerAddressResolver resolver;
    private final EventExecutorGroup eventExecutorGroup;
    private volatile boolean useInitialRouter;

    public Rediscovery(BoltServerAddress initialRouter, RoutingSettings settings, ClusterCompositionProvider provider, EventExecutorGroup eventExecutorGroup, ServerAddressResolver resolver, Logger logger) {
        this(initialRouter, settings, provider, resolver, eventExecutorGroup, logger, true);
    }

    Rediscovery(BoltServerAddress initialRouter, RoutingSettings settings, ClusterCompositionProvider provider, ServerAddressResolver resolver, EventExecutorGroup eventExecutorGroup, Logger logger, boolean useInitialRouter) {
        this.initialRouter = initialRouter;
        this.settings = settings;
        this.logger = logger;
        this.provider = provider;
        this.resolver = resolver;
        this.eventExecutorGroup = eventExecutorGroup;
        this.useInitialRouter = useInitialRouter;
    }

    public CompletionStage<ClusterComposition> lookupClusterComposition(RoutingTable routingTable, ConnectionPool connectionPool) {
        CompletableFuture<ClusterComposition> result = new CompletableFuture<ClusterComposition>();
        this.lookupClusterComposition(routingTable, connectionPool, 0, 0L, result);
        return result;
    }

    private void lookupClusterComposition(RoutingTable routingTable, ConnectionPool pool, int failures, long previousDelay, CompletableFuture<ClusterComposition> result) {
        this.lookup(routingTable, pool).whenComplete((composition, completionError) -> {
            Throwable error = Futures.completionExceptionCause(completionError);
            if (error != null) {
                result.completeExceptionally(error);
            } else if (composition != null) {
                result.complete((ClusterComposition)composition);
            } else {
                int newFailures = failures + 1;
                if (newFailures >= this.settings.maxRoutingFailures()) {
                    result.completeExceptionally(new ServiceUnavailableException(NO_ROUTERS_AVAILABLE));
                } else {
                    long nextDelay = Math.max(this.settings.retryTimeoutDelay(), previousDelay * 2L);
                    this.logger.info("Unable to fetch new routing table, will try again in " + nextDelay + "ms", new Object[0]);
                    this.eventExecutorGroup.next().schedule(() -> this.lookupClusterComposition(routingTable, pool, newFailures, nextDelay, result), nextDelay, TimeUnit.MILLISECONDS);
                }
            }
        });
    }

    private CompletionStage<ClusterComposition> lookup(RoutingTable routingTable, ConnectionPool connectionPool) {
        CompletionStage<ClusterComposition> compositionStage;
        if (this.useInitialRouter) {
            compositionStage = this.lookupOnInitialRouterThenOnKnownRouters(routingTable, connectionPool);
            this.useInitialRouter = false;
        } else {
            compositionStage = this.lookupOnKnownRoutersThenOnInitialRouter(routingTable, connectionPool);
        }
        return compositionStage.whenComplete((composition, error) -> {
            if (composition != null && !composition.hasWriters()) {
                this.useInitialRouter = true;
            }
        });
    }

    private CompletionStage<ClusterComposition> lookupOnKnownRoutersThenOnInitialRouter(RoutingTable routingTable, ConnectionPool connectionPool) {
        HashSet<BoltServerAddress> seenServers = new HashSet<BoltServerAddress>();
        return this.lookupOnKnownRouters(routingTable, connectionPool, seenServers).thenCompose(composition -> {
            if (composition != null) {
                return CompletableFuture.completedFuture(composition);
            }
            return this.lookupOnInitialRouter(routingTable, connectionPool, seenServers);
        });
    }

    private CompletionStage<ClusterComposition> lookupOnInitialRouterThenOnKnownRouters(RoutingTable routingTable, ConnectionPool connectionPool) {
        Set<BoltServerAddress> seenServers = Collections.emptySet();
        return this.lookupOnInitialRouter(routingTable, connectionPool, seenServers).thenCompose(composition -> {
            if (composition != null) {
                return CompletableFuture.completedFuture(composition);
            }
            return this.lookupOnKnownRouters(routingTable, connectionPool, new HashSet<BoltServerAddress>());
        });
    }

    private CompletionStage<ClusterComposition> lookupOnKnownRouters(RoutingTable routingTable, ConnectionPool connectionPool, Set<BoltServerAddress> seenServers) {
        BoltServerAddress[] addresses = routingTable.routers().toArray();
        CompletionStage<ClusterComposition> result = Futures.completedWithNull();
        for (BoltServerAddress address : addresses) {
            result = result.thenCompose(composition -> {
                if (composition != null) {
                    return CompletableFuture.completedFuture(composition);
                }
                return this.lookupOnRouter(address, routingTable, connectionPool).whenComplete((ignore, error) -> seenServers.add(address));
            });
        }
        return result;
    }

    private CompletionStage<ClusterComposition> lookupOnInitialRouter(RoutingTable routingTable, ConnectionPool connectionPool, Set<BoltServerAddress> seenServers) {
        List<BoltServerAddress> addresses;
        try {
            addresses = this.resolve(this.initialRouter);
        }
        catch (Throwable error) {
            return Futures.failedFuture(error);
        }
        addresses.removeAll(seenServers);
        CompletionStage<ClusterComposition> result = Futures.completedWithNull();
        for (BoltServerAddress address : addresses) {
            result = result.thenCompose(composition -> {
                if (composition != null) {
                    return CompletableFuture.completedFuture(composition);
                }
                return this.lookupOnRouter(address, routingTable, connectionPool);
            });
        }
        return result;
    }

    private CompletionStage<ClusterComposition> lookupOnRouter(BoltServerAddress routerAddress, RoutingTable routingTable, ConnectionPool connectionPool) {
        CompletionStage<Connection> connectionStage = connectionPool.acquire(routerAddress);
        return this.provider.getClusterComposition(connectionStage).handle((response, error) -> {
            Throwable cause = Futures.completionExceptionCause(error);
            if (cause != null) {
                return this.handleRoutingProcedureError(cause, routingTable, routerAddress);
            }
            return this.handleClusterComposition(routerAddress, (ClusterCompositionResponse)response);
        });
    }

    private ClusterComposition handleRoutingProcedureError(Throwable error, RoutingTable routingTable, BoltServerAddress routerAddress) {
        if (error instanceof SecurityException) {
            throw new CompletionException(error);
        }
        this.logger.info(String.format("Failed to connect to routing server '%s'.", routerAddress), error);
        routingTable.forget(routerAddress);
        return null;
    }

    private ClusterComposition handleClusterComposition(BoltServerAddress routerAddress, ClusterCompositionResponse response) {
        ClusterComposition result = null;
        try {
            result = response.clusterComposition();
        }
        catch (Exception exc) {
            this.logger.warn(String.format("Unable to process routing table received from '%s'.", routerAddress), exc);
        }
        return result;
    }

    private List<BoltServerAddress> resolve(BoltServerAddress address) {
        return this.resolver.resolve(address).stream().flatMap(resolved -> this.resolveAll(BoltServerAddress.from(resolved))).collect(Collectors.toList());
    }

    private Stream<BoltServerAddress> resolveAll(BoltServerAddress address) {
        try {
            return address.resolveAll().stream();
        }
        catch (UnknownHostException e) {
            this.logger.error("Failed to resolve address `" + address + "` to IPs due to error: " + e.getMessage(), e);
            return Stream.of(address);
        }
    }
}

