/*
 * Decompiled with CFR 0.152.
 */
package org.apache.james.dnsservice.dnsjava;

import com.google.common.collect.ImmutableList;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import javax.annotation.PostConstruct;
import javax.inject.Inject;
import org.apache.commons.configuration.ConfigurationException;
import org.apache.commons.configuration.HierarchicalConfiguration;
import org.apache.james.dnsservice.api.DNSService;
import org.apache.james.dnsservice.api.DNSServiceMBean;
import org.apache.james.dnsservice.api.TemporaryResolutionException;
import org.apache.james.lifecycle.api.Configurable;
import org.apache.james.lifecycle.api.LogEnabled;
import org.apache.james.metrics.api.MetricFactory;
import org.apache.james.metrics.api.TimeMetric;
import org.slf4j.Logger;
import org.xbill.DNS.ARecord;
import org.xbill.DNS.Address;
import org.xbill.DNS.Cache;
import org.xbill.DNS.ExtendedResolver;
import org.xbill.DNS.Lookup;
import org.xbill.DNS.MXRecord;
import org.xbill.DNS.Name;
import org.xbill.DNS.PTRRecord;
import org.xbill.DNS.Record;
import org.xbill.DNS.Resolver;
import org.xbill.DNS.ResolverConfig;
import org.xbill.DNS.ReverseMap;
import org.xbill.DNS.TXTRecord;
import org.xbill.DNS.TextParseException;

public class DNSJavaService
implements DNSService,
DNSServiceMBean,
LogEnabled,
Configurable {
    protected Resolver resolver;
    protected Cache cache;
    private int maxCacheSize = 50000;
    private int dnsCredibility;
    private final List<String> dnsServers = new ArrayList<String>();
    private final MetricFactory metricFactory;
    private Name[] searchPaths = null;
    private final Comparator<MXRecord> mxComparator = new MXRecordComparator();
    private boolean setAsDNSJavaDefault;
    private String localHostName;
    private String localCanonicalHostName;
    private String localAddress;
    private Logger logger;

    public void setLog(Logger logger) {
        this.logger = logger;
    }

    @Inject
    public DNSJavaService(MetricFactory metricFactory) {
        this.metricFactory = metricFactory;
    }

    public void configure(HierarchicalConfiguration configuration) throws ConfigurationException {
        boolean authoritative;
        boolean autodiscover = configuration.getBoolean("autodiscover", true);
        ArrayList<Name> sPaths = new ArrayList<Name>();
        if (autodiscover) {
            Name[] systemSearchPath;
            this.logger.info("Autodiscovery is enabled - trying to discover your system's DNS Servers");
            String[] serversArray = ResolverConfig.getCurrentConfig().servers();
            if (serversArray != null) {
                String[] stringArray = serversArray;
                int n = stringArray.length;
                for (int i = 0; i < n; ++i) {
                    String aServersArray = stringArray[i];
                    this.dnsServers.add(aServersArray);
                    this.logger.info("Adding autodiscovered server " + aServersArray);
                }
            }
            if ((systemSearchPath = ResolverConfig.getCurrentConfig().searchPath()) != null && systemSearchPath.length > 0) {
                sPaths.addAll(Arrays.asList(systemSearchPath));
            }
            if (this.logger.isInfoEnabled()) {
                for (Name searchPath : sPaths) {
                    this.logger.info("Adding autodiscovered search path " + searchPath.toString());
                }
            }
        }
        this.setAsDNSJavaDefault = configuration.getBoolean("setAsDNSJavaDefault", true);
        Collections.addAll(this.dnsServers, configuration.getStringArray("servers.server"));
        for (String aSearchPathsConfiguration : configuration.getStringArray("searchpaths.searchpath")) {
            try {
                sPaths.add(Name.fromString((String)aSearchPathsConfiguration));
            }
            catch (TextParseException e) {
                throw new ConfigurationException("Unable to parse searchpath host: " + aSearchPathsConfiguration, (Throwable)e);
            }
        }
        this.searchPaths = sPaths.toArray(new Name[sPaths.size()]);
        if (this.dnsServers.isEmpty()) {
            this.logger.info("No DNS servers have been specified or found by autodiscovery - adding 127.0.0.1");
            this.dnsServers.add("127.0.0.1");
        }
        this.dnsCredibility = (authoritative = configuration.getBoolean("authoritative", false)) ? 4 : 3;
        this.maxCacheSize = configuration.getInt("maxcachesize", this.maxCacheSize);
    }

    @PostConstruct
    public void init() throws Exception {
        this.logger.debug("DNSService init...");
        if (this.dnsServers.isEmpty()) {
            try {
                this.dnsServers.add(InetAddress.getLocalHost().getHostName());
            }
            catch (UnknownHostException ue) {
                this.dnsServers.add("127.0.0.1");
            }
        }
        String[] serversArray = this.dnsServers.toArray(new String[this.dnsServers.size()]);
        if (this.logger.isInfoEnabled()) {
            for (String aServersArray : serversArray) {
                this.logger.info("DNS Server is: " + aServersArray);
            }
        }
        try {
            this.resolver = new ExtendedResolver(serversArray);
        }
        catch (UnknownHostException uhe) {
            this.logger.error("DNS service could not be initialized.  The DNS servers specified are not recognized hosts.", (Throwable)uhe);
            throw uhe;
        }
        this.cache = new Cache(1);
        this.cache.setMaxEntries(this.maxCacheSize);
        if (this.setAsDNSJavaDefault) {
            Lookup.setDefaultResolver((Resolver)this.resolver);
            Lookup.setDefaultCache((Cache)this.cache, (int)1);
            Lookup.setDefaultSearchPath((Name[])this.searchPaths);
            this.logger.info("Registered cache, resolver and search paths as DNSJava defaults");
        }
        InetAddress addr = this.getLocalHost();
        this.localCanonicalHostName = addr.getCanonicalHostName();
        this.localHostName = addr.getHostName();
        this.localAddress = addr.getHostAddress();
        this.logger.debug("DNSService ...init end");
    }

    public String[] getDNSServers() {
        return this.dnsServers.toArray(new String[this.dnsServers.size()]);
    }

    public Name[] getSearchPaths() {
        return this.searchPaths;
    }

    private List<String> findMXRecordsRaw(String hostname) throws TemporaryResolutionException {
        Record[] answers = this.lookup(hostname, 15, "MX");
        ArrayList<String> servers = new ArrayList<String>();
        if (answers == null) {
            return servers;
        }
        MXRecord[] mxAnswers = new MXRecord[answers.length];
        for (int i = 0; i < answers.length; ++i) {
            mxAnswers[i] = (MXRecord)answers[i];
        }
        Arrays.sort(mxAnswers, this.mxComparator);
        int currentPrio = -1;
        ArrayList<String> samePrio = new ArrayList<String>();
        for (int i = 0; i < mxAnswers.length; ++i) {
            boolean same = false;
            boolean lastItem = i + 1 == mxAnswers.length;
            MXRecord mx = mxAnswers[i];
            if (i == 0) {
                currentPrio = mx.getPriority();
            } else {
                same = currentPrio == mx.getPriority();
            }
            String mxRecord = mx.getTarget().toString();
            if (same) {
                samePrio.add(mxRecord);
            } else {
                Collections.shuffle(samePrio);
                servers.addAll(samePrio);
                samePrio.clear();
                samePrio.add(mxRecord);
            }
            if (lastItem) {
                Collections.shuffle(samePrio);
                servers.addAll(samePrio);
            }
            this.logger.debug("Found MX record " + mxRecord);
        }
        return servers;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public Collection<String> findMXRecords(String hostname) throws TemporaryResolutionException {
        TimeMetric timeMetric = this.metricFactory.timer("findMXRecords");
        List<Object> servers = new ArrayList();
        try {
            servers = this.findMXRecordsRaw(hostname);
            Collection<String> collection = Collections.unmodifiableCollection(servers);
            return collection;
        }
        finally {
            if (servers.size() == 0) {
                StringBuffer logBuffer = new StringBuffer(128).append("Couldn't resolve MX records for domain ").append(hostname).append(".");
                this.logger.info(logBuffer.toString());
                try {
                    this.getByName(hostname);
                    servers.add(hostname);
                }
                catch (UnknownHostException uhe) {
                    logBuffer = new StringBuffer(128).append("Couldn't resolve IP address for host ").append(hostname).append(".");
                    this.logger.error(logBuffer.toString());
                }
            }
            timeMetric.stopAndPublish();
        }
    }

    protected Record[] lookup(String namestr, int type, String typeDesc) throws TemporaryResolutionException {
        try {
            Lookup l = new Lookup(namestr, type);
            l.setCache(this.cache);
            l.setResolver(this.resolver);
            l.setCredibility(this.dnsCredibility);
            l.setSearchPath(this.searchPaths);
            Record[] r = l.run();
            try {
                if (l.getResult() == 2) {
                    throw new TemporaryResolutionException("DNSService is temporary not reachable");
                }
                return r;
            }
            catch (IllegalStateException ise) {
                this.logger.debug("Error determining result ", (Throwable)ise);
                throw new TemporaryResolutionException("DNSService is temporary not reachable");
            }
        }
        catch (TextParseException tpe) {
            this.logger.error("Couldn't parse name " + namestr, (Throwable)tpe);
            return null;
        }
    }

    protected Record[] lookupNoException(String namestr, int type, String typeDesc) {
        try {
            return this.lookup(namestr, type, typeDesc);
        }
        catch (TemporaryResolutionException e) {
            return null;
        }
    }

    private static String allowIPLiteral(String host) {
        String possible_ip_literal;
        if (host.charAt(host.length() - 1) == '.' && Address.isDottedQuad((String)(possible_ip_literal = host.substring(0, host.length() - 1)))) {
            host = possible_ip_literal;
        }
        return host;
    }

    public InetAddress getByName(String host) throws UnknownHostException {
        TimeMetric timeMetric = this.metricFactory.timer("getByName");
        String name = DNSJavaService.allowIPLiteral(host);
        try {
            if (name.equalsIgnoreCase(this.localHostName) || name.equalsIgnoreCase(this.localCanonicalHostName) || name.equals(this.localAddress)) {
                InetAddress inetAddress = this.getLocalHost();
                return inetAddress;
            }
            InetAddress inetAddress = Address.getByAddress((String)name);
            return inetAddress;
        }
        catch (UnknownHostException e) {
            Record[] records = this.lookupNoException(name, 1, "A");
            if (records != null && records.length >= 1) {
                ARecord a = (ARecord)records[0];
                InetAddress inetAddress = InetAddress.getByAddress(name, a.getAddress().getAddress());
                return inetAddress;
            }
            throw e;
        }
        finally {
            timeMetric.stopAndPublish();
        }
    }

    public Collection<InetAddress> getAllByName(String host) throws UnknownHostException {
        TimeMetric timeMetric = this.metricFactory.timer("getAllByName");
        String name = DNSJavaService.allowIPLiteral(host);
        try {
            if (name.equalsIgnoreCase(this.localHostName) || name.equalsIgnoreCase(this.localCanonicalHostName) || name.equals(this.localAddress)) {
                ImmutableList immutableList = ImmutableList.of((Object)this.getLocalHost());
                return immutableList;
            }
            InetAddress addr = Address.getByAddress((String)name);
            ImmutableList immutableList = ImmutableList.of((Object)addr);
            return immutableList;
        }
        catch (UnknownHostException e) {
            Record[] records = this.lookupNoException(name, 1, "A");
            if (records != null && records.length >= 1) {
                Object[] addrs = new InetAddress[records.length];
                for (int i = 0; i < records.length; ++i) {
                    ARecord a = (ARecord)records[i];
                    addrs[i] = InetAddress.getByAddress(name, a.getAddress().getAddress());
                }
                ImmutableList immutableList = ImmutableList.copyOf((Object[])addrs);
                return immutableList;
            }
            throw e;
        }
        finally {
            timeMetric.stopAndPublish();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public Collection<String> findTXTRecords(String hostname) {
        TimeMetric timeMetric = this.metricFactory.timer("findTXTRecords");
        ArrayList<String> txtR = new ArrayList<String>();
        Record[] records = this.lookupNoException(hostname, 16, "TXT");
        try {
            if (records != null) {
                for (Record record : records) {
                    TXTRecord txt = (TXTRecord)record;
                    txtR.add(txt.rdataToString());
                }
            }
            ArrayList<String> arrayList = txtR;
            return arrayList;
        }
        finally {
            timeMetric.stopAndPublish();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public String getHostName(InetAddress addr) {
        TimeMetric timeMetric = this.metricFactory.timer("getHostName");
        Name name = ReverseMap.fromAddress((InetAddress)addr);
        Record[] records = this.lookupNoException(name.toString(), 12, "PTR");
        try {
            String result;
            if (records == null) {
                result = addr.getHostAddress();
            } else {
                PTRRecord ptr = (PTRRecord)records[0];
                result = ptr.getTarget().toString();
            }
            String string = result;
            return string;
        }
        finally {
            timeMetric.stopAndPublish();
        }
    }

    public InetAddress getLocalHost() throws UnknownHostException {
        return InetAddress.getLocalHost();
    }

    public int getMaximumCacheSize() {
        return this.maxCacheSize;
    }

    public int getCurrentCacheSize() {
        return this.cache.getSize();
    }

    public void clearCache() {
        this.cache.clearCache();
    }

    private static class MXRecordComparator
    implements Comparator<MXRecord> {
        private MXRecordComparator() {
        }

        @Override
        public int compare(MXRecord a, MXRecord b) {
            int pa = a.getPriority();
            int pb = b.getPriority();
            return pa - pb;
        }
    }
}

