/*
 * Decompiled with CFR 0.152.
 */
package org.apache.james.cli;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Joiner;
import com.google.common.base.Strings;
import java.io.IOException;
import java.io.PrintStream;
import java.util.Arrays;
import java.util.Collection;
import java.util.Map;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.HelpFormatter;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;
import org.apache.commons.cli.PosixParser;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang.time.StopWatch;
import org.apache.james.cli.exceptions.InvalidArgumentNumberException;
import org.apache.james.cli.exceptions.InvalidPortException;
import org.apache.james.cli.exceptions.JamesCliException;
import org.apache.james.cli.exceptions.MissingCommandException;
import org.apache.james.cli.exceptions.UnrecognizedCommandException;
import org.apache.james.cli.probe.impl.JmxConnection;
import org.apache.james.cli.probe.impl.JmxDataProbe;
import org.apache.james.cli.probe.impl.JmxMailboxProbe;
import org.apache.james.cli.probe.impl.JmxQuotaProbe;
import org.apache.james.cli.probe.impl.JmxSieveProbe;
import org.apache.james.cli.type.CmdType;
import org.apache.james.cli.utils.ValueWithUnit;
import org.apache.james.mailbox.store.mail.model.SerializableQuota;
import org.apache.james.mailbox.store.probe.MailboxProbe;
import org.apache.james.mailbox.store.probe.QuotaProbe;
import org.apache.james.mailbox.store.probe.SieveProbe;
import org.apache.james.probe.DataProbe;
import org.apache.james.rrt.lib.Mappings;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ServerCmd {
    public static final String HOST_OPT_LONG = "host";
    public static final String HOST_OPT_SHORT = "h";
    public static final String PORT_OPT_LONG = "port";
    public static final String PORT_OPT_SHORT = "p";
    private static final String DEFAULT_HOST = "127.0.0.1";
    private static final int DEFAULT_PORT = 9999;
    private static final Logger LOG = LoggerFactory.getLogger(ServerCmd.class);
    private final DataProbe probe;
    private final MailboxProbe mailboxProbe;
    private final QuotaProbe quotaProbe;
    private final SieveProbe sieveProbe;

    private static Options createOptions() {
        return new Options().addOption(HOST_OPT_SHORT, HOST_OPT_LONG, true, "node hostname or ip address").addOption(PORT_OPT_SHORT, PORT_OPT_LONG, true, "remote jmx agent port number");
    }

    public static void main(String[] args) {
        try {
            ServerCmd.doMain(args);
            System.exit(0);
        }
        catch (JamesCliException e) {
            ServerCmd.failWithMessage(e.getMessage());
        }
        catch (ParseException e) {
            ServerCmd.failWithMessage("Error parsing command line : " + e.getMessage());
        }
        catch (IOException ioe) {
            ServerCmd.failWithMessage("Error connecting to remote JMX agent : " + ioe.getMessage());
        }
        catch (Exception e) {
            LOG.error("Error on command: {}", (Throwable)e);
            ServerCmd.failWithMessage("Error " + e.getClass() + " while executing command:" + e.getMessage());
        }
    }

    public static void doMain(String[] args) throws Exception {
        PrintStream printStream = System.out;
        ServerCmd.executeAndOutputToStream(args, printStream);
    }

    public static void executeAndOutputToStream(String[] args, PrintStream printStream) throws Exception {
        StopWatch stopWatch = new StopWatch();
        stopWatch.start();
        CommandLine cmd = ServerCmd.parseCommandLine(args);
        JmxConnection jmxConnection = new JmxConnection(ServerCmd.getHost(cmd), ServerCmd.getPort(cmd));
        CmdType cmdType = new ServerCmd(new JmxDataProbe().connect(jmxConnection), new JmxMailboxProbe().connect(jmxConnection), new JmxQuotaProbe().connect(jmxConnection), new JmxSieveProbe().connect(jmxConnection)).executeCommandLine(cmd, printStream);
        stopWatch.split();
        ServerCmd.print(new String[]{Joiner.on((char)' ').join((Object)cmdType.getCommand(), (Object)"command executed sucessfully in", new Object[]{stopWatch.getSplitTime(), "ms."})}, printStream);
        stopWatch.stop();
    }

    public ServerCmd(DataProbe probe, MailboxProbe mailboxProbe, QuotaProbe quotaProbe, SieveProbe sieveProbe) {
        this.probe = probe;
        this.mailboxProbe = mailboxProbe;
        this.quotaProbe = quotaProbe;
        this.sieveProbe = sieveProbe;
    }

    @VisibleForTesting
    static CommandLine parseCommandLine(String[] args) throws ParseException {
        PosixParser parser = new PosixParser();
        CommandLine commandLine = parser.parse(ServerCmd.createOptions(), args);
        if (commandLine.getArgs().length < 1) {
            throw new MissingCommandException();
        }
        return commandLine;
    }

    @VisibleForTesting
    static String getHost(CommandLine cmd) {
        String host = cmd.getOptionValue(HOST_OPT_LONG);
        if (Strings.isNullOrEmpty((String)host)) {
            return DEFAULT_HOST;
        }
        return host;
    }

    @VisibleForTesting
    static int getPort(CommandLine cmd) throws ParseException {
        String portNum = cmd.getOptionValue(PORT_OPT_LONG);
        if (!Strings.isNullOrEmpty((String)portNum)) {
            try {
                return ServerCmd.validatePortNumber(Integer.parseInt(portNum));
            }
            catch (NumberFormatException e) {
                throw new ParseException("Port must be a number");
            }
        }
        return 9999;
    }

    private static int validatePortNumber(int portNumber) {
        if (portNumber < 1 || portNumber > 65535) {
            throw new InvalidPortException(portNumber);
        }
        return portNumber;
    }

    private static void failWithMessage(String s) {
        System.err.println(s);
        ServerCmd.printUsage();
        System.exit(1);
    }

    @VisibleForTesting
    private CmdType executeCommandLine(CommandLine commandLine, PrintStream printStream) throws Exception {
        String[] arguments = commandLine.getArgs();
        String cmdName = arguments[0];
        CmdType cmdType = CmdType.lookup(cmdName);
        if (cmdType == null) {
            throw new UnrecognizedCommandException(cmdName);
        }
        if (!cmdType.hasCorrectArguments(arguments.length)) {
            throw new InvalidArgumentNumberException(cmdType, arguments.length);
        }
        this.executeCommand(arguments, cmdType, printStream);
        return cmdType;
    }

    @VisibleForTesting
    CmdType executeCommandLine(CommandLine commandLine) throws Exception {
        return this.executeCommandLine(commandLine, new PrintStream(System.out));
    }

    private void executeCommand(String[] arguments, CmdType cmdType, PrintStream printStream) throws Exception {
        switch (cmdType) {
            case ADDUSER: {
                this.probe.addUser(arguments[1], arguments[2]);
                break;
            }
            case REMOVEUSER: {
                this.probe.removeUser(arguments[1]);
                break;
            }
            case LISTUSERS: {
                ServerCmd.print(this.probe.listUsers(), printStream);
                break;
            }
            case ADDDOMAIN: {
                this.probe.addDomain(arguments[1]);
                break;
            }
            case REMOVEDOMAIN: {
                this.probe.removeDomain(arguments[1]);
                break;
            }
            case CONTAINSDOMAIN: {
                if (this.probe.containsDomain(arguments[1])) {
                    printStream.println(arguments[1] + " exists");
                    break;
                }
                printStream.println(arguments[1] + " does not exists");
                break;
            }
            case LISTDOMAINS: {
                ServerCmd.print(this.probe.listDomains(), printStream);
                break;
            }
            case LISTMAPPINGS: {
                this.print(this.probe.listMappings(), printStream);
                break;
            }
            case LISTUSERDOMAINMAPPINGS: {
                Mappings userDomainMappings = this.probe.listUserDomainMappings(arguments[1], arguments[2]);
                ServerCmd.print(userDomainMappings.asStrings(), printStream);
                break;
            }
            case ADDADDRESSMAPPING: {
                this.probe.addAddressMapping(arguments[1], arguments[2], arguments[3]);
                break;
            }
            case REMOVEADDRESSMAPPING: {
                this.probe.removeAddressMapping(arguments[1], arguments[2], arguments[3]);
                break;
            }
            case ADDREGEXMAPPING: {
                this.probe.addRegexMapping(arguments[1], arguments[2], arguments[3]);
                break;
            }
            case REMOVEREGEXMAPPING: {
                this.probe.removeRegexMapping(arguments[1], arguments[2], arguments[3]);
                break;
            }
            case SETPASSWORD: {
                this.probe.setPassword(arguments[1], arguments[2]);
                break;
            }
            case COPYMAILBOX: {
                this.mailboxProbe.copyMailbox(arguments[1], arguments[2]);
                break;
            }
            case DELETEUSERMAILBOXES: {
                this.mailboxProbe.deleteUserMailboxesNames(arguments[1]);
                break;
            }
            case CREATEMAILBOX: {
                this.mailboxProbe.createMailbox(arguments[1], arguments[2], arguments[3]);
                break;
            }
            case LISTUSERMAILBOXES: {
                Collection mailboxes = this.mailboxProbe.listUserMailboxes(arguments[1]);
                ServerCmd.print(mailboxes.toArray(new String[0]), printStream);
                break;
            }
            case DELETEMAILBOX: {
                this.mailboxProbe.deleteMailbox(arguments[1], arguments[2], arguments[3]);
                break;
            }
            case GETSTORAGEQUOTA: {
                this.printStorageQuota(arguments[1], this.quotaProbe.getStorageQuota(arguments[1]), printStream);
                break;
            }
            case GETMESSAGECOUNTQUOTA: {
                this.printMessageQuota(arguments[1], this.quotaProbe.getMessageCountQuota(arguments[1]), printStream);
                break;
            }
            case GETQUOTAROOT: {
                printStream.println("Quota Root: " + this.quotaProbe.getQuotaRoot(arguments[1], arguments[2], arguments[3]));
                break;
            }
            case GETMAXSTORAGEQUOTA: {
                printStream.println("Storage space allowed for Quota Root " + arguments[1] + ": " + this.formatStorageValue(this.quotaProbe.getMaxStorage(arguments[1])));
                break;
            }
            case GETMAXMESSAGECOUNTQUOTA: {
                printStream.println("MailboxMessage count allowed for Quota Root " + arguments[1] + ": " + this.formatMessageValue(this.quotaProbe.getMaxMessageCount(arguments[1])));
                break;
            }
            case SETMAXSTORAGEQUOTA: {
                this.quotaProbe.setMaxStorage(arguments[1], ValueWithUnit.parse(arguments[2]).getConvertedValue());
                break;
            }
            case SETMAXMESSAGECOUNTQUOTA: {
                this.quotaProbe.setMaxMessageCount(arguments[1], Long.parseLong(arguments[2]));
                break;
            }
            case SETDEFAULTMAXSTORAGEQUOTA: {
                this.quotaProbe.setDefaultMaxStorage(ValueWithUnit.parse(arguments[1]).getConvertedValue());
                break;
            }
            case SETDEFAULTMAXMESSAGECOUNTQUOTA: {
                this.quotaProbe.setDefaultMaxMessageCount(Long.parseLong(arguments[1]));
                break;
            }
            case GETDEFAULTMAXSTORAGEQUOTA: {
                printStream.println("Default Maximum Storage Quota: " + this.formatStorageValue(this.quotaProbe.getDefaultMaxStorage()));
                break;
            }
            case GETDEFAULTMAXMESSAGECOUNTQUOTA: {
                printStream.println("Default Maximum message count Quota: " + this.formatMessageValue(this.quotaProbe.getDefaultMaxMessageCount()));
                break;
            }
            case REINDEXMAILBOX: {
                this.mailboxProbe.reIndexMailbox(arguments[1], arguments[2], arguments[3]);
                break;
            }
            case REINDEXALL: {
                this.mailboxProbe.reIndexAll();
                break;
            }
            case SETSIEVEQUOTA: {
                this.sieveProbe.setSieveQuota(ValueWithUnit.parse(arguments[1]).getConvertedValue());
                break;
            }
            case SETSIEVEUSERQUOTA: {
                this.sieveProbe.setSieveQuota(arguments[1], ValueWithUnit.parse(arguments[2]).getConvertedValue());
                break;
            }
            case GETSIEVEQUOTA: {
                printStream.println("Storage space allowed for Sieve scripts by default: " + this.formatStorageValue(this.sieveProbe.getSieveQuota()));
                break;
            }
            case GETSIEVEUSERQUOTA: {
                printStream.println("Storage space allowed for " + arguments[1] + " Sieve scripts: " + this.formatStorageValue(this.sieveProbe.getSieveQuota(arguments[1])));
                break;
            }
            case REMOVESIEVEQUOTA: {
                this.sieveProbe.removeSieveQuota();
                break;
            }
            case REMOVESIEVEUSERQUOTA: {
                this.sieveProbe.removeSieveQuota(arguments[1]);
                break;
            }
            default: {
                throw new UnrecognizedCommandException(cmdType.getCommand());
            }
        }
    }

    private static void print(String[] data, PrintStream out) {
        ServerCmd.print(Arrays.asList(data), out);
    }

    private static void print(Iterable<String> data, PrintStream out) {
        if (data != null) {
            out.println(Joiner.on((char)'\n').join(data));
        }
    }

    private void printStorageQuota(String quotaRootString, SerializableQuota quota, PrintStream printStream) {
        printStream.println(String.format("Storage quota for %s is: %s / %s", quotaRootString, this.formatStorageValue(quota.getUsed()), this.formatStorageValue(quota.getMax())));
    }

    private void printMessageQuota(String quotaRootString, SerializableQuota quota, PrintStream printStream) {
        printStream.println(String.format("MailboxMessage count quota for %s is: %s / %s", quotaRootString, this.formatMessageValue(quota.getUsed()), this.formatMessageValue(quota.getMax())));
    }

    private String formatStorageValue(long value) {
        if (value == Long.MIN_VALUE) {
            return "UNKNOWN";
        }
        if (value == -1L) {
            return "UNLIMITED";
        }
        return FileUtils.byteCountToDisplaySize((long)value);
    }

    private String formatMessageValue(long value) {
        if (value == Long.MIN_VALUE) {
            return "UNKNOWN";
        }
        if (value == -1L) {
            return "UNLIMITED";
        }
        return String.valueOf(value);
    }

    private void print(Map<String, Mappings> map, PrintStream out) {
        if (map != null) {
            for (Map.Entry<String, Mappings> entry : map.entrySet()) {
                out.println(entry.getKey() + '=' + entry.getValue().serialize());
            }
            out.println();
        }
    }

    private static void printUsage() {
        StringBuilder footerBuilder = new StringBuilder();
        for (CmdType cmdType : CmdType.values()) {
            footerBuilder.append(cmdType.getUsage()).append("\n");
        }
        new HelpFormatter().printHelp(String.format("java %s --host <arg> <command>%n", ServerCmd.class.getName()), "", ServerCmd.createOptions(), footerBuilder.toString());
    }
}

