/*
 * Decompiled with CFR 0.152.
 */
package org.apache.james.transport.mailets.remoteDelivery;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.collect.Iterables;
import java.io.IOException;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;
import javax.mail.Address;
import javax.mail.MessagingException;
import javax.mail.SendFailedException;
import javax.mail.internet.InternetAddress;
import org.apache.james.dnsservice.api.DNSService;
import org.apache.james.dnsservice.api.TemporaryResolutionException;
import org.apache.james.transport.mailets.remoteDelivery.AddressesArrayToMailAddressListConverter;
import org.apache.james.transport.mailets.remoteDelivery.Bouncer;
import org.apache.james.transport.mailets.remoteDelivery.DeliveryRetriesHelper;
import org.apache.james.transport.mailets.remoteDelivery.DnsHelper;
import org.apache.james.transport.mailets.remoteDelivery.EnhancedMessagingException;
import org.apache.james.transport.mailets.remoteDelivery.ExecutionResult;
import org.apache.james.transport.mailets.remoteDelivery.InternetAddressConverter;
import org.apache.james.transport.mailets.remoteDelivery.MailDelivrerToHost;
import org.apache.james.transport.mailets.remoteDelivery.MessageComposer;
import org.apache.james.transport.mailets.remoteDelivery.RemoteDeliveryConfiguration;
import org.apache.mailet.HostAddress;
import org.apache.mailet.Mail;
import org.apache.mailet.MailAddress;
import org.slf4j.Logger;

public class MailDelivrer {
    private final RemoteDeliveryConfiguration configuration;
    private final MailDelivrerToHost mailDelivrerToHost;
    private final DnsHelper dnsHelper;
    private final MessageComposer messageComposer;
    private final Bouncer bouncer;
    private final Logger logger;

    public MailDelivrer(RemoteDeliveryConfiguration configuration, MailDelivrerToHost mailDelivrerToHost, DNSService dnsServer, Bouncer bouncer, Logger logger) {
        this(configuration, mailDelivrerToHost, new DnsHelper(dnsServer, configuration, logger), bouncer, logger);
    }

    @VisibleForTesting
    MailDelivrer(RemoteDeliveryConfiguration configuration, MailDelivrerToHost mailDelivrerToHost, DnsHelper dnsHelper, Bouncer bouncer, Logger logger) {
        this.configuration = configuration;
        this.mailDelivrerToHost = mailDelivrerToHost;
        this.dnsHelper = dnsHelper;
        this.messageComposer = new MessageComposer(configuration);
        this.bouncer = bouncer;
        this.logger = logger;
    }

    public ExecutionResult deliver(Mail mail) {
        try {
            return this.tryDeliver(mail);
        }
        catch (SendFailedException sfe) {
            return this.handleSenderFailedException(mail, sfe);
        }
        catch (MessagingException ex) {
            boolean isPermanent = new EnhancedMessagingException(ex).isServerError();
            return this.logAndReturn(mail, ExecutionResult.onFailure(isPermanent, (Exception)((Object)ex)));
        }
        catch (Exception ex) {
            this.logger.error("Generic exception = permanent failure: {}", (Object)ex.getMessage(), (Object)ex);
            return this.logAndReturn(mail, ExecutionResult.permanentFailure(ex));
        }
    }

    private ExecutionResult tryDeliver(Mail mail) throws MessagingException {
        if (mail.getRecipients().isEmpty()) {
            this.logger.info("No recipients specified... not sure how this could have happened.");
            return ExecutionResult.permanentFailure(new Exception("No recipients specified for " + mail.getName() + " sent by " + mail.getSender()));
        }
        if (this.configuration.isDebug()) {
            this.logger.debug("Attempting to deliver {}", (Object)mail.getName());
        }
        String host = this.retrieveTargetHostname(mail);
        try {
            Iterator<HostAddress> targetServers = this.dnsHelper.retrieveHostAddressIterator(host);
            if (!targetServers.hasNext()) {
                return this.handleNoTargetServer(mail, host);
            }
            return this.doDeliver(mail, InternetAddressConverter.convert(mail.getRecipients()), targetServers);
        }
        catch (TemporaryResolutionException e) {
            return this.logAndReturn(mail, ExecutionResult.temporaryFailure((Exception)((Object)new MessagingException("Temporary problem looking up mail server for host: " + host + ".  I cannot determine where to send this message."))));
        }
    }

    private String retrieveTargetHostname(Mail mail) {
        Preconditions.checkArgument((!mail.getRecipients().isEmpty() ? 1 : 0) != 0, (Object)"Mail should have recipients to attempt delivery");
        MailAddress rcpt = (MailAddress)Iterables.getFirst((Iterable)mail.getRecipients(), null);
        return rcpt.getDomain();
    }

    private ExecutionResult doDeliver(Mail mail, InternetAddress[] addr, Iterator<HostAddress> targetServers) throws MessagingException {
        MessagingException lastError = null;
        while (targetServers.hasNext()) {
            try {
                return this.mailDelivrerToHost.tryDeliveryToHost(mail, addr, targetServers.next());
            }
            catch (SendFailedException sfe) {
                lastError = this.handleSendFailExceptionOnMxIteration(mail, sfe);
            }
            catch (MessagingException me) {
                lastError = this.handleMessagingException(mail, me);
                if (this.configuration.isDebug()) {
                    this.logger.debug(me.getMessage(), me.getCause());
                    continue;
                }
                this.logger.info(me.getMessage());
            }
        }
        if (lastError != null) {
            throw lastError;
        }
        return ExecutionResult.temporaryFailure();
    }

    private MessagingException handleMessagingException(Mail mail, MessagingException me) throws MessagingException {
        this.logger.debug("Exception delivering message ({}) - {}", (Object)mail.getName(), (Object)me.getMessage());
        if (me.getNextException() != null && me.getNextException() instanceof IOException) {
            return me;
        }
        throw me;
    }

    @VisibleForTesting
    ExecutionResult handleSenderFailedException(Mail mail, SendFailedException sfe) {
        this.logSendFailedException(sfe);
        EnhancedMessagingException enhancedMessagingException = new EnhancedMessagingException((MessagingException)((Object)sfe));
        List<MailAddress> invalidAddresses = AddressesArrayToMailAddressListConverter.getAddressesAsMailAddress(sfe.getInvalidAddresses(), this.logger);
        List<MailAddress> validUnsentAddresses = AddressesArrayToMailAddressListConverter.getAddressesAsMailAddress(sfe.getValidUnsentAddresses(), this.logger);
        if (this.configuration.isDebug()) {
            this.logger.debug("Mail {} has initially recipients: {}", (Object)mail.getName(), (Object)mail.getRecipients());
            if (!invalidAddresses.isEmpty()) {
                this.logger.debug("Invalid recipients: {}", invalidAddresses);
            }
            if (!validUnsentAddresses.isEmpty()) {
                this.logger.debug("Unsent recipients: {}", validUnsentAddresses);
            }
        }
        if (!validUnsentAddresses.isEmpty()) {
            if (!invalidAddresses.isEmpty()) {
                mail.setRecipients(invalidAddresses);
                this.bouncer.bounce(mail, (Exception)((Object)sfe));
            }
            mail.setRecipients(validUnsentAddresses);
            if (enhancedMessagingException.hasReturnCode()) {
                boolean isPermanent = enhancedMessagingException.isServerError();
                return this.logAndReturn(mail, ExecutionResult.onFailure(isPermanent, (Exception)((Object)sfe)));
            }
            return this.logAndReturn(mail, ExecutionResult.temporaryFailure((Exception)((Object)sfe)));
        }
        if (!invalidAddresses.isEmpty()) {
            mail.setRecipients(invalidAddresses);
            return this.logAndReturn(mail, ExecutionResult.permanentFailure((Exception)((Object)sfe)));
        }
        if ((enhancedMessagingException.hasReturnCode() || enhancedMessagingException.hasNestedReturnCode()) && enhancedMessagingException.isServerError()) {
            return ExecutionResult.permanentFailure((Exception)((Object)sfe));
        }
        return ExecutionResult.temporaryFailure((Exception)((Object)sfe));
    }

    private ExecutionResult logAndReturn(Mail mail, ExecutionResult executionResult) {
        this.logger.debug(this.messageComposer.composeFailLogMessage(mail, executionResult));
        return executionResult;
    }

    private MessagingException handleSendFailExceptionOnMxIteration(Mail mail, SendFailedException sfe) throws SendFailedException {
        EnhancedMessagingException enhancedMessagingException;
        Address[] validSent;
        this.logSendFailedException(sfe);
        if (sfe.getValidSentAddresses() != null && (validSent = sfe.getValidSentAddresses()).length > 0) {
            this.logger.debug("Mail ({}) sent successfully for {}", (Object)mail.getName(), Arrays.asList(validSent));
        }
        if ((enhancedMessagingException = new EnhancedMessagingException((MessagingException)((Object)sfe))).isServerError()) {
            throw sfe;
        }
        if (sfe.getValidUnsentAddresses() != null && sfe.getValidUnsentAddresses().length > 0) {
            if (this.configuration.isDebug()) {
                this.logger.debug("Send failed, {} valid addresses remain, continuing with any other servers", (Object)sfe.getValidUnsentAddresses().length);
            }
            return sfe;
        }
        throw sfe;
    }

    private ExecutionResult handleNoTargetServer(Mail mail, String host) {
        this.logger.info("No mail server found for: {}", (Object)host);
        MessagingException messagingException = new MessagingException("There are no DNS entries for the hostname " + host + ".  I cannot determine where to send this message.");
        int retry = DeliveryRetriesHelper.retrieveRetries(mail);
        if (retry >= this.configuration.getDnsProblemRetry()) {
            return this.logAndReturn(mail, ExecutionResult.permanentFailure((Exception)((Object)messagingException)));
        }
        return this.logAndReturn(mail, ExecutionResult.temporaryFailure((Exception)((Object)messagingException)));
    }

    private void logSendFailedException(SendFailedException sfe) {
        if (this.configuration.isDebug()) {
            EnhancedMessagingException enhancedMessagingException = new EnhancedMessagingException((MessagingException)((Object)sfe));
            if (enhancedMessagingException.hasReturnCode()) {
                this.logger.debug("SMTP SEND FAILED: Command [{}] RetCode: [{}] Response[{}]", new Object[]{enhancedMessagingException.computeCommand(), enhancedMessagingException.getReturnCode(), sfe.getMessage()});
            } else {
                this.logger.debug("Send failed: {}", (Object)sfe.toString());
            }
            this.logLevels((MessagingException)((Object)sfe));
        }
    }

    private void logLevels(MessagingException me) {
        Exception ne;
        while ((ne = me.getNextException()) != null && ne instanceof MessagingException) {
            me = (MessagingException)((Object)ne);
            EnhancedMessagingException enhancedMessagingException = new EnhancedMessagingException(me);
            if (!((Object)((Object)me)).getClass().getName().endsWith(".SMTPAddressFailedException") && !((Object)((Object)me)).getClass().getName().endsWith(".SMTPAddressSucceededException")) continue;
            this.logger.debug("ADDRESS :[{}] Address:[{}] Command : [{}] RetCode[{}] Response [{}]", new Object[]{enhancedMessagingException.computeAction(), me.toString(), enhancedMessagingException.computeAddress(), enhancedMessagingException.computeCommand(), enhancedMessagingException.getReturnCode()});
        }
    }
}

