/*
 * Decompiled with CFR 0.152.
 */
package org.apache.james.task.eventsourcing.distributed;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.ImmutableMap;
import com.rabbitmq.client.AMQP;
import com.rabbitmq.client.Delivery;
import com.rabbitmq.client.MessageProperties;
import java.nio.charset.StandardCharsets;
import java.time.Duration;
import java.util.Map;
import java.util.Optional;
import org.apache.james.backends.rabbitmq.Constants;
import org.apache.james.backends.rabbitmq.ReceiverProvider;
import org.apache.james.server.task.json.JsonTaskSerializer;
import org.apache.james.task.Task;
import org.apache.james.task.TaskId;
import org.apache.james.task.TaskManagerWorker;
import org.apache.james.task.TaskWithId;
import org.apache.james.task.WorkQueue;
import org.apache.james.task.eventsourcing.distributed.CancelRequestQueueName;
import org.apache.james.task.eventsourcing.distributed.RabbitMQWorkQueueConfiguration;
import org.reactivestreams.Publisher;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import reactor.core.Disposable;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;
import reactor.core.publisher.UnicastProcessor;
import reactor.core.scheduler.Schedulers;
import reactor.rabbitmq.AcknowledgableDelivery;
import reactor.rabbitmq.BindingSpecification;
import reactor.rabbitmq.ConsumeOptions;
import reactor.rabbitmq.ExchangeSpecification;
import reactor.rabbitmq.OutboundMessage;
import reactor.rabbitmq.QueueSpecification;
import reactor.rabbitmq.Receiver;
import reactor.rabbitmq.Sender;
import reactor.util.retry.Retry;

public class RabbitMQWorkQueue
implements WorkQueue {
    private static final Logger LOGGER = LoggerFactory.getLogger(RabbitMQWorkQueue.class);
    static final String EXCHANGE_NAME = "taskManagerWorkQueueExchange";
    static final String QUEUE_NAME = "taskManagerWorkQueue";
    static final String ROUTING_KEY = "taskManagerWorkQueueRoutingKey";
    static final String CANCEL_REQUESTS_EXCHANGE_NAME = "taskManagerCancelRequestsExchange";
    static final String CANCEL_REQUESTS_ROUTING_KEY = "taskManagerCancelRequestsRoutingKey";
    public static final String TASK_ID = "taskId";
    public static final int NUM_RETRIES = 8;
    public static final Duration FIRST_BACKOFF = Duration.ofMillis(100L);
    private final TaskManagerWorker worker;
    private final JsonTaskSerializer taskSerializer;
    private final RabbitMQWorkQueueConfiguration configuration;
    private final Sender sender;
    private final ReceiverProvider receiverProvider;
    private final CancelRequestQueueName cancelRequestQueueName;
    private UnicastProcessor<TaskId> sendCancelRequestsQueue;
    private Disposable sendCancelRequestsQueueHandle;
    private Disposable receiverHandle;
    private Disposable cancelRequestListenerHandle;

    public RabbitMQWorkQueue(TaskManagerWorker worker, Sender sender, ReceiverProvider receiverProvider, JsonTaskSerializer taskSerializer, RabbitMQWorkQueueConfiguration configuration, CancelRequestQueueName cancelRequestQueueName) {
        this.cancelRequestQueueName = cancelRequestQueueName;
        this.worker = worker;
        this.receiverProvider = receiverProvider;
        this.sender = sender;
        this.taskSerializer = taskSerializer;
        this.configuration = configuration;
    }

    public void start() {
        this.startWorkqueue();
        this.listenToCancelRequests();
    }

    private void startWorkqueue() {
        this.declareQueue();
        if (this.configuration.enabled()) {
            this.consumeWorkqueue();
        }
    }

    @VisibleForTesting
    void declareQueue() {
        Mono declareExchange = this.sender.declareExchange(ExchangeSpecification.exchange((String)EXCHANGE_NAME)).retryWhen((Retry)Retry.backoff((long)8L, (Duration)FIRST_BACKOFF));
        Mono declareQueue = this.sender.declare(QueueSpecification.queue((String)QUEUE_NAME).durable(true).arguments(Constants.WITH_SINGLE_ACTIVE_CONSUMER)).retryWhen((Retry)Retry.backoff((long)8L, (Duration)FIRST_BACKOFF));
        Mono bindQueueToExchange = this.sender.bind(BindingSpecification.binding((String)EXCHANGE_NAME, (String)ROUTING_KEY, (String)QUEUE_NAME)).retryWhen((Retry)Retry.backoff((long)8L, (Duration)FIRST_BACKOFF));
        declareExchange.then(declareQueue).then(bindQueueToExchange).block();
    }

    public void restart() {
        Disposable previousWorkQueueHandler = this.receiverHandle;
        this.consumeWorkqueue();
        previousWorkQueueHandler.dispose();
        Disposable previousCancelHandler = this.cancelRequestListenerHandle;
        this.registerCancelRequestsListener(this.cancelRequestQueueName.asString());
        previousCancelHandler.dispose();
    }

    private void consumeWorkqueue() {
        this.receiverHandle = Flux.using(() -> ((ReceiverProvider)this.receiverProvider).createReceiver(), receiver -> receiver.consumeManualAck(QUEUE_NAME, new ConsumeOptions()), Receiver::close).subscribeOn(Schedulers.elastic()).concatMap(this::executeTask).subscribe();
    }

    private Mono<Task.Result> executeTask(AcknowledgableDelivery delivery) {
        return Mono.fromCallable(() -> TaskId.fromString((String)delivery.getProperties().getHeaders().get(TASK_ID).toString())).flatMap(taskId -> this.deserialize(new String(delivery.getBody(), StandardCharsets.UTF_8), (TaskId)taskId).doOnNext(task -> delivery.ack()).flatMap(task -> this.executeOnWorker((TaskId)taskId, (Task)task))).onErrorResume(error -> {
            Optional taskId = Optional.ofNullable(delivery.getProperties()).flatMap(props -> Optional.ofNullable(props.getHeaders())).flatMap(headers -> Optional.ofNullable(headers.get(TASK_ID)));
            LOGGER.error("Unable to process {} {}", new Object[]{TASK_ID, taskId, error});
            delivery.nack(false);
            return Mono.empty();
        });
    }

    private Mono<Task> deserialize(String json, TaskId taskId) {
        return Mono.fromCallable(() -> this.taskSerializer.deserialize(json)).onErrorResume(error -> {
            String errorMessage = String.format("Unable to deserialize submitted Task %s", taskId.asString());
            LOGGER.error(errorMessage, error);
            return Mono.from((Publisher)this.worker.fail(taskId, Optional.empty(), errorMessage, error)).then(Mono.empty());
        });
    }

    private Mono<Task.Result> executeOnWorker(TaskId taskId, Task task) {
        return this.worker.executeTask(new TaskWithId(taskId, task)).onErrorResume(error -> {
            String errorMessage = String.format("Unable to run submitted Task %s", taskId.asString());
            LOGGER.warn(errorMessage, error);
            return Mono.from((Publisher)this.worker.fail(taskId, task.details(), errorMessage, error)).then(Mono.empty());
        });
    }

    private void listenToCancelRequests() {
        this.sender.declareExchange(ExchangeSpecification.exchange((String)CANCEL_REQUESTS_EXCHANGE_NAME)).block();
        this.sender.declare(QueueSpecification.queue((String)this.cancelRequestQueueName.asString()).durable(false).autoDelete(true)).block();
        this.sender.bind(BindingSpecification.binding((String)CANCEL_REQUESTS_EXCHANGE_NAME, (String)CANCEL_REQUESTS_ROUTING_KEY, (String)this.cancelRequestQueueName.asString())).block();
        this.registerCancelRequestsListener(this.cancelRequestQueueName.asString());
        this.sendCancelRequestsQueue = UnicastProcessor.create();
        this.sendCancelRequestsQueueHandle = this.sender.send((Publisher)this.sendCancelRequestsQueue.map(this::makeCancelRequestMessage)).subscribeOn(Schedulers.elastic()).subscribe();
    }

    private void registerCancelRequestsListener(String queueName) {
        this.cancelRequestListenerHandle = Flux.using(() -> ((ReceiverProvider)this.receiverProvider).createReceiver(), receiver -> receiver.consumeAutoAck(queueName), Receiver::close).subscribeOn(Schedulers.elastic()).map(this::readCancelRequestMessage).doOnNext(arg_0 -> ((TaskManagerWorker)this.worker).cancelTask(arg_0)).subscribe();
    }

    private TaskId readCancelRequestMessage(Delivery delivery) {
        String message = new String(delivery.getBody(), StandardCharsets.UTF_8);
        return TaskId.fromString((String)message);
    }

    private OutboundMessage makeCancelRequestMessage(TaskId taskId) {
        byte[] payload = taskId.asString().getBytes(StandardCharsets.UTF_8);
        AMQP.BasicProperties basicProperties = new AMQP.BasicProperties.Builder().build();
        return new OutboundMessage(CANCEL_REQUESTS_EXCHANGE_NAME, CANCEL_REQUESTS_ROUTING_KEY, basicProperties, payload);
    }

    public void submit(TaskWithId taskWithId) {
        try {
            byte[] payload = this.taskSerializer.serialize(taskWithId.getTask()).getBytes(StandardCharsets.UTF_8);
            AMQP.BasicProperties basicProperties = new AMQP.BasicProperties.Builder().deliveryMode(MessageProperties.PERSISTENT_TEXT_PLAIN.getDeliveryMode()).priority(MessageProperties.PERSISTENT_TEXT_PLAIN.getPriority()).contentType(MessageProperties.PERSISTENT_TEXT_PLAIN.getContentType()).headers((Map)ImmutableMap.of((Object)TASK_ID, (Object)taskWithId.getId().asString())).build();
            OutboundMessage outboundMessage = new OutboundMessage(EXCHANGE_NAME, ROUTING_KEY, basicProperties, payload);
            this.sender.send((Publisher)Mono.just((Object)outboundMessage)).block();
        }
        catch (JsonProcessingException e) {
            throw new RuntimeException(e);
        }
    }

    public void cancel(TaskId taskId) {
        this.sendCancelRequestsQueue.onNext((Object)taskId);
    }

    public void close() {
        Optional.ofNullable(this.receiverHandle).ifPresent(Disposable::dispose);
        Optional.ofNullable(this.sendCancelRequestsQueueHandle).ifPresent(Disposable::dispose);
        Optional.ofNullable(this.cancelRequestListenerHandle).ifPresent(Disposable::dispose);
    }
}

