/*
 * Decompiled with CFR 0.152.
 */
package org.apache.mailbox.tools.indexer;

import com.google.common.collect.ImmutableList;
import io.vavr.control.Either;
import java.time.Duration;
import java.util.Collection;
import javax.inject.Inject;
import javax.mail.Flags;
import org.apache.james.core.Username;
import org.apache.james.mailbox.MailboxManager;
import org.apache.james.mailbox.MailboxSession;
import org.apache.james.mailbox.MessageUid;
import org.apache.james.mailbox.indexer.ReIndexer;
import org.apache.james.mailbox.indexer.ReIndexingExecutionFailures;
import org.apache.james.mailbox.model.Mailbox;
import org.apache.james.mailbox.model.MailboxId;
import org.apache.james.mailbox.model.MessageId;
import org.apache.james.mailbox.model.MessageRange;
import org.apache.james.mailbox.model.search.MailboxQuery;
import org.apache.james.mailbox.store.MailboxSessionMapperFactory;
import org.apache.james.mailbox.store.mail.MailboxMapper;
import org.apache.james.mailbox.store.mail.MessageMapper;
import org.apache.james.mailbox.store.mail.model.MailboxMessage;
import org.apache.james.mailbox.store.search.ListeningMessageSearchIndex;
import org.apache.james.task.Task;
import org.apache.james.util.ReactorUtils;
import org.apache.mailbox.tools.indexer.ReprocessingContext;
import org.reactivestreams.Publisher;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;

public class ReIndexerPerformer {
    public static final int MAILBOX_CONCURRENCY = 1;
    public static final int ONE = 1;
    private static final Logger LOGGER = LoggerFactory.getLogger(ReIndexerPerformer.class);
    private static final int SINGLE_MESSAGE = 1;
    private static final String RE_INDEXING = "re-indexing";
    private static final Username RE_INDEXER_PERFORMER_USER = Username.of((String)"re-indexing");
    private final MailboxManager mailboxManager;
    private final ListeningMessageSearchIndex messageSearchIndex;
    private final MailboxSessionMapperFactory mailboxSessionMapperFactory;

    @Inject
    public ReIndexerPerformer(MailboxManager mailboxManager, ListeningMessageSearchIndex messageSearchIndex, MailboxSessionMapperFactory mailboxSessionMapperFactory) {
        this.mailboxManager = mailboxManager;
        this.messageSearchIndex = messageSearchIndex;
        this.mailboxSessionMapperFactory = mailboxSessionMapperFactory;
    }

    Mono<Task.Result> reIndexAllMessages(ReprocessingContext reprocessingContext, ReIndexer.RunningOptions runningOptions) {
        MailboxSession mailboxSession = this.mailboxManager.createSystemSession(RE_INDEXER_PERFORMER_USER);
        LOGGER.info("Starting a full reindex");
        Flux entriesToIndex = this.mailboxSessionMapperFactory.getMailboxMapper(mailboxSession).list().flatMap(mailbox -> this.reIndexingEntriesForMailbox((Mailbox)mailbox, mailboxSession, runningOptions), 1);
        return this.reIndexMessages((Flux<Either<Failure, ReIndexingEntry>>)entriesToIndex, runningOptions, reprocessingContext).doFinally(any -> LOGGER.info("Full reindex finished"));
    }

    Mono<Task.Result> reIndexSingleMailbox(MailboxId mailboxId, ReprocessingContext reprocessingContext, ReIndexer.RunningOptions runningOptions) {
        MailboxSession mailboxSession = this.mailboxManager.createSystemSession(RE_INDEXER_PERFORMER_USER);
        Flux entriesToIndex = this.mailboxSessionMapperFactory.getMailboxMapper(mailboxSession).findMailboxById(mailboxId).flatMapMany(mailbox -> this.reIndexingEntriesForMailbox((Mailbox)mailbox, mailboxSession, runningOptions));
        return this.reIndexMessages((Flux<Either<Failure, ReIndexingEntry>>)entriesToIndex, runningOptions, reprocessingContext);
    }

    Mono<Task.Result> reIndexUserMailboxes(Username username, ReprocessingContext reprocessingContext, ReIndexer.RunningOptions runningOptions) {
        MailboxSession mailboxSession = this.mailboxManager.createSystemSession(username);
        MailboxMapper mailboxMapper = this.mailboxSessionMapperFactory.getMailboxMapper(mailboxSession);
        LOGGER.info("Starting a reindex for user {}", (Object)username.asString());
        MailboxQuery mailboxQuery = MailboxQuery.privateMailboxesBuilder((MailboxSession)mailboxSession).build();
        try {
            Flux entriesToIndex = mailboxMapper.findMailboxWithPathLike(mailboxQuery.asUserBound()).flatMap(mailbox -> this.reIndexingEntriesForMailbox((Mailbox)mailbox, mailboxSession, runningOptions), 1);
            return this.reIndexMessages((Flux<Either<Failure, ReIndexingEntry>>)entriesToIndex, runningOptions, reprocessingContext).doFinally(any -> LOGGER.info("User {} reindex finished", (Object)username.asString()));
        }
        catch (Exception e) {
            LOGGER.error("Error fetching mailboxes for user: {}", (Object)username.asString());
            return Mono.just((Object)Task.Result.PARTIAL);
        }
    }

    Mono<Task.Result> reIndexSingleMessage(MailboxId mailboxId, MessageUid uid, ReprocessingContext reprocessingContext) {
        MailboxSession mailboxSession = this.mailboxManager.createSystemSession(RE_INDEXER_PERFORMER_USER);
        return this.mailboxSessionMapperFactory.getMailboxMapper(mailboxSession).findMailboxById(mailboxId).map(mailbox -> new ReIndexingEntry((Mailbox)mailbox, mailboxSession, uid)).flatMap(this::fullyReadMessage).flatMap(message -> this.reIndex((MailboxMessage)message, mailboxSession)).switchIfEmpty(Mono.just((Object)Task.Result.COMPLETED));
    }

    Mono<Task.Result> reIndexMessageId(MessageId messageId) {
        MailboxSession session = this.mailboxManager.createSystemSession(RE_INDEXER_PERFORMER_USER);
        return this.mailboxSessionMapperFactory.getMessageIdMapper(session).findReactive((Collection)ImmutableList.of((Object)messageId), MessageMapper.FetchType.Full).flatMap(mailboxMessage -> this.reIndex((MailboxMessage)mailboxMessage, session), 16).reduce(Task::combine).switchIfEmpty(Mono.just((Object)Task.Result.COMPLETED)).onErrorResume(e -> {
            LOGGER.warn("Failed to re-index {}", (Object)messageId, e);
            return Mono.just((Object)Task.Result.PARTIAL);
        });
    }

    Mono<Task.Result> reIndexErrors(ReprocessingContext reprocessingContext, ReIndexingExecutionFailures previousReIndexingFailures, ReIndexer.RunningOptions runningOptions) {
        MailboxSession mailboxSession = this.mailboxManager.createSystemSession(RE_INDEXER_PERFORMER_USER);
        MailboxMapper mapper = this.mailboxSessionMapperFactory.getMailboxMapper(mailboxSession);
        Flux entriesToIndex = Flux.merge((Publisher[])new Publisher[]{Flux.fromIterable((Iterable)previousReIndexingFailures.messageFailures()).flatMap(this::createReindexingEntryFromFailure, 16), Flux.fromIterable((Iterable)previousReIndexingFailures.mailboxFailures()).flatMap(mailboxId -> mapper.findMailboxById(mailboxId).flatMapMany(mailbox -> this.reIndexingEntriesForMailbox((Mailbox)mailbox, mailboxSession, runningOptions)).onErrorResume(e -> {
            LOGGER.warn("Failed to re-index {}", mailboxId, e);
            return Mono.just((Object)Either.left((Object)new MailboxFailure((MailboxId)mailboxId)));
        }), 1)});
        return this.reIndexMessages((Flux<Either<Failure, ReIndexingEntry>>)entriesToIndex, runningOptions, reprocessingContext);
    }

    private Mono<Task.Result> reIndex(MailboxMessage mailboxMessage, MailboxSession session) {
        return this.mailboxSessionMapperFactory.getMailboxMapper(session).findMailboxById(mailboxMessage.getMailboxId()).flatMap(mailbox -> this.messageSearchIndex.add(session, mailbox, mailboxMessage)).thenReturn((Object)Task.Result.COMPLETED).onErrorResume(e -> {
            LOGGER.warn("Failed to re-index {} in {}", new Object[]{mailboxMessage.getUid(), mailboxMessage.getMailboxId(), e});
            return Mono.just((Object)Task.Result.PARTIAL);
        });
    }

    private Mono<MailboxMessage> fullyReadMessage(ReIndexingEntry entry) {
        return this.mailboxSessionMapperFactory.getMessageMapper(entry.getMailboxSession()).findInMailboxReactive(entry.getMailbox(), MessageRange.one((MessageUid)entry.getUid()), MessageMapper.FetchType.Full, 1).next();
    }

    private Mono<Either<Failure, ReIndexingEntry>> createReindexingEntryFromFailure(ReIndexingExecutionFailures.ReIndexingFailure previousFailure) {
        MailboxSession mailboxSession = this.mailboxManager.createSystemSession(RE_INDEXER_PERFORMER_USER);
        return this.mailboxSessionMapperFactory.getMailboxMapper(mailboxSession).findMailboxById(previousFailure.getMailboxId()).map(mailbox -> Either.right((Object)new ReIndexingEntry((Mailbox)mailbox, mailboxSession, previousFailure.getUid()))).onErrorResume(e -> {
            LOGGER.warn("ReIndexing failed for {}", (Object)previousFailure, e);
            return Mono.just((Object)Either.left((Object)new MessageFailure(previousFailure.getMailboxId(), previousFailure.getUid())));
        });
    }

    private Flux<Either<Failure, ReIndexingEntry>> reIndexingEntriesForMailbox(Mailbox mailbox, MailboxSession mailboxSession, ReIndexer.RunningOptions runningOptions) {
        MessageMapper messageMapper = this.mailboxSessionMapperFactory.getMessageMapper(mailboxSession);
        return this.updateSearchIndex(mailbox, mailboxSession, runningOptions).thenMany((Publisher)messageMapper.listAllMessageUids(mailbox)).map(uid -> Either.right((Object)new ReIndexingEntry(mailbox, mailboxSession, (MessageUid)uid))).onErrorResume(e -> {
            LOGGER.warn("ReIndexing failed for {}", (Object)mailbox.generateAssociatedPath(), e);
            return Mono.just((Object)Either.left((Object)new MailboxFailure(mailbox.getMailboxId())));
        });
    }

    private Mono<Void> updateSearchIndex(Mailbox mailbox, MailboxSession mailboxSession, ReIndexer.RunningOptions runningOptions) {
        switch (runningOptions.getMode()) {
            case REBUILD_ALL: {
                return this.messageSearchIndex.deleteAll(mailboxSession, mailbox.getMailboxId());
            }
        }
        return Mono.empty();
    }

    private Mono<Task.Result> reIndexMessages(Flux<Either<Failure, ReIndexingEntry>> entriesToIndex, ReIndexer.RunningOptions runningOptions, ReprocessingContext reprocessingContext) {
        return entriesToIndex.transform(ReactorUtils.throttle().elements(runningOptions.getMessagesPerSecond()).per(Duration.ofSeconds(1L)).forOperation(entry -> this.reIndex((Either<Failure, ReIndexingEntry>)entry, reprocessingContext, runningOptions))).reduce(Task::combine).switchIfEmpty(Mono.just((Object)Task.Result.COMPLETED));
    }

    private Mono<Task.Result> reIndex(Either<Failure, ReIndexingEntry> failureOrEntry, ReprocessingContext reprocessingContext, ReIndexer.RunningOptions runningOptions) {
        return this.toMono(failureOrEntry.map(entry -> this.reIndex((ReIndexingEntry)entry, runningOptions))).map(this::flatten).map(failureOrTaskResult -> this.recordIndexingResult((Either<Failure, Task.Result>)failureOrTaskResult, reprocessingContext));
    }

    private Task.Result recordIndexingResult(Either<Failure, Task.Result> failureOrTaskResult, ReprocessingContext reprocessingContext) {
        return (Task.Result)failureOrTaskResult.fold(failure -> {
            failure.recordFailure(reprocessingContext);
            return Task.Result.PARTIAL;
        }, result -> {
            Task.Operation[] operationArray = new Task.Operation[1];
            operationArray[0] = reprocessingContext::recordSuccess;
            return result.onComplete(operationArray);
        });
    }

    private Mono<Either<Failure, Task.Result>> reIndex(ReIndexingEntry entry, ReIndexer.RunningOptions runningOptions) {
        if (runningOptions.getMode() == ReIndexer.RunningOptions.Mode.FIX_OUTDATED) {
            return this.correctIfNeeded(entry);
        }
        return this.index(entry);
    }

    private Mono<Either<Failure, Task.Result>> index(ReIndexingEntry entry) {
        return this.fullyReadMessage(entry).flatMap(message -> this.messageSearchIndex.add(entry.getMailboxSession(), entry.getMailbox(), message)).thenReturn((Object)Either.right((Object)Task.Result.COMPLETED)).onErrorResume(e -> {
            LOGGER.warn("ReIndexing failed for {} {}", new Object[]{entry.getMailbox().generateAssociatedPath(), entry.getUid(), e});
            return Mono.just((Object)Either.left((Object)new MessageFailure(entry.getMailbox().getMailboxId(), entry.getUid())));
        });
    }

    private Mono<Either<Failure, Task.Result>> correctIfNeeded(ReIndexingEntry entry) {
        MessageMapper messageMapper = this.mailboxSessionMapperFactory.getMessageMapper(entry.getMailboxSession());
        return messageMapper.findInMailboxReactive(entry.getMailbox(), MessageRange.one((MessageUid)entry.getUid()), MessageMapper.FetchType.Metadata, 1).next().flatMap(message -> this.isIndexUpToDate(entry.getMailbox(), (MailboxMessage)message).flatMap(upToDate -> {
            if (upToDate.booleanValue()) {
                return Mono.just((Object)Either.right((Object)Task.Result.COMPLETED));
            }
            return this.correct(entry);
        })).onErrorResume(e -> {
            LOGGER.warn("ReIndexing failed for {} {}", new Object[]{entry.getMailbox().generateAssociatedPath(), entry.getUid(), e});
            return Mono.just((Object)Either.left((Object)new MessageFailure(entry.getMailbox().getMailboxId(), entry.getUid())));
        });
    }

    private Mono<Either<Failure, Task.Result>> correct(ReIndexingEntry entry) {
        return this.index(entry);
    }

    private Mono<Boolean> isIndexUpToDate(Mailbox mailbox, MailboxMessage message) {
        return this.messageSearchIndex.retrieveIndexedFlags(mailbox, message.getUid()).map(flags -> this.isIndexUpToDate(message, (Flags)flags)).switchIfEmpty(Mono.just((Object)false));
    }

    private boolean isIndexUpToDate(MailboxMessage message, Flags flags) {
        return message.createFlags().equals((Object)flags);
    }

    private <X, Y> Either<X, Y> flatten(Either<X, Either<X, Y>> nestedEither) {
        return (Either)nestedEither.getOrElseGet(Either::left);
    }

    private <X, Y> Mono<Either<X, Y>> toMono(Either<X, Mono<Y>> either) {
        return (Mono)either.fold(x -> Mono.just((Object)Either.left((Object)x)), yMono -> yMono.map(Either::right));
    }

    private static class MessageFailure
    implements Failure {
        private final MailboxId mailboxId;
        private final MessageUid uid;

        private MessageFailure(MailboxId mailboxId, MessageUid uid) {
            this.mailboxId = mailboxId;
            this.uid = uid;
        }

        public MailboxId getMailboxId() {
            return this.mailboxId;
        }

        public MessageUid getUid() {
            return this.uid;
        }

        @Override
        public void recordFailure(ReprocessingContext context) {
            context.recordFailureDetailsForMessage(this.mailboxId, this.uid);
        }
    }

    private static class MailboxFailure
    implements Failure {
        private final MailboxId mailboxId;

        private MailboxFailure(MailboxId mailboxId) {
            this.mailboxId = mailboxId;
        }

        public MailboxId getMailboxId() {
            return this.mailboxId;
        }

        @Override
        public void recordFailure(ReprocessingContext context) {
            context.recordMailboxFailure(this.mailboxId);
        }
    }

    private static interface Failure {
        public void recordFailure(ReprocessingContext var1);
    }

    private static class ReIndexingEntry {
        private final Mailbox mailbox;
        private final MailboxSession mailboxSession;
        private final MessageUid uid;

        ReIndexingEntry(Mailbox mailbox, MailboxSession mailboxSession, MessageUid uid) {
            this.mailbox = mailbox;
            this.mailboxSession = mailboxSession;
            this.uid = uid;
        }

        public Mailbox getMailbox() {
            return this.mailbox;
        }

        public MessageUid getUid() {
            return this.uid;
        }

        public MailboxSession getMailboxSession() {
            return this.mailboxSession;
        }
    }
}

