/*
 * Decompiled with CFR 0.152.
 */
package org.apache.james.blob.cassandra.cache;

import com.github.fge.lambdas.Throwing;
import com.google.common.base.Preconditions;
import com.google.common.io.ByteSource;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.PushbackInputStream;
import java.util.Arrays;
import java.util.Optional;
import java.util.function.Function;
import javax.inject.Inject;
import javax.inject.Named;
import org.apache.commons.io.IOUtils;
import org.apache.james.blob.api.BlobId;
import org.apache.james.blob.api.BlobStore;
import org.apache.james.blob.api.BucketName;
import org.apache.james.blob.api.ObjectNotFoundException;
import org.apache.james.blob.api.ObjectStoreIOException;
import org.apache.james.blob.cassandra.cache.BlobStoreCache;
import org.apache.james.blob.cassandra.cache.CassandraCacheConfiguration;
import org.apache.james.metrics.api.Metric;
import org.apache.james.metrics.api.MetricFactory;
import org.reactivestreams.Publisher;
import reactor.core.publisher.Mono;

public class CachedBlobStore
implements BlobStore {
    public static final String BACKEND = "blobStoreBackend";
    public static final String BLOBSTORE_CACHED_LATENCY_METRIC_NAME = "blobStoreCacheLatency";
    public static final String BLOBSTORE_BACKEND_LATENCY_METRIC_NAME = "blobStoreBackEndLatency";
    public static final String BLOBSTORE_CACHED_MISS_COUNT_METRIC_NAME = "blobStoreCacheMisses";
    public static final String BLOBSTORE_CACHED_HIT_COUNT_METRIC_NAME = "blobStoreCacheHits";
    private final MetricFactory metricFactory;
    private final Metric metricRetrieveHitCount;
    private final Metric metricRetrieveMissCount;
    private final BlobStoreCache cache;
    private final BlobStore backend;
    private final Integer sizeThresholdInBytes;

    @Inject
    public CachedBlobStore(BlobStoreCache cache, @Named(value="blobStoreBackend") BlobStore backend, CassandraCacheConfiguration cacheConfiguration, MetricFactory metricFactory) {
        this.cache = cache;
        this.backend = backend;
        this.sizeThresholdInBytes = cacheConfiguration.getSizeThresholdInBytes();
        this.metricFactory = metricFactory;
        this.metricRetrieveMissCount = metricFactory.generate(BLOBSTORE_CACHED_MISS_COUNT_METRIC_NAME);
        this.metricRetrieveHitCount = metricFactory.generate(BLOBSTORE_CACHED_HIT_COUNT_METRIC_NAME);
    }

    public InputStream read(BucketName bucketName, BlobId blobId, BlobStore.StoragePolicy storagePolicy) throws ObjectStoreIOException, ObjectNotFoundException {
        if (storagePolicy == BlobStore.StoragePolicy.LOW_COST) {
            return this.backend.read(bucketName, blobId);
        }
        return (InputStream)Mono.just((Object)bucketName).filter(arg_0 -> ((BucketName)this.getDefaultBucketName()).equals(arg_0)).flatMap(defaultBucket -> this.readInDefaultBucket(bucketName, blobId)).switchIfEmpty(this.readFromBackend(bucketName, blobId)).blockOptional().orElseThrow(() -> new ObjectNotFoundException(String.format("Could not retrieve blob metadata for %s", blobId.asString())));
    }

    private Mono<InputStream> readInDefaultBucket(BucketName bucketName, BlobId blobId) {
        return this.readFromCache(blobId).flatMap(this::toInputStream).switchIfEmpty(this.readFromBackend(bucketName, blobId).flatMap(inputStream -> Mono.fromCallable(() -> ReadAheadInputStream.eager().of((InputStream)inputStream).length(this.sizeThresholdInBytes)).flatMap(readAheadInputStream -> this.putInCacheIfNeeded(bucketName, (ReadAheadInputStream)readAheadInputStream, blobId).thenReturn((Object)readAheadInputStream.in))));
    }

    public Mono<byte[]> readBytes(BucketName bucketName, BlobId blobId, BlobStore.StoragePolicy storagePolicy) {
        if (storagePolicy == BlobStore.StoragePolicy.LOW_COST) {
            return this.readBytesFromBackend(bucketName, blobId);
        }
        if (this.getDefaultBucketName().equals((Object)bucketName)) {
            return this.readBytesInDefaultBucket(bucketName, blobId);
        }
        return this.readBytesFromBackend(bucketName, blobId);
    }

    public Publisher<byte[]> readBytes(BucketName bucketName, BlobId blobId) {
        return this.readBytes(bucketName, blobId, BlobStore.StoragePolicy.LOW_COST);
    }

    public InputStream read(BucketName bucketName, BlobId blobId) {
        return this.read(bucketName, blobId, BlobStore.StoragePolicy.LOW_COST);
    }

    private Mono<byte[]> readBytesInDefaultBucket(BucketName bucketName, BlobId blobId) {
        return this.readFromCache(blobId).switchIfEmpty(this.readBytesFromBackend(bucketName, blobId).flatMap(bytes -> {
            if (this.isAbleToCache((byte[])bytes)) {
                this.metricRetrieveMissCount.increment();
                return this.saveInCache(blobId, (byte[])bytes).then(Mono.just((Object)bytes));
            }
            return Mono.just((Object)bytes);
        }));
    }

    public Mono<BlobId> save(BucketName bucketName, byte[] bytes, BlobStore.StoragePolicy storagePolicy) {
        return Mono.from((Publisher)this.backend.save(bucketName, bytes, storagePolicy)).flatMap(blobId -> {
            if (this.isAbleToCache(bucketName, bytes, storagePolicy)) {
                return this.saveInCache((BlobId)blobId, bytes).thenReturn(blobId);
            }
            return Mono.just((Object)blobId);
        });
    }

    public Publisher<BlobId> save(BucketName bucketName, InputStream inputStream, BlobStore.StoragePolicy storagePolicy) {
        Preconditions.checkNotNull((Object)inputStream, (Object)"InputStream must not be null");
        if (this.isAbleToCache(bucketName, storagePolicy)) {
            return this.saveInCache(bucketName, inputStream, storagePolicy);
        }
        return this.backend.save(bucketName, inputStream, storagePolicy);
    }

    public Publisher<BlobId> save(BucketName bucketName, ByteSource byteSource, BlobStore.StoragePolicy storagePolicy) {
        Preconditions.checkNotNull((Object)byteSource, (Object)"ByteSource must not be null");
        if (this.isAbleToCache(bucketName, storagePolicy)) {
            return this.saveInCache(bucketName, byteSource, storagePolicy);
        }
        return this.backend.save(bucketName, byteSource, storagePolicy);
    }

    private Mono<BlobId> saveInCache(BucketName bucketName, ByteSource byteSource, BlobStore.StoragePolicy storagePolicy) {
        return Mono.from((Publisher)this.backend.save(bucketName, byteSource, storagePolicy)).flatMap((Function)Throwing.function(blobId -> ReadAheadInputStream.eager().of((InputStream)byteSource.openStream()).length((int)this.sizeThresholdInBytes.intValue()).firstBytes.map(bytes -> Mono.from(this.cache.cache((BlobId)blobId, (byte[])bytes)).thenReturn(blobId)).orElse(Mono.just((Object)blobId))));
    }

    public BucketName getDefaultBucketName() {
        return this.backend.getDefaultBucketName();
    }

    public Mono<Boolean> delete(BucketName bucketName, BlobId blobId) {
        return Mono.from((Publisher)this.backend.delete(bucketName, blobId)).flatMap(deleted -> {
            if (this.backend.getDefaultBucketName().equals((Object)bucketName) && deleted.booleanValue()) {
                return Mono.from(this.cache.remove(blobId)).thenReturn(deleted);
            }
            return Mono.just((Object)deleted);
        });
    }

    public Publisher<Void> deleteBucket(BucketName bucketName) {
        return Mono.from((Publisher)this.backend.deleteBucket(bucketName));
    }

    private Mono<BlobId> saveInCache(BucketName bucketName, InputStream inputStream, BlobStore.StoragePolicy storagePolicy) {
        return Mono.fromCallable(() -> ReadAheadInputStream.eager().of(inputStream).length(this.sizeThresholdInBytes)).flatMap(readAhead -> this.saveToBackend(bucketName, storagePolicy, (ReadAheadInputStream)readAhead).flatMap(blobId -> this.putInCacheIfNeeded(bucketName, storagePolicy, (ReadAheadInputStream)readAhead, (BlobId)blobId).thenReturn(blobId)));
    }

    private Mono<BlobId> saveToBackend(BucketName bucketName, BlobStore.StoragePolicy storagePolicy, ReadAheadInputStream readAhead) {
        return Mono.from((Publisher)this.backend.save(bucketName, (InputStream)readAhead.in, storagePolicy));
    }

    private Mono<Void> putInCacheIfNeeded(BucketName bucketName, BlobStore.StoragePolicy storagePolicy, ReadAheadInputStream readAhead, BlobId blobId) {
        return Mono.justOrEmpty(readAhead.firstBytes).filter(bytes -> this.isAbleToCache(bucketName, readAhead, storagePolicy)).flatMap(bytes -> Mono.from(this.cache.cache(blobId, (byte[])bytes)));
    }

    private Mono<Void> putInCacheIfNeeded(BucketName bucketName, ReadAheadInputStream readAhead, BlobId blobId) {
        return Mono.justOrEmpty(readAhead.firstBytes).filter(bytes -> this.isAbleToCache(readAhead, bucketName)).doOnNext(any -> this.metricRetrieveMissCount.increment()).flatMap(bytes -> Mono.from(this.cache.cache(blobId, (byte[])bytes)));
    }

    private Mono<Void> saveInCache(BlobId blobId, byte[] bytes) {
        return Mono.from(this.cache.cache(blobId, bytes));
    }

    private boolean isAbleToCache(BucketName bucketName, byte[] bytes, BlobStore.StoragePolicy storagePolicy) {
        return this.isAbleToCache(bucketName, storagePolicy) && this.isAbleToCache(bytes);
    }

    private boolean isAbleToCache(BucketName bucketName, ReadAheadInputStream readAhead, BlobStore.StoragePolicy storagePolicy) {
        return this.isAbleToCache(bucketName, storagePolicy) && !readAhead.hasMore;
    }

    private boolean isAbleToCache(BucketName bucketName, BlobStore.StoragePolicy storagePolicy) {
        return this.backend.getDefaultBucketName().equals((Object)bucketName) && !storagePolicy.equals((Object)BlobStore.StoragePolicy.LOW_COST);
    }

    private boolean isAbleToCache(ReadAheadInputStream readAhead, BucketName bucketName) {
        return !readAhead.hasMore && this.backend.getDefaultBucketName().equals((Object)bucketName);
    }

    private boolean isAbleToCache(byte[] bytes) {
        return bytes.length <= this.sizeThresholdInBytes;
    }

    private Mono<InputStream> toInputStream(byte[] bytes) {
        return Mono.fromCallable(() -> new ByteArrayInputStream(bytes));
    }

    private Mono<byte[]> readFromCache(BlobId blobId) {
        return Mono.from((Publisher)this.metricFactory.decoratePublisherWithTimerMetric(BLOBSTORE_CACHED_LATENCY_METRIC_NAME, this.cache.read(blobId))).doOnNext(any -> this.metricRetrieveHitCount.increment());
    }

    private Mono<InputStream> readFromBackend(BucketName bucketName, BlobId blobId) {
        return Mono.from((Publisher)this.metricFactory.decoratePublisherWithTimerMetric(BLOBSTORE_BACKEND_LATENCY_METRIC_NAME, (Publisher)Mono.fromCallable(() -> this.backend.read(bucketName, blobId))));
    }

    private Mono<byte[]> readBytesFromBackend(BucketName bucketName, BlobId blobId) {
        return Mono.fromCallable(() -> this.metricFactory.timer(BLOBSTORE_BACKEND_LATENCY_METRIC_NAME)).flatMap(timer -> Mono.from((Publisher)this.backend.readBytes(bucketName, blobId)).doOnSuccess(any -> timer.stopAndPublish()).doOnError(ObjectNotFoundException.class, any -> timer.stopAndPublish()));
    }

    private static class ReadAheadInputStream {
        final PushbackInputStream in;
        final Optional<byte[]> firstBytes;
        final boolean hasMore;

        static RequireStream eager() {
            return in -> length -> {
                boolean hasMore;
                Optional<byte[]> firstBytes;
                PushbackInputStream stream = new PushbackInputStream(in, length + 1);
                byte[] bytes = new byte[length];
                int readByteCount = IOUtils.read((InputStream)stream, (byte[])bytes);
                if (readByteCount < 0) {
                    firstBytes = Optional.empty();
                    hasMore = false;
                } else {
                    byte[] readBytes = Arrays.copyOf(bytes, readByteCount);
                    hasMore = ReadAheadInputStream.hasMore(stream);
                    stream.unread(readBytes);
                    firstBytes = Optional.of(readBytes);
                }
                return new ReadAheadInputStream(stream, firstBytes, hasMore);
            };
        }

        private static boolean hasMore(PushbackInputStream stream) throws IOException {
            int nextByte = stream.read();
            if (nextByte >= 0) {
                stream.unread(nextByte);
                return true;
            }
            return false;
        }

        private ReadAheadInputStream(PushbackInputStream in, Optional<byte[]> firstBytes, boolean hasMore) {
            this.in = in;
            this.firstBytes = firstBytes;
            this.hasMore = hasMore;
        }

        static interface RequireLength {
            public ReadAheadInputStream length(int var1) throws IOException;
        }

        @FunctionalInterface
        static interface RequireStream {
            public RequireLength of(InputStream var1);
        }
    }
}

