/*
 * Decompiled with CFR 0.152.
 */
package org.apache.james.dnsservice.dnsjava;

import com.google.common.collect.ImmutableList;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import javax.annotation.PostConstruct;
import javax.inject.Inject;
import org.apache.commons.configuration2.HierarchicalConfiguration;
import org.apache.commons.configuration2.ex.ConfigurationException;
import org.apache.commons.configuration2.tree.ImmutableNode;
import org.apache.james.dnsservice.api.DNSService;
import org.apache.james.dnsservice.api.DNSServiceMBean;
import org.apache.james.dnsservice.api.TemporaryResolutionException;
import org.apache.james.lifecycle.api.Configurable;
import org.apache.james.metrics.api.MetricFactory;
import org.apache.james.metrics.api.TimeMetric;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.xbill.DNS.ARecord;
import org.xbill.DNS.Address;
import org.xbill.DNS.Cache;
import org.xbill.DNS.ExtendedResolver;
import org.xbill.DNS.Lookup;
import org.xbill.DNS.MXRecord;
import org.xbill.DNS.Name;
import org.xbill.DNS.PTRRecord;
import org.xbill.DNS.Record;
import org.xbill.DNS.Resolver;
import org.xbill.DNS.ResolverConfig;
import org.xbill.DNS.ReverseMap;
import org.xbill.DNS.TXTRecord;
import org.xbill.DNS.TextParseException;

public class DNSJavaService
implements DNSService,
DNSServiceMBean,
Configurable {
    private static final Logger LOGGER = LoggerFactory.getLogger(DNSJavaService.class);
    private static final int CACHE_TTL_DISABLE = -1;
    protected Resolver resolver;
    protected Cache cache;
    private int maxCacheSize = 50000;
    private int negativeCacheTTL;
    private int dnsCredibility;
    private final List<String> dnsServers = new ArrayList<String>();
    private final MetricFactory metricFactory;
    private Name[] searchPaths = null;
    private final Comparator<MXRecord> mxComparator = Comparator.comparing(MXRecord::getPriority);
    private boolean setAsDNSJavaDefault;
    private String localHostName;
    private String localCanonicalHostName;
    private String localAddress;

    @Inject
    public DNSJavaService(MetricFactory metricFactory) {
        this.metricFactory = metricFactory;
    }

    public void configure(HierarchicalConfiguration<ImmutableNode> configuration) throws ConfigurationException {
        boolean authoritative;
        boolean verbose = configuration.getBoolean("verbose", false);
        if (verbose) {
            System.setProperty("dnsjava.options", "verbose,verbosemsg,verbosecache");
        }
        boolean autodiscover = configuration.getBoolean("autodiscover", true);
        ArrayList<Name> sPaths = new ArrayList<Name>();
        if (autodiscover) {
            Name[] systemSearchPath;
            LOGGER.info("Autodiscovery is enabled - trying to discover your system's DNS Servers");
            String[] serversArray = ResolverConfig.getCurrentConfig().servers();
            if (serversArray != null) {
                String[] stringArray = serversArray;
                int n = stringArray.length;
                for (int i = 0; i < n; ++i) {
                    String aServersArray = stringArray[i];
                    this.dnsServers.add(aServersArray);
                    LOGGER.info("Adding autodiscovered server {}", (Object)aServersArray);
                }
            }
            if ((systemSearchPath = ResolverConfig.getCurrentConfig().searchPath()) != null && systemSearchPath.length > 0) {
                sPaths.addAll(Arrays.asList(systemSearchPath));
            }
            if (LOGGER.isInfoEnabled()) {
                for (Name searchPath : sPaths) {
                    LOGGER.info("Adding autodiscovered search path " + searchPath);
                }
            }
        }
        this.setAsDNSJavaDefault = configuration.getBoolean("setAsDNSJavaDefault", true);
        Collections.addAll(this.dnsServers, configuration.getStringArray("servers.server"));
        for (String aSearchPathsConfiguration : configuration.getStringArray("searchpaths.searchpath")) {
            try {
                sPaths.add(Name.fromString((String)aSearchPathsConfiguration));
            }
            catch (TextParseException e) {
                throw new ConfigurationException("Unable to parse searchpath host: " + aSearchPathsConfiguration, (Throwable)e);
            }
        }
        this.searchPaths = sPaths.toArray(new Name[sPaths.size()]);
        if (this.dnsServers.isEmpty()) {
            LOGGER.info("No DNS servers have been specified or found by autodiscovery - adding 127.0.0.1");
            this.dnsServers.add("127.0.0.1");
        }
        this.dnsCredibility = (authoritative = configuration.getBoolean("authoritative", false)) ? 4 : 3;
        this.maxCacheSize = configuration.getInt("maxcachesize", this.maxCacheSize);
        this.negativeCacheTTL = configuration.getInt("negativeCacheTTL", -1);
    }

    @PostConstruct
    public void init() throws Exception {
        LOGGER.debug("DNSService init...");
        if (this.dnsServers.isEmpty()) {
            try {
                this.dnsServers.add(InetAddress.getLocalHost().getHostName());
            }
            catch (UnknownHostException ue) {
                this.dnsServers.add("127.0.0.1");
            }
        }
        String[] serversArray = this.dnsServers.toArray(new String[this.dnsServers.size()]);
        if (LOGGER.isInfoEnabled()) {
            for (String aServersArray : serversArray) {
                LOGGER.info("DNS Server is: " + aServersArray);
            }
        }
        try {
            this.resolver = new ExtendedResolver(serversArray);
        }
        catch (UnknownHostException uhe) {
            LOGGER.error("DNS service could not be initialized.  The DNS servers specified are not recognized hosts.", (Throwable)uhe);
            throw uhe;
        }
        this.cache = new Cache(1);
        this.cache.setMaxEntries(this.maxCacheSize);
        this.cache.setMaxNCache(this.negativeCacheTTL);
        if (this.setAsDNSJavaDefault) {
            Lookup.setDefaultResolver((Resolver)this.resolver);
            Lookup.setDefaultCache((Cache)this.cache, (int)1);
            Lookup.setDefaultSearchPath((Name[])this.searchPaths);
            LOGGER.info("Registered cache, resolver and search paths as DNSJava defaults");
        }
        InetAddress addr = this.getLocalHost();
        this.localCanonicalHostName = addr.getCanonicalHostName();
        this.localHostName = addr.getHostName();
        this.localAddress = addr.getHostAddress();
        LOGGER.debug("DNSService ...init end");
    }

    public String[] getDNSServers() {
        return this.dnsServers.toArray(new String[this.dnsServers.size()]);
    }

    public Name[] getSearchPaths() {
        return this.searchPaths;
    }

    private List<String> findMXRecordsRaw(String hostname) throws TemporaryResolutionException {
        Record[] answers = this.lookup(hostname, 15, "MX");
        ArrayList<String> servers = new ArrayList<String>();
        if (answers == null) {
            return servers;
        }
        MXRecord[] mxAnswers = new MXRecord[answers.length];
        for (int i = 0; i < answers.length; ++i) {
            mxAnswers[i] = (MXRecord)answers[i];
        }
        Arrays.sort(mxAnswers, this.mxComparator);
        int currentPrio = -1;
        ArrayList<String> samePrio = new ArrayList<String>();
        for (int i = 0; i < mxAnswers.length; ++i) {
            boolean same = false;
            boolean lastItem = i + 1 == mxAnswers.length;
            MXRecord mx = mxAnswers[i];
            if (i == 0) {
                currentPrio = mx.getPriority();
            } else {
                same = currentPrio == mx.getPriority();
            }
            String mxRecord = mx.getTarget().toString();
            if (same) {
                samePrio.add(mxRecord);
            } else {
                Collections.shuffle(samePrio);
                servers.addAll(samePrio);
                samePrio.clear();
                samePrio.add(mxRecord);
            }
            if (lastItem) {
                Collections.shuffle(samePrio);
                servers.addAll(samePrio);
            }
            LOGGER.debug("Found MX record {}", (Object)mxRecord);
        }
        return servers;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public Collection<String> findMXRecords(String hostname) throws TemporaryResolutionException {
        TimeMetric timeMetric = this.metricFactory.timer("findMXRecords");
        List<Object> servers = new ArrayList();
        try {
            servers = this.findMXRecordsRaw(hostname);
            Collection<String> collection = Collections.unmodifiableCollection(servers);
            return collection;
        }
        finally {
            if (servers.size() == 0) {
                LOGGER.info("Couldn't resolve MX records for domain {}.", (Object)hostname);
                try {
                    this.getByName(hostname);
                    servers.add(hostname);
                }
                catch (UnknownHostException uhe) {
                    LOGGER.error("Couldn't resolve IP address for host {}.", (Object)hostname, (Object)uhe);
                }
            }
            timeMetric.stopAndPublish();
        }
    }

    protected Record[] lookup(String namestr, int type, String typeDesc) throws TemporaryResolutionException {
        try {
            Lookup l = new Lookup(namestr, type);
            l.setCache(this.cache);
            l.setResolver(this.resolver);
            l.setCredibility(this.dnsCredibility);
            l.setSearchPath(this.searchPaths);
            Record[] r = l.run();
            try {
                if (l.getResult() == 2) {
                    throw new TemporaryResolutionException("DNSService is temporary not reachable");
                }
                return r;
            }
            catch (IllegalStateException ise) {
                LOGGER.warn("Error determining result ", (Throwable)ise);
                throw new TemporaryResolutionException("DNSService is temporary not reachable");
            }
        }
        catch (TextParseException tpe) {
            LOGGER.error("Couldn't parse name {}", (Object)namestr, (Object)tpe);
            return null;
        }
    }

    protected Record[] lookupNoException(String namestr, int type, String typeDesc) {
        try {
            return this.lookup(namestr, type, typeDesc);
        }
        catch (TemporaryResolutionException e) {
            return null;
        }
    }

    private static String allowIPLiteral(String host) {
        String possibleIpLiteral;
        if (host.charAt(host.length() - 1) == '.' && Address.isDottedQuad((String)(possibleIpLiteral = host.substring(0, host.length() - 1)))) {
            host = possibleIpLiteral;
        }
        return host;
    }

    public InetAddress getByName(String host) throws UnknownHostException {
        TimeMetric timeMetric = this.metricFactory.timer("getByName");
        String name = DNSJavaService.allowIPLiteral(host);
        try {
            if (name.equalsIgnoreCase(this.localHostName) || name.equalsIgnoreCase(this.localCanonicalHostName) || name.equals(this.localAddress)) {
                InetAddress inetAddress = this.getLocalHost();
                return inetAddress;
            }
            InetAddress inetAddress = Address.getByAddress((String)name);
            return inetAddress;
        }
        catch (UnknownHostException e) {
            Record[] records = this.lookupNoException(name, 1, "A");
            if (records != null && records.length >= 1) {
                ARecord a = (ARecord)records[0];
                InetAddress inetAddress = InetAddress.getByAddress(name, a.getAddress().getAddress());
                return inetAddress;
            }
            throw e;
        }
        finally {
            timeMetric.stopAndPublish();
        }
    }

    public Collection<InetAddress> getAllByName(String host) throws UnknownHostException {
        TimeMetric timeMetric = this.metricFactory.timer("getAllByName");
        String name = DNSJavaService.allowIPLiteral(host);
        try {
            if (name.equalsIgnoreCase(this.localHostName) || name.equalsIgnoreCase(this.localCanonicalHostName) || name.equals(this.localAddress)) {
                ImmutableList immutableList = ImmutableList.of((Object)this.getLocalHost());
                return immutableList;
            }
            InetAddress addr = Address.getByAddress((String)name);
            ImmutableList immutableList = ImmutableList.of((Object)addr);
            return immutableList;
        }
        catch (UnknownHostException e) {
            Record[] records = this.lookupNoException(name, 1, "A");
            if (records != null && records.length >= 1) {
                Object[] addrs = new InetAddress[records.length];
                for (int i = 0; i < records.length; ++i) {
                    ARecord a = (ARecord)records[i];
                    addrs[i] = InetAddress.getByAddress(name, a.getAddress().getAddress());
                }
                ImmutableList immutableList = ImmutableList.copyOf((Object[])addrs);
                return immutableList;
            }
            throw e;
        }
        finally {
            timeMetric.stopAndPublish();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public Collection<String> findTXTRecords(String hostname) {
        TimeMetric timeMetric = this.metricFactory.timer("findTXTRecords");
        ArrayList<String> txtR = new ArrayList<String>();
        Record[] records = this.lookupNoException(hostname, 16, "TXT");
        try {
            if (records != null) {
                for (Record record : records) {
                    TXTRecord txt = (TXTRecord)record;
                    txtR.add(txt.rdataToString());
                }
            }
            ArrayList<String> arrayList = txtR;
            return arrayList;
        }
        finally {
            timeMetric.stopAndPublish();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public String getHostName(InetAddress addr) {
        TimeMetric timeMetric = this.metricFactory.timer("getHostName");
        Name name = ReverseMap.fromAddress((InetAddress)addr);
        Record[] records = this.lookupNoException(name.toString(), 12, "PTR");
        try {
            String result;
            if (records == null) {
                result = addr.getHostAddress();
            } else {
                PTRRecord ptr = (PTRRecord)records[0];
                result = ptr.getTarget().toString();
            }
            String string = result;
            return string;
        }
        finally {
            timeMetric.stopAndPublish();
        }
    }

    public InetAddress getLocalHost() throws UnknownHostException {
        return InetAddress.getLocalHost();
    }

    public int getMaximumCacheSize() {
        return this.maxCacheSize;
    }

    public int getCurrentCacheSize() {
        return this.cache.getSize();
    }

    public void clearCache() {
        this.cache.clearCache();
    }
}

