/*
 * Decompiled with CFR 0.152.
 */
package org.apache.james.mailbox.store.search;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.Lists;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.Reader;
import java.io.SequenceInputStream;
import java.io.StringReader;
import java.text.SimpleDateFormat;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Collection;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Optional;
import java.util.Set;
import java.util.TimeZone;
import java.util.stream.Stream;
import javax.mail.Flags;
import org.apache.james.mailbox.MessageUid;
import org.apache.james.mailbox.ModSeq;
import org.apache.james.mailbox.exception.MailboxException;
import org.apache.james.mailbox.exception.UnsupportedSearchException;
import org.apache.james.mailbox.extractor.TextExtractor;
import org.apache.james.mailbox.model.Attachment;
import org.apache.james.mailbox.model.Header;
import org.apache.james.mailbox.model.MessageAttachment;
import org.apache.james.mailbox.model.SearchQuery;
import org.apache.james.mailbox.store.ResultUtils;
import org.apache.james.mailbox.store.mail.model.MailboxMessage;
import org.apache.james.mailbox.store.mail.model.impl.PropertyBuilder;
import org.apache.james.mailbox.store.search.MessageSearchIndex;
import org.apache.james.mailbox.store.search.comparator.CombinedComparator;
import org.apache.james.mime4j.MimeException;
import org.apache.james.mime4j.MimeIOException;
import org.apache.james.mime4j.dom.Message;
import org.apache.james.mime4j.dom.address.Address;
import org.apache.james.mime4j.dom.address.AddressList;
import org.apache.james.mime4j.dom.address.Group;
import org.apache.james.mime4j.dom.address.Mailbox;
import org.apache.james.mime4j.dom.address.MailboxList;
import org.apache.james.mime4j.dom.datetime.DateTime;
import org.apache.james.mime4j.field.Fields;
import org.apache.james.mime4j.field.address.AddressFormatter;
import org.apache.james.mime4j.field.address.LenientAddressParser;
import org.apache.james.mime4j.field.datetime.parser.DateTimeParser;
import org.apache.james.mime4j.field.datetime.parser.ParseException;
import org.apache.james.mime4j.message.DefaultMessageBuilder;
import org.apache.james.mime4j.message.DefaultMessageWriter;
import org.apache.james.mime4j.message.HeaderImpl;
import org.apache.james.mime4j.stream.Field;
import org.apache.james.mime4j.stream.MimeConfig;
import org.apache.james.mime4j.util.MimeUtil;
import org.apache.james.mime4j.utils.search.MessageMatcher;
import org.apache.james.util.OptionalUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MessageSearches
implements Iterable<MessageSearchIndex.SearchResult> {
    private static final Logger LOGGER = LoggerFactory.getLogger(MessageSearches.class);
    private final Iterator<MailboxMessage> messages;
    private final SearchQuery query;
    private final TextExtractor textExtractor;

    public MessageSearches(Iterator<MailboxMessage> messages, SearchQuery query, TextExtractor textExtractor) {
        this.messages = messages;
        this.query = query;
        this.textExtractor = textExtractor;
    }

    @Override
    public Iterator<MessageSearchIndex.SearchResult> iterator() {
        ImmutableList.Builder builder = ImmutableList.builder();
        while (this.messages.hasNext()) {
            MailboxMessage m = this.messages.next();
            try {
                if (!this.isMatch(m)) continue;
                builder.add((Object)m);
            }
            catch (MailboxException e) {
                LOGGER.error("Unable to search message {}", (Object)m.getUid(), (Object)e);
            }
        }
        return builder.build().stream().sorted(CombinedComparator.create(this.query.getSorts())).map(mailboxMessage -> new MessageSearchIndex.SearchResult(Optional.of(mailboxMessage.getMessageId()), mailboxMessage.getMailboxId(), mailboxMessage.getUid())).iterator();
    }

    private boolean isMatch(MailboxMessage message) throws MailboxException {
        List criteria = this.query.getCriterias();
        Set recentMessageUids = this.query.getRecentMessageUids();
        if (criteria != null) {
            for (SearchQuery.Criterion criterion : criteria) {
                if (this.isMatch(criterion, message, recentMessageUids)) continue;
                return false;
            }
        }
        return true;
    }

    public boolean isMatch(SearchQuery.Criterion criterion, MailboxMessage message, Collection<MessageUid> recentMessageUids) throws MailboxException {
        if (criterion instanceof SearchQuery.InternalDateCriterion) {
            return this.matches((SearchQuery.InternalDateCriterion)criterion, message);
        }
        if (criterion instanceof SearchQuery.SizeCriterion) {
            return this.matches((SearchQuery.SizeCriterion)criterion, message);
        }
        if (criterion instanceof SearchQuery.HeaderCriterion) {
            try {
                return this.matches((SearchQuery.HeaderCriterion)criterion, message);
            }
            catch (IOException e) {
                throw new MailboxException("Unable to search header", (Throwable)e);
            }
        }
        if (criterion instanceof SearchQuery.UidCriterion) {
            return this.matches((SearchQuery.UidCriterion)criterion, message);
        }
        if (criterion instanceof SearchQuery.FlagCriterion) {
            return this.matches((SearchQuery.FlagCriterion)criterion, message, recentMessageUids);
        }
        if (criterion instanceof SearchQuery.CustomFlagCriterion) {
            return this.matches((SearchQuery.CustomFlagCriterion)criterion, message);
        }
        if (criterion instanceof SearchQuery.TextCriterion) {
            return this.matches((SearchQuery.TextCriterion)criterion, message);
        }
        if (criterion instanceof SearchQuery.AllCriterion) {
            return true;
        }
        if (criterion instanceof SearchQuery.ConjunctionCriterion) {
            return this.matches((SearchQuery.ConjunctionCriterion)criterion, message, recentMessageUids);
        }
        if (criterion instanceof SearchQuery.AttachmentCriterion) {
            return this.matches((SearchQuery.AttachmentCriterion)criterion, message);
        }
        if (criterion instanceof SearchQuery.ModSeqCriterion) {
            return this.matches((SearchQuery.ModSeqCriterion)criterion, message);
        }
        if (criterion instanceof SearchQuery.MimeMessageIDCriterion) {
            SearchQuery.MimeMessageIDCriterion mimeMessageIDCriterion = (SearchQuery.MimeMessageIDCriterion)criterion;
            return this.isMatch((SearchQuery.Criterion)mimeMessageIDCriterion.asHeaderCriterion(), message, recentMessageUids);
        }
        throw new UnsupportedSearchException();
    }

    private boolean matches(SearchQuery.TextCriterion criterion, MailboxMessage message) throws MailboxException {
        try {
            SearchQuery.ContainsOperator operator = criterion.getOperator();
            String value = operator.getValue();
            switch (criterion.getType()) {
                case BODY: {
                    return this.bodyContains(value, message);
                }
                case TEXT: {
                    return this.textContains(value, message);
                }
                case FULL: {
                    return this.messageContains(value, message);
                }
                case ATTACHMENTS: {
                    return this.attachmentsContain(value, message);
                }
                case ATTACHMENT_FILE_NAME: {
                    return this.hasFileName(value, message);
                }
            }
            throw new UnsupportedSearchException();
        }
        catch (IOException | MimeException e) {
            throw new MailboxException("Unable to parse message", e);
        }
    }

    private boolean bodyContains(String value, MailboxMessage message) throws IOException, MimeException {
        InputStream input = message.getFullContent();
        return this.isInMessage(value, input, false);
    }

    private boolean isInMessage(String value, InputStream input, boolean header) throws IOException, MimeException {
        return MessageMatcher.builder().searchContents((List)Lists.newArrayList((Object[])new CharSequence[]{value})).caseInsensitive(true).includeHeaders(header).logger(LOGGER).build().messageMatches(input);
    }

    private boolean messageContains(String value, MailboxMessage message) throws IOException, MimeException {
        InputStream input = message.getFullContent();
        return this.isInMessage(value, input, true);
    }

    private boolean textContains(String value, MailboxMessage message) throws IOException, MimeException, MailboxException {
        InputStream bodyContent = message.getBodyContent();
        return this.isInMessage(value, new SequenceInputStream(this.textHeaders(message), bodyContent), true);
    }

    private boolean attachmentsContain(String value, MailboxMessage message) throws IOException, MimeException {
        List<MessageAttachment> attachments = message.getAttachments();
        return this.isInAttachments(value, attachments);
    }

    private boolean hasFileName(String value, MailboxMessage message) throws IOException, MimeException {
        return message.getAttachments().stream().map(MessageAttachment::getName).anyMatch(nameOptional -> nameOptional.map(value::equals).orElse(false));
    }

    private boolean isInAttachments(String value, List<MessageAttachment> attachments) {
        return attachments.stream().map(MessageAttachment::getAttachment).flatMap(this::toAttachmentContent).anyMatch(string -> string.contains(value));
    }

    private Stream<String> toAttachmentContent(Attachment attachment) {
        try {
            return OptionalUtils.toStream((Optional)this.textExtractor.extractContent(attachment.getStream(), attachment.getType()).getTextualContent());
        }
        catch (Exception e) {
            LOGGER.error("Error while parsing attachment content", (Throwable)e);
            return Stream.of(new String[0]);
        }
    }

    private InputStream textHeaders(MailboxMessage message) throws MimeIOException, IOException {
        ByteArrayOutputStream out = new ByteArrayOutputStream();
        new DefaultMessageWriter().writeHeader((org.apache.james.mime4j.dom.Header)this.buildTextHeaders(message), (OutputStream)out);
        return new ByteArrayInputStream(out.toByteArray());
    }

    private HeaderImpl buildTextHeaders(MailboxMessage message) throws IOException, MimeIOException {
        DefaultMessageBuilder defaultMessageBuilder = new DefaultMessageBuilder();
        defaultMessageBuilder.setMimeEntityConfig(MimeConfig.PERMISSIVE);
        Message headersMessage = defaultMessageBuilder.parseMessage(message.getHeaderContent());
        HeaderImpl headerImpl = new HeaderImpl();
        this.addFrom(headerImpl, headersMessage.getFrom());
        this.addAddressList(headerImpl, headersMessage.getTo());
        this.addAddressList(headerImpl, headersMessage.getCc());
        this.addAddressList(headerImpl, headersMessage.getBcc());
        headerImpl.addField((Field)Fields.subject((String)headersMessage.getSubject()));
        return headerImpl;
    }

    private void addFrom(HeaderImpl headerImpl, MailboxList from) {
        if (from != null) {
            headerImpl.addField((Field)Fields.from((Iterable)Lists.newArrayList((Iterator)from.iterator())));
        }
    }

    private void addAddressList(HeaderImpl headerImpl, AddressList addressList) {
        if (addressList != null) {
            headerImpl.addField((Field)Fields.to((Iterable)Lists.newArrayList((Iterator)addressList.iterator())));
        }
    }

    private boolean matches(SearchQuery.ConjunctionCriterion criterion, MailboxMessage message, Collection<MessageUid> recentMessageUids) throws MailboxException {
        List criteria = criterion.getCriteria();
        switch (criterion.getType()) {
            case NOR: {
                return this.nor(criteria, message, recentMessageUids);
            }
            case OR: {
                return this.or(criteria, message, recentMessageUids);
            }
            case AND: {
                return this.and(criteria, message, recentMessageUids);
            }
        }
        return false;
    }

    private boolean and(List<SearchQuery.Criterion> criteria, MailboxMessage message, Collection<MessageUid> recentMessageUids) throws MailboxException {
        for (SearchQuery.Criterion criterion : criteria) {
            boolean matches = this.isMatch(criterion, message, recentMessageUids);
            if (matches) continue;
            return false;
        }
        return true;
    }

    private boolean or(List<SearchQuery.Criterion> criteria, MailboxMessage message, Collection<MessageUid> recentMessageUids) throws MailboxException {
        for (SearchQuery.Criterion criterion : criteria) {
            boolean matches = this.isMatch(criterion, message, recentMessageUids);
            if (!matches) continue;
            return true;
        }
        return false;
    }

    private boolean nor(List<SearchQuery.Criterion> criteria, MailboxMessage message, Collection<MessageUid> recentMessageUids) throws MailboxException {
        for (SearchQuery.Criterion criterion : criteria) {
            boolean matches = this.isMatch(criterion, message, recentMessageUids);
            if (!matches) continue;
            return false;
        }
        return true;
    }

    private boolean matches(SearchQuery.FlagCriterion criterion, MailboxMessage message, Collection<MessageUid> recentMessageUids) {
        SearchQuery.BooleanOperator operator = criterion.getOperator();
        boolean isSet = operator.isSet();
        Flags.Flag flag = criterion.getFlag();
        if (flag == Flags.Flag.ANSWERED) {
            return isSet == message.isAnswered();
        }
        if (flag == Flags.Flag.SEEN) {
            return isSet == message.isSeen();
        }
        if (flag == Flags.Flag.DRAFT) {
            return isSet == message.isDraft();
        }
        if (flag == Flags.Flag.FLAGGED) {
            return isSet == message.isFlagged();
        }
        if (flag == Flags.Flag.RECENT) {
            MessageUid uid = message.getUid();
            return isSet == recentMessageUids.contains(uid);
        }
        if (flag == Flags.Flag.DELETED) {
            return isSet == message.isDeleted();
        }
        return false;
    }

    private boolean matches(SearchQuery.CustomFlagCriterion criterion, MailboxMessage message) {
        SearchQuery.BooleanOperator operator = criterion.getOperator();
        boolean isSet = operator.isSet();
        String flag = criterion.getFlag();
        return isSet == message.createFlags().contains(flag);
    }

    private boolean matches(SearchQuery.UidCriterion criterion, MailboxMessage message) {
        SearchQuery.UidInOperator operator = criterion.getOperator();
        SearchQuery.UidRange[] ranges = operator.getRange();
        MessageUid uid = message.getUid();
        return Arrays.stream(ranges).anyMatch(numericRange -> numericRange.isIn(uid));
    }

    private boolean matches(SearchQuery.HeaderCriterion criterion, MailboxMessage message) throws MailboxException, IOException {
        SearchQuery.HeaderOperator operator = criterion.getOperator();
        String headerName = criterion.getHeaderName();
        if (operator instanceof SearchQuery.DateOperator) {
            return this.matches((SearchQuery.DateOperator)operator, headerName, message);
        }
        if (operator instanceof SearchQuery.ContainsOperator) {
            return this.matches((SearchQuery.ContainsOperator)operator, headerName, message);
        }
        if (operator instanceof SearchQuery.ExistsOperator) {
            return this.exists(headerName, message);
        }
        if (operator instanceof SearchQuery.AddressOperator) {
            return this.matchesAddress((SearchQuery.AddressOperator)operator, headerName, message);
        }
        throw new UnsupportedSearchException();
    }

    private boolean matchesAddress(SearchQuery.AddressOperator operator, String headerName, MailboxMessage message) throws MailboxException, IOException {
        String text = operator.getAddress();
        List<Header> headers = ResultUtils.createHeaders(message);
        for (Header header : headers) {
            String name = header.getName();
            if (!headerName.equalsIgnoreCase(name)) continue;
            String value = header.getValue();
            AddressList addressList = LenientAddressParser.DEFAULT.parseAddressList((CharSequence)value);
            if (this.matchesAddress(addressList, text)) {
                return true;
            }
            return value.toUpperCase(Locale.US).contains(text.toUpperCase(Locale.US));
        }
        return false;
    }

    private boolean matchesAddress(AddressList addressList, String valueToMatch) {
        for (Address address : addressList) {
            if (address instanceof Mailbox) {
                if (!this.doesMailboxContains((Mailbox)address, valueToMatch)) continue;
                return true;
            }
            if (!(address instanceof Group)) continue;
            MailboxList mList = ((Group)address).getMailboxes();
            for (Mailbox mailbox : mList) {
                if (!this.doesMailboxContains(mailbox, valueToMatch)) continue;
                return true;
            }
        }
        return false;
    }

    private boolean doesMailboxContains(Mailbox mailbox, String searchedText) {
        String mailboxAsString = this.encodeAndUnscramble(mailbox);
        return mailboxAsString.toUpperCase(Locale.US).contains(searchedText.toUpperCase(Locale.US));
    }

    private String encodeAndUnscramble(Mailbox mailbox) {
        return MimeUtil.unscrambleHeaderValue((String)AddressFormatter.DEFAULT.encode(mailbox));
    }

    private boolean exists(String headerName, MailboxMessage message) throws MailboxException, IOException {
        List<Header> headers = ResultUtils.createHeaders(message);
        return headers.stream().map(Header::getName).anyMatch(headerName::equalsIgnoreCase);
    }

    private boolean matches(SearchQuery.ContainsOperator operator, String headerName, MailboxMessage message) throws MailboxException, IOException {
        String text = operator.getValue().toUpperCase(Locale.US);
        List<Header> headers = ResultUtils.createHeaders(message);
        for (Header header : headers) {
            String value;
            String name = header.getName();
            if (!headerName.equalsIgnoreCase(name) || (value = MimeUtil.unscrambleHeaderValue((String)header.getValue())) == null || !value.toUpperCase(Locale.US).contains(text)) continue;
            return true;
        }
        return false;
    }

    private boolean matches(SearchQuery.DateOperator operator, String headerName, MailboxMessage message) throws MailboxException {
        Date date = operator.getDate();
        SearchQuery.DateResolution res = operator.getDateResultion();
        try {
            String value = this.headerValue(headerName, message);
            if (value == null) {
                return false;
            }
            try {
                Date isoFieldValue = this.toISODate(value);
                SearchQuery.DateComparator type = operator.getType();
                switch (type) {
                    case AFTER: {
                        return this.after(isoFieldValue, date, res);
                    }
                    case BEFORE: {
                        return this.before(isoFieldValue, date, res);
                    }
                    case ON: {
                        return this.on(isoFieldValue, date, res);
                    }
                }
                throw new UnsupportedSearchException();
            }
            catch (ParseException e) {
                return false;
            }
        }
        catch (IOException e) {
            return false;
        }
    }

    private String headerValue(String headerName, MailboxMessage message) throws MailboxException, IOException {
        List<Header> headers = ResultUtils.createHeaders(message);
        for (Header header : headers) {
            String name = header.getName();
            if (!headerName.equalsIgnoreCase(name)) continue;
            return MimeUtil.unscrambleHeaderValue((String)header.getValue());
        }
        return null;
    }

    private Date toISODate(String value) throws ParseException {
        StringReader reader = new StringReader(value);
        DateTime dateTime = new DateTimeParser((Reader)reader).parseAll();
        Calendar cal = this.getGMT();
        cal.set(dateTime.getYear(), dateTime.getMonth() - 1, dateTime.getDay(), dateTime.getHour(), dateTime.getMinute(), dateTime.getSecond());
        return cal.getTime();
    }

    private boolean matches(SearchQuery.AttachmentCriterion criterion, MailboxMessage message) throws UnsupportedSearchException {
        boolean mailHasAttachments = message.getProperties().stream().anyMatch(PropertyBuilder.isHasAttachmentProperty());
        return mailHasAttachments == criterion.getOperator().isSet();
    }

    private boolean matches(SearchQuery.SizeCriterion criterion, MailboxMessage message) throws UnsupportedSearchException {
        SearchQuery.NumericOperator operator = criterion.getOperator();
        long size = message.getFullContentOctets();
        long value = operator.getValue();
        switch (operator.getType()) {
            case LESS_THAN: {
                return size < value;
            }
            case GREATER_THAN: {
                return size > value;
            }
            case EQUALS: {
                return size == value;
            }
        }
        throw new UnsupportedSearchException();
    }

    private boolean matches(SearchQuery.ModSeqCriterion criterion, MailboxMessage message) throws UnsupportedSearchException {
        SearchQuery.NumericOperator operator = criterion.getOperator();
        ModSeq modSeq = message.getModSeq();
        long value = operator.getValue();
        switch (operator.getType()) {
            case LESS_THAN: {
                return modSeq.asLong() < value;
            }
            case GREATER_THAN: {
                return modSeq.asLong() > value;
            }
            case EQUALS: {
                return modSeq.asLong() == value;
            }
        }
        throw new UnsupportedSearchException();
    }

    private boolean matches(SearchQuery.InternalDateCriterion criterion, MailboxMessage message) throws UnsupportedSearchException {
        SearchQuery.DateOperator operator = criterion.getOperator();
        return this.matchesInternalDate(operator, message);
    }

    private boolean matchesInternalDate(SearchQuery.DateOperator operator, MailboxMessage message) throws UnsupportedSearchException {
        Date date = operator.getDate();
        SearchQuery.DateResolution dateResultion = operator.getDateResultion();
        Date internalDate = message.getInternalDate();
        SearchQuery.DateComparator type = operator.getType();
        switch (type) {
            case ON: {
                return this.on(internalDate, date, dateResultion);
            }
            case BEFORE: {
                return this.before(internalDate, date, dateResultion);
            }
            case AFTER: {
                return this.after(internalDate, date, dateResultion);
            }
        }
        throw new UnsupportedSearchException();
    }

    private boolean on(Date date1, Date date2, SearchQuery.DateResolution dateResolution) {
        String d2;
        String d1 = this.createDateString(date1, dateResolution);
        return d1.compareTo(d2 = this.createDateString(date2, dateResolution)) == 0;
    }

    private boolean before(Date date1, Date date2, SearchQuery.DateResolution dateResolution) {
        String d2;
        String d1 = this.createDateString(date1, dateResolution);
        return d1.compareTo(d2 = this.createDateString(date2, dateResolution)) < 0;
    }

    private boolean after(Date date1, Date date2, SearchQuery.DateResolution dateResolution) {
        String d2;
        String d1 = this.createDateString(date1, dateResolution);
        return d1.compareTo(d2 = this.createDateString(date2, dateResolution)) > 0;
    }

    private String createDateString(Date date, SearchQuery.DateResolution dateResolution) {
        SimpleDateFormat format = this.createFormat(dateResolution);
        format.setCalendar(this.getGMT());
        return format.format(date);
    }

    private SimpleDateFormat createFormat(SearchQuery.DateResolution dateResolution) {
        switch (dateResolution) {
            case Year: {
                return new SimpleDateFormat("yyyy");
            }
            case Month: {
                return new SimpleDateFormat("yyyyMM");
            }
            case Day: {
                return new SimpleDateFormat("yyyyMMdd");
            }
            case Hour: {
                return new SimpleDateFormat("yyyyMMddhh");
            }
            case Minute: {
                return new SimpleDateFormat("yyyyMMddhhmm");
            }
            case Second: {
                return new SimpleDateFormat("yyyyMMddhhmmss");
            }
        }
        return new SimpleDateFormat("yyyyMMddhhmmssSSS");
    }

    private Calendar getGMT() {
        return Calendar.getInstance(TimeZone.getTimeZone("GMT"), Locale.ENGLISH);
    }
}

