/*
 * Decompiled with CFR 0.152.
 */
package org.apache.james.queue.file;

import com.github.fge.lambdas.Throwing;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.OutputStream;
import java.security.SecureRandom;
import java.time.Duration;
import java.time.Instant;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.time.temporal.ChronoUnit;
import java.util.Collections;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Optional;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.Executors;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import javax.mail.MessagingException;
import javax.mail.internet.MimeMessage;
import javax.mail.util.SharedFileInputStream;
import org.apache.commons.io.FileUtils;
import org.apache.james.lifecycle.api.Disposable;
import org.apache.james.lifecycle.api.LifecycleUtil;
import org.apache.james.queue.api.MailQueue;
import org.apache.james.queue.api.MailQueueItemDecoratorFactory;
import org.apache.james.queue.api.ManageableMailQueue;
import org.apache.james.server.core.MimeMessageCopyOnWriteProxy;
import org.apache.james.server.core.MimeMessageSource;
import org.apache.james.util.concurrent.NamedThreadFactory;
import org.apache.mailet.Attribute;
import org.apache.mailet.AttributeName;
import org.apache.mailet.AttributeUtils;
import org.apache.mailet.AttributeValue;
import org.apache.mailet.Mail;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;

public class FileMailQueue
implements ManageableMailQueue {
    private static final Logger LOGGER = LoggerFactory.getLogger(FileMailQueue.class);
    private final Map<String, FileItem> keyMappings = Collections.synchronizedMap(new LinkedHashMap());
    private final BlockingQueue<String> inmemoryQueue = new LinkedBlockingQueue<String>();
    private final ScheduledExecutorService scheduler = Executors.newSingleThreadScheduledExecutor((ThreadFactory)NamedThreadFactory.withClassName(this.getClass()));
    private static final AtomicLong COUNTER = new AtomicLong();
    private final String queueDirName;
    private final File queueDir;
    private final MailQueueItemDecoratorFactory mailQueueItemDecoratorFactory;
    private final boolean sync;
    private static final String MSG_EXTENSION = ".msg";
    private static final String OBJECT_EXTENSION = ".obj";
    private static final AttributeName NEXT_DELIVERY = AttributeName.of((String)"FileQueueNextDelivery");
    private static final int SPLITCOUNT = 10;
    private static final SecureRandom RANDOM = new SecureRandom();
    private final String queueName;
    private final Flux<MailQueue.MailQueueItem> flux;

    public FileMailQueue(MailQueueItemDecoratorFactory mailQueueItemDecoratorFactory, File parentDir, String queuename, boolean sync) throws IOException {
        this.mailQueueItemDecoratorFactory = mailQueueItemDecoratorFactory;
        this.sync = sync;
        this.queueName = queuename;
        this.queueDir = new File(parentDir, this.queueName);
        this.queueDirName = this.queueDir.getAbsolutePath();
        this.init();
        this.flux = Mono.defer(this::deQueueOneItem).repeat().limitRate(1);
    }

    public String getName() {
        return this.queueName;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void init() throws IOException {
        for (int i = 1; i <= 10; ++i) {
            String[] files;
            File qDir = new File(this.queueDir, Integer.toString(i));
            FileUtils.forceMkdir((File)qDir);
            for (String name2 : files = qDir.list((dir, name) -> name.endsWith(OBJECT_EXTENSION))) {
                ObjectInputStream oin = null;
                try {
                    String msgFileName = name2.substring(0, name2.length() - OBJECT_EXTENSION.length()) + MSG_EXTENSION;
                    FileItem item = new FileItem(qDir.getAbsolutePath() + File.separator + name2, qDir.getAbsolutePath() + File.separator + msgFileName);
                    oin = new ObjectInputStream(new FileInputStream(item.getObjectFile()));
                    Mail mail = (Mail)oin.readObject();
                    Optional<ZonedDateTime> next = this.getNextDelivery(mail);
                    String key = mail.getName();
                    this.keyMappings.put(key, item);
                    if (!next.isPresent() || next.get().isBefore(ZonedDateTime.now())) {
                        try {
                            this.inmemoryQueue.put(key);
                            continue;
                        }
                        catch (InterruptedException e) {
                            Thread.currentThread().interrupt();
                            throw new RuntimeException("Unable to init", e);
                        }
                    }
                    long nextDeliveryDelay = ZonedDateTime.now().until(next.get(), ChronoUnit.MILLIS);
                    this.scheduler.schedule(() -> {
                        try {
                            this.inmemoryQueue.put(key);
                        }
                        catch (InterruptedException e) {
                            Thread.currentThread().interrupt();
                            throw new RuntimeException("Unable to init", e);
                        }
                    }, nextDeliveryDelay, TimeUnit.MILLISECONDS);
                }
                catch (IOException | ClassNotFoundException e) {
                    LOGGER.error("Unable to load Mail", (Throwable)e);
                }
                finally {
                    if (oin != null) {
                        try {
                            oin.close();
                        }
                        catch (Exception exception) {}
                    }
                }
            }
        }
    }

    private Optional<ZonedDateTime> getNextDelivery(Mail mail) {
        return AttributeUtils.getValueAndCastFromMail((Mail)mail, (AttributeName)NEXT_DELIVERY, Long.class).map(next -> Instant.ofEpochMilli(next).atZone(ZoneId.systemDefault()));
    }

    public void enQueue(Mail mail, Duration delay) throws MailQueue.MailQueueException {
        String key = mail.getName() + "-" + COUNTER.incrementAndGet();
        try {
            int i = RANDOM.nextInt(10) + 1;
            String name = this.queueDirName + "/" + i + "/" + key;
            FileItem item = new FileItem(name + OBJECT_EXTENSION, name + MSG_EXTENSION);
            if (!delay.isNegative()) {
                mail.setAttribute(new Attribute(NEXT_DELIVERY, AttributeValue.of((Long)this.computeNextDelivery(delay))));
            }
            try (FileOutputStream foout = new FileOutputStream(item.getObjectFile());
                 ObjectOutputStream oout = new ObjectOutputStream(foout);){
                oout.writeObject(mail);
                oout.flush();
                if (this.sync) {
                    foout.getFD().sync();
                }
            }
            try (FileOutputStream out = new FileOutputStream(item.getMessageFile());){
                mail.getMessage().writeTo((OutputStream)out);
                out.flush();
                if (this.sync) {
                    out.getFD().sync();
                }
            }
            this.keyMappings.put(key, item);
            if (!delay.isNegative()) {
                this.scheduler.schedule(() -> {
                    try {
                        this.inmemoryQueue.put(key);
                    }
                    catch (InterruptedException e) {
                        Thread.currentThread().interrupt();
                        throw new RuntimeException("Unable to init", e);
                    }
                }, delay.getSeconds(), TimeUnit.SECONDS);
            } else {
                this.inmemoryQueue.put(key);
            }
        }
        catch (IOException | InterruptedException | MessagingException e) {
            throw new MailQueue.MailQueueException("Unable to enqueue mail", (Exception)e);
        }
    }

    private long computeNextDelivery(Duration delay) {
        try {
            return Instant.now().plus(delay).getEpochSecond();
        }
        catch (ArithmeticException e) {
            return Long.MAX_VALUE;
        }
    }

    public void enQueue(Mail mail) throws MailQueue.MailQueueException {
        this.enQueue(mail, 0L, TimeUnit.MILLISECONDS);
    }

    public Flux<MailQueue.MailQueueItem> deQueue() {
        return this.flux;
    }

    private Mono<MailQueue.MailQueueItem> deQueueOneItem() {
        Mono mono;
        FileItem item = null;
        String k = null;
        while (item == null) {
            k = this.inmemoryQueue.take();
            item = this.keyMappings.get(k);
        }
        final String key = k;
        final FileItem fitem = item;
        File objectFile = new File(fitem.getObjectFile());
        File msgFile = new File(fitem.getMessageFile());
        ObjectInputStream oin = new ObjectInputStream(new FileInputStream(objectFile));
        try {
            final Mail mail = (Mail)oin.readObject();
            mail.setMessage((MimeMessage)new MimeMessageCopyOnWriteProxy((MimeMessageSource)new FileMimeMessageSource(msgFile)));
            MailQueue.MailQueueItem fileMailQueueItem = new MailQueue.MailQueueItem(){

                public Mail getMail() {
                    return mail;
                }

                public void done(boolean success) throws MailQueue.MailQueueException {
                    if (!success) {
                        try {
                            FileMailQueue.this.inmemoryQueue.put(key);
                        }
                        catch (InterruptedException e) {
                            Thread.currentThread().interrupt();
                            throw new MailQueue.MailQueueException("Unable to rollback", (Exception)e);
                        }
                    } else {
                        fitem.delete();
                        FileMailQueue.this.keyMappings.remove(key);
                    }
                    LifecycleUtil.dispose((Object)mail);
                }
            };
            mono = Mono.just((Object)this.mailQueueItemDecoratorFactory.decorate(fileMailQueueItem));
        }
        catch (Throwable throwable) {
            try {
                try {
                    try {
                        oin.close();
                    }
                    catch (Throwable throwable2) {
                        throwable.addSuppressed(throwable2);
                    }
                    throw throwable;
                }
                catch (IOException | ClassNotFoundException | MessagingException e) {
                    return Mono.error((Throwable)new MailQueue.MailQueueException("Unable to dequeue", (Exception)e));
                }
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                return Mono.error((Throwable)new MailQueue.MailQueueException("Unable to dequeue", (Exception)e));
            }
        }
        oin.close();
        return mono;
    }

    public long getSize() throws MailQueue.MailQueueException {
        return this.keyMappings.size();
    }

    public long flush() throws MailQueue.MailQueueException {
        Iterator<String> keys = this.keyMappings.keySet().iterator();
        long i = 0L;
        while (keys.hasNext()) {
            String key = keys.next();
            if (this.inmemoryQueue.contains(key)) continue;
            this.inmemoryQueue.add(key);
            ++i;
        }
        return i;
    }

    public long clear() throws MailQueue.MailQueueException {
        long count = this.getSize();
        this.keyMappings.values().forEach(Throwing.consumer(FileItem::delete));
        this.keyMappings.clear();
        this.inmemoryQueue.clear();
        return count;
    }

    public long remove(ManageableMailQueue.Type type, String value) throws MailQueue.MailQueueException {
        switch (type) {
            case Name: {
                FileItem item = this.keyMappings.remove(value);
                if (item != null) {
                    item.delete();
                    return 1L;
                }
                return 0L;
            }
        }
        throw new MailQueue.MailQueueException("Not supported yet");
    }

    public ManageableMailQueue.MailQueueIterator browse() throws MailQueue.MailQueueException {
        final Iterator<FileItem> items = this.keyMappings.values().iterator();
        return new ManageableMailQueue.MailQueueIterator(){
            private ManageableMailQueue.MailQueueItemView item;

            public void remove() {
                throw new UnsupportedOperationException("Read-only");
            }

            public ManageableMailQueue.MailQueueItemView next() {
                if (this.hasNext()) {
                    ManageableMailQueue.MailQueueItemView itemView = this.item;
                    this.item = null;
                    return itemView;
                }
                throw new NoSuchElementException();
            }

            public boolean hasNext() {
                if (this.item != null) {
                    return true;
                }
                while (items.hasNext()) {
                    boolean bl;
                    ObjectInputStream in = new ObjectInputStream(new FileInputStream(((FileItem)items.next()).getObjectFile()));
                    try {
                        Mail mail = (Mail)in.readObject();
                        this.item = new ManageableMailQueue.MailQueueItemView(mail, FileMailQueue.this.getNextDelivery(mail));
                        bl = true;
                    }
                    catch (Throwable throwable) {
                        try {
                            try {
                                in.close();
                            }
                            catch (Throwable throwable2) {
                                throwable.addSuppressed(throwable2);
                            }
                            throw throwable;
                        }
                        catch (IOException | ClassNotFoundException e) {
                            LOGGER.info("Unable to load mail", (Throwable)e);
                        }
                    }
                    in.close();
                    return bl;
                }
                return false;
            }

            public void close() {
            }
        };
    }

    private final class FileItem {
        private final String objectfile;
        private final String messagefile;

        public FileItem(String objectfile, String messagefile) {
            this.objectfile = objectfile;
            this.messagefile = messagefile;
        }

        public String getObjectFile() {
            return this.objectfile;
        }

        public String getMessageFile() {
            return this.messagefile;
        }

        public void delete() throws MailQueue.MailQueueException {
            try {
                FileUtils.forceDelete((File)new File(this.getObjectFile()));
            }
            catch (IOException e) {
                throw new MailQueue.MailQueueException("Unable to delete mail");
            }
            try {
                FileUtils.forceDelete((File)new File(this.getMessageFile()));
            }
            catch (IOException e) {
                LOGGER.debug("Remove of msg file for mail failed");
            }
        }
    }

    private final class FileMimeMessageSource
    extends MimeMessageSource
    implements Disposable {
        private File file;
        private final SharedFileInputStream in;

        public FileMimeMessageSource(File file) throws IOException {
            this.file = file;
            this.in = new SharedFileInputStream(file);
        }

        public String getSourceId() {
            return this.file.getAbsolutePath();
        }

        public InputStream getInputStream() throws IOException {
            return this.in.newStream(0L, -1L);
        }

        public long getMessageSize() throws IOException {
            return this.file.length();
        }

        public void dispose() {
            try {
                this.in.close();
            }
            catch (IOException iOException) {
                // empty catch block
            }
            this.file = null;
        }
    }
}

