/*
 * Decompiled with CFR 0.152.
 */
package org.apache.james.task;

import com.google.common.collect.Sets;
import java.io.IOException;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.Callable;
import java.util.concurrent.CancellationException;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.Semaphore;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Function;
import org.apache.james.task.Task;
import org.apache.james.task.TaskId;
import org.apache.james.task.TaskManagerWorker;
import org.apache.james.task.TaskWithId;
import org.apache.james.util.MDCBuilder;
import org.apache.james.util.concurrent.NamedThreadFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import reactor.core.publisher.Mono;
import reactor.util.function.Tuple2;
import reactor.util.function.Tuples;

public class SerialTaskManagerWorker
implements TaskManagerWorker {
    private static final Logger LOGGER = LoggerFactory.getLogger(SerialTaskManagerWorker.class);
    private final ExecutorService taskExecutor = Executors.newSingleThreadExecutor((ThreadFactory)NamedThreadFactory.withName((String)"task executor"));
    private final TaskManagerWorker.Listener listener;
    private final AtomicReference<Tuple2<TaskId, Future<?>>> runningTask;
    private final Semaphore semaphore;
    private final Set<TaskId> cancelledTasks;

    public SerialTaskManagerWorker(TaskManagerWorker.Listener listener) {
        this.listener = listener;
        this.cancelledTasks = Sets.newConcurrentHashSet();
        this.runningTask = new AtomicReference();
        this.semaphore = new Semaphore(1);
    }

    @Override
    public Mono<Task.Result> executeTask(TaskWithId taskWithId) {
        return Mono.using(this.acquireSemaphore(taskWithId, this.listener), this.executeWithSemaphore(taskWithId, this.listener), Semaphore::release);
    }

    private Callable<Semaphore> acquireSemaphore(TaskWithId taskWithId, TaskManagerWorker.Listener listener) {
        return () -> {
            try {
                this.semaphore.acquire();
                return this.semaphore;
            }
            catch (InterruptedException e) {
                listener.cancelled(taskWithId.getId());
                throw e;
            }
        };
    }

    private Function<Semaphore, Mono<Task.Result>> executeWithSemaphore(TaskWithId taskWithId, TaskManagerWorker.Listener listener) {
        return semaphore -> {
            if (!this.cancelledTasks.remove(taskWithId.getId())) {
                CompletableFuture<Task.Result> future = CompletableFuture.supplyAsync(() -> this.runWithMdc(taskWithId, listener), this.taskExecutor);
                this.runningTask.set(Tuples.of((Object)taskWithId.getId(), future));
                return Mono.fromFuture(future).doOnError(exception -> {
                    if (exception instanceof CancellationException) {
                        listener.cancelled(taskWithId.getId());
                    } else {
                        listener.failed(taskWithId.getId(), (Throwable)exception);
                    }
                }).onErrorReturn((Object)Task.Result.PARTIAL);
            }
            listener.cancelled(taskWithId.getId());
            return Mono.empty();
        };
    }

    private Task.Result runWithMdc(TaskWithId taskWithId, TaskManagerWorker.Listener listener) {
        return (Task.Result)((Object)MDCBuilder.withMdc((MDCBuilder)MDCBuilder.create().addContext("taskId", (Object)taskWithId.getId()).addContext("taskType", (Object)taskWithId.getTask().type()).addContext("taskDetails", taskWithId.getTask().details()), () -> this.run(taskWithId, listener)));
    }

    private Task.Result run(TaskWithId taskWithId, TaskManagerWorker.Listener listener) {
        listener.started(taskWithId.getId());
        try {
            return taskWithId.getTask().run().onComplete(result -> listener.completed(taskWithId.getId(), result)).onFailure(() -> {
                LOGGER.error("Task was partially performed. Check logs for more details. Taskid : " + taskWithId.getId());
                listener.failed(taskWithId.getId());
            });
        }
        catch (InterruptedException e) {
            listener.cancelled(taskWithId.getId());
            return Task.Result.PARTIAL;
        }
        catch (Exception e) {
            LOGGER.error("Error while running task {}", (Object)taskWithId.getId(), (Object)e);
            listener.failed(taskWithId.getId(), e);
            return Task.Result.PARTIAL;
        }
    }

    @Override
    public void cancelTask(TaskId taskId) {
        this.cancelledTasks.add(taskId);
        Optional.ofNullable(this.runningTask.get()).filter(task -> ((TaskId)task.getT1()).equals(taskId)).ifPresent(task -> ((Future)task.getT2()).cancel(true));
    }

    @Override
    public void fail(TaskId taskId, Throwable reason) {
        this.listener.failed(taskId, reason);
    }

    @Override
    public void close() throws IOException {
        this.taskExecutor.shutdownNow();
    }
}

