/****************************************************************
 * Licensed to the Apache Software Foundation (ASF) under one   *
 * or more contributor license agreements.  See the NOTICE file *
 * distributed with this work for additional information        *
 * regarding copyright ownership.  The ASF licenses this file   *
 * to you under the Apache License, Version 2.0 (the            *
 * "License"); you may not use this file except in compliance   *
 * with the License.  You may obtain a copy of the License at   *
 *                                                              *
 *   http://www.apache.org/licenses/LICENSE-2.0                 *
 *                                                              *
 * Unless required by applicable law or agreed to in writing,   *
 * software distributed under the License is distributed on an  *
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY       *
 * KIND, either express or implied.  See the License for the    *
 * specific language governing permissions and limitations      *
 * under the License.                                           *
 ****************************************************************/

package org.apache.james.modules.server;

import static org.assertj.core.api.Assertions.assertThat;

import java.io.StringReader;
import java.util.Optional;

import org.apache.commons.configuration.PropertiesConfiguration;
import org.apache.james.util.Host;
import org.junit.jupiter.api.Test;

import nl.jqno.equalsverifier.EqualsVerifier;

public class JmxConfigurationTest {
    @Test
    void shouldMatchBeanContract() {
        EqualsVerifier.forClass(JmxConfiguration.class)
            .verify();
    }

    @Test
    void fromPropertiesShouldReturnDefaultWhenEmpty() {
        assertThat(JmxConfiguration.fromProperties(new PropertiesConfiguration()))
            .isEqualTo(JmxConfiguration.DEFAULT_CONFIGURATION);
    }

    @Test
    void fromPropertiesShouldReturnConfiguredValues() throws Exception {
        PropertiesConfiguration configuration = new PropertiesConfiguration();
        configuration.load(new StringReader(
                "jmx.address=172.0.0.5\n" +
                "jmx.port=889\n"));

        assertThat(JmxConfiguration.fromProperties(configuration))
            .isEqualTo(new JmxConfiguration(JmxConfiguration.ENABLED, Optional.of(Host.from("172.0.0.5", 889))));
    }

    @Test
    void fromPropertiesShouldReturnDisabledWhenConfiguredAsDisabled() throws Exception {
        PropertiesConfiguration configuration = new PropertiesConfiguration();
        configuration.load(new StringReader(
                    "jmx.enabled=false\n"));

        assertThat(JmxConfiguration.fromProperties(configuration))
            .isEqualTo(JmxConfiguration.DISABLED);
    }

    @Test
    void fromPropertiesShouldReturnDisabledWhenConfiguredAsDisabledWithHost() throws Exception {
        PropertiesConfiguration configuration = new PropertiesConfiguration();
        configuration.load(new StringReader(
                    "jmx.enabled=false\n" +
                "jmx.address=172.0.0.5\n" +
                "jmx.port=889\n"));

        assertThat(JmxConfiguration.fromProperties(configuration))
            .isEqualTo(JmxConfiguration.DISABLED);
    }
}
