/****************************************************************
 * Licensed to the Apache Software Foundation (ASF) under one   *
 * or more contributor license agreements.  See the NOTICE file *
 * distributed with this work for additional information        *
 * regarding copyright ownership.  The ASF licenses this file   *
 * to you under the Apache License, Version 2.0 (the            *
 * "License"); you may not use this file except in compliance   *
 * with the License.  You may obtain a copy of the License at   *
 *                                                              *
 *   http://www.apache.org/licenses/LICENSE-2.0                 *
 *                                                              *
 * Unless required by applicable law or agreed to in writing,   *
 * software distributed under the License is distributed on an  *
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY       *
 * KIND, either express or implied.  See the License for the    *
 * specific language governing permissions and limitations      *
 * under the License.                                           *
 ****************************************************************/

package org.apache.james.webadmin.routes;

import java.util.List;
import java.util.Set;

import javax.inject.Inject;
import javax.ws.rs.GET;
import javax.ws.rs.Path;

import org.apache.james.core.healthcheck.HealthCheck;
import org.apache.james.core.healthcheck.Result;
import org.apache.james.webadmin.PublicRoutes;
import org.apache.james.webadmin.dto.HealthCheckExecutionResultDto;
import org.apache.james.webadmin.utils.ErrorResponder;
import org.apache.james.webadmin.utils.JsonTransformer;
import org.eclipse.jetty.http.HttpStatus;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.github.steveash.guavate.Guavate;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiResponse;
import io.swagger.annotations.ApiResponses;
import spark.HaltException;
import spark.Request;
import spark.Response;
import spark.Service;

@Api(tags = "Healthchecks")
@Path(HealthCheckRoutes.HEALTHCHECK)
public class HealthCheckRoutes implements PublicRoutes {

    private static final Logger LOGGER = LoggerFactory.getLogger(HealthCheckRoutes.class);

    public static final String HEALTHCHECK = "/healthcheck";
    
    private static final String PARAM_COMPONENT_NAME = "componentName";

    private final JsonTransformer jsonTransformer;
    private final Set<HealthCheck> healthChecks;

    @Inject
    public HealthCheckRoutes(Set<HealthCheck> healthChecks, JsonTransformer jsonTransformer) {
        this.healthChecks = healthChecks;
        this.jsonTransformer = jsonTransformer;
    }

    @Override
    public String getBasePath() {
        return HEALTHCHECK;
    }

    @Override
    public void define(Service service) {
        service.get(HEALTHCHECK, this::validateHealthchecks, jsonTransformer);
        service.get(HEALTHCHECK + "/checks/:" + PARAM_COMPONENT_NAME, this::performHealthCheckForComponent, jsonTransformer);
    }

    @GET
    @ApiOperation(value = "Validate all health checks")
    @ApiResponses(value = {
        @ApiResponse(code = HttpStatus.OK_200, message = "OK"),
        @ApiResponse(code = HttpStatus.INTERNAL_SERVER_ERROR_500,
            message = "Internal server error - When one check has failed.")
    })
    public Object validateHealthchecks(Request request, Response response) {
        List<Result> anyUnhealthyOrDegraded = retrieveUnhealthyOrDegradedHealthChecks();

        anyUnhealthyOrDegraded.forEach(this::logFailedCheck);
        response.status(getCorrespondingStatusCode(anyUnhealthyOrDegraded));
        return response;
    }
    
    @GET
    @Path("/checks/{" + PARAM_COMPONENT_NAME + "}")
    @ApiOperation(value = "Perform the component's health check")
    @ApiImplicitParams({
        @ApiImplicitParam(
            name = PARAM_COMPONENT_NAME,
            required = true,
            paramType = "path",
            dataType = "String",
            defaultValue = "None",
            example = "/checks/Cassandra%20Backend",
            value = "The URL encoded name of the component to check.")
    })
    public Object performHealthCheckForComponent(Request request, Response response) {
        String componentName = request.params(PARAM_COMPONENT_NAME);
        HealthCheck healthCheck = healthChecks.stream()
            .filter(c -> c.componentName().getName().equals(componentName))
            .findFirst()
            .orElseThrow(() -> throw404(componentName));
        
        Result result = healthCheck.check();
        logFailedCheck(result);
        response.status(getCorrespondingStatusCode(result));
        return new HealthCheckExecutionResultDto(result);
    }

    private int getCorrespondingStatusCode(List<Result> anyUnhealthy) {
        if (anyUnhealthy.isEmpty()) {
            return HttpStatus.OK_200;
        } else {
            return HttpStatus.INTERNAL_SERVER_ERROR_500;
        }
    }
    
    private int getCorrespondingStatusCode(Result result) {
        switch (result.getStatus()) {
        case HEALTHY:
            return HttpStatus.OK_200;
        case DEGRADED:
        case UNHEALTHY:
        default:
            return HttpStatus.INTERNAL_SERVER_ERROR_500;
        }
    }

    private void logFailedCheck(Result result) {
        switch (result.getStatus()) {
        case UNHEALTHY:
            LOGGER.error("HealthCheck failed for {} : {}",
                    result.getComponentName().getName(),
                    result.getCause().orElse(""));
            break;
        case DEGRADED:
            LOGGER.warn("HealthCheck is unstable for {} : {}",
                    result.getComponentName().getName(),
                    result.getCause().orElse(""));
            break;
        case HEALTHY:
            // Here only to fix a warning, such cases are already filtered
            break;
        }
    }

    private List<Result> retrieveUnhealthyOrDegradedHealthChecks() {
        return healthChecks.stream()
            .map(HealthCheck::check)
            .filter(result -> result.isUnHealthy() || result.isDegraded())
            .collect(Guavate.toImmutableList());
    }
    
    private HaltException throw404(String componentName) {
        return ErrorResponder.builder()
            .message(String.format("Component with name %s cannot be found", componentName))
            .statusCode(HttpStatus.NOT_FOUND_404)
            .type(ErrorResponder.ErrorType.NOT_FOUND)
            .haltError();
    }
        
}
