/*
 * Decompiled with CFR 0.152.
 */
package org.apache.james.mailbox.lucene.search;

import com.github.steveash.guavate.Guavate;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.io.StringReader;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collection;
import java.util.Date;
import java.util.EnumSet;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Optional;
import java.util.TimeZone;
import javax.annotation.PreDestroy;
import javax.inject.Inject;
import javax.mail.Flags;
import org.apache.james.mailbox.MailboxListener;
import org.apache.james.mailbox.MailboxManager;
import org.apache.james.mailbox.MailboxSession;
import org.apache.james.mailbox.MessageUid;
import org.apache.james.mailbox.exception.MailboxException;
import org.apache.james.mailbox.exception.UnsupportedSearchException;
import org.apache.james.mailbox.lucene.search.LenientImapSearchAnalyzer;
import org.apache.james.mailbox.lucene.search.StrictImapSearchAnalyzer;
import org.apache.james.mailbox.model.MailboxId;
import org.apache.james.mailbox.model.MessageId;
import org.apache.james.mailbox.model.MessageRange;
import org.apache.james.mailbox.model.SearchQuery;
import org.apache.james.mailbox.model.UpdatedFlags;
import org.apache.james.mailbox.store.mail.MessageMapperFactory;
import org.apache.james.mailbox.store.mail.model.MailboxMessage;
import org.apache.james.mailbox.store.mail.model.impl.PropertyBuilder;
import org.apache.james.mailbox.store.search.ListeningMessageSearchIndex;
import org.apache.james.mailbox.store.search.MessageSearchIndex;
import org.apache.james.mailbox.store.search.SearchUtil;
import org.apache.james.mime4j.MimeException;
import org.apache.james.mime4j.dom.Header;
import org.apache.james.mime4j.dom.address.Address;
import org.apache.james.mime4j.dom.address.AddressList;
import org.apache.james.mime4j.dom.address.Group;
import org.apache.james.mime4j.dom.address.Mailbox;
import org.apache.james.mime4j.dom.address.MailboxList;
import org.apache.james.mime4j.dom.datetime.DateTime;
import org.apache.james.mime4j.dom.field.DateTimeField;
import org.apache.james.mime4j.field.address.AddressFormatter;
import org.apache.james.mime4j.field.address.LenientAddressParser;
import org.apache.james.mime4j.field.datetime.parser.DateTimeParser;
import org.apache.james.mime4j.field.datetime.parser.ParseException;
import org.apache.james.mime4j.message.SimpleContentHandler;
import org.apache.james.mime4j.parser.ContentHandler;
import org.apache.james.mime4j.parser.MimeStreamParser;
import org.apache.james.mime4j.stream.BodyDescriptor;
import org.apache.james.mime4j.stream.Field;
import org.apache.james.mime4j.stream.MimeConfig;
import org.apache.james.mime4j.util.MimeUtil;
import org.apache.lucene.analysis.Analyzer;
import org.apache.lucene.document.DateTools;
import org.apache.lucene.document.Document;
import org.apache.lucene.document.Field;
import org.apache.lucene.document.Fieldable;
import org.apache.lucene.document.NumericField;
import org.apache.lucene.index.IndexReader;
import org.apache.lucene.index.IndexWriter;
import org.apache.lucene.index.IndexWriterConfig;
import org.apache.lucene.index.Term;
import org.apache.lucene.search.BooleanClause;
import org.apache.lucene.search.BooleanQuery;
import org.apache.lucene.search.IndexSearcher;
import org.apache.lucene.search.NumericRangeQuery;
import org.apache.lucene.search.PrefixQuery;
import org.apache.lucene.search.Query;
import org.apache.lucene.search.ScoreDoc;
import org.apache.lucene.search.Sort;
import org.apache.lucene.search.SortField;
import org.apache.lucene.search.TermQuery;
import org.apache.lucene.search.TermRangeQuery;
import org.apache.lucene.search.TopDocs;
import org.apache.lucene.search.TopFieldDocs;
import org.apache.lucene.search.WildcardQuery;
import org.apache.lucene.store.Directory;
import org.apache.lucene.util.Version;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class LuceneMessageSearchIndex
extends ListeningMessageSearchIndex {
    private static final Logger LOGGER = LoggerFactory.getLogger(LuceneMessageSearchIndex.class);
    private static final Date MAX_DATE;
    private static final Date MIN_DATE;
    private static final int DEFAULT_MAX_QUERY_RESULTS = 100000;
    private static final String ID_FIELD = "id";
    private static final String UID_FIELD = "uid";
    private static final String HAS_ATTACHMENT_FIELD = "hasAttachment";
    private static final String FLAGS_FIELD = "flags";
    private static final String SIZE_FIELD = "size";
    private static final String BODY_FIELD = "body";
    private static final String PREFIX_HEADER_FIELD = "header_";
    private static final String HEADERS_FIELD = "headers";
    private static final String MODSEQ_FIELD = "modSeq";
    private static final String TO_FIELD = "to";
    private static final String FIRST_TO_MAILBOX_NAME_FIELD = "firstToMailboxName";
    private static final String FIRST_TO_MAILBOX_DISPLAY_FIELD = "firstToMailboxDisplay";
    private static final String CC_FIELD = "cc";
    private static final String FIRST_CC_MAILBOX_NAME_FIELD = "firstCcMailboxName";
    private static final String FROM_FIELD = "from";
    private static final String FIRST_FROM_MAILBOX_NAME_FIELD = "firstFromMailboxName";
    private static final String FIRST_FROM_MAILBOX_DISPLAY_FIELD = "firstFromMailboxDisplay";
    private static final String BCC_FIELD = "bcc";
    private static final String BASE_SUBJECT_FIELD = "baseSubject";
    private static final String INTERNAL_DATE_FIELD_YEAR_RESOLUTION = "internaldateYearResolution";
    private static final String INTERNAL_DATE_FIELD_MONTH_RESOLUTION = "internaldateMonthResolution";
    private static final String INTERNAL_DATE_FIELD_DAY_RESOLUTION = "internaldateDayResolution";
    private static final String INTERNAL_DATE_FIELD_HOUR_RESOLUTION = "internaldateHourResolution";
    private static final String INTERNAL_DATE_FIELD_MINUTE_RESOLUTION = "internaldateMinuteResolution";
    private static final String INTERNAL_DATE_FIELD_SECOND_RESOLUTION = "internaldateSecondResolution";
    private static final String INTERNAL_DATE_FIELD_MILLISECOND_RESOLUTION = "internaldateMillisecondResolution";
    private static final String MAILBOX_ID_FIELD = "mailboxid";
    private static final String USERS = "userSession";
    private static final String MESSAGE_ID_FIELD = "messageid";
    private static final String SENT_DATE_FIELD_YEAR_RESOLUTION = "sentdateYearResolution";
    private static final String SENT_DATE_FIELD_MONTH_RESOLUTION = "sentdateMonthResolution";
    private static final String SENT_DATE_FIELD_DAY_RESOLUTION = "sentdateDayResolution";
    private static final String SENT_DATE_FIELD_HOUR_RESOLUTION = "sentdateHourResolution";
    private static final String SENT_DATE_FIELD_MINUTE_RESOLUTION = "sentdateMinuteResolution";
    private static final String SENT_DATE_FIELD_SECOND_RESOLUTION = "sentdateSecondResolution";
    private static final String SENT_DATE_FIELD_MILLISECOND_RESOLUTION = "sentdateMillisecondResolution";
    private static final String SENT_DATE_SORT_FIELD_MILLISECOND_RESOLUTION = "sentdateSort";
    private static final String MEDIA_TYPE_TEXT = "text";
    private static final String MEDIA_TYPE_MESSAGE = "message";
    private static final String DEFAULT_ENCODING = "US-ASCII";
    private static final SortField UID_SORT;
    private static final SortField UID_SORT_REVERSE;
    private static final SortField SIZE_SORT;
    private static final SortField SIZE_SORT_REVERSE;
    private static final SortField FIRST_CC_MAILBOX_SORT;
    private static final SortField FIRST_CC_MAILBOX_SORT_REVERSE;
    private static final SortField FIRST_TO_MAILBOX_SORT;
    private static final SortField FIRST_TO_MAILBOX_SORT_REVERSE;
    private static final SortField FIRST_FROM_MAILBOX_SORT;
    private static final SortField FIRST_FROM_MAILBOX_SORT_REVERSE;
    private static final SortField ARRIVAL_MAILBOX_SORT;
    private static final SortField ARRIVAL_MAILBOX_SORT_REVERSE;
    private static final SortField BASE_SUBJECT_SORT;
    private static final SortField BASE_SUBJECT_SORT_REVERSE;
    private static final SortField SENT_DATE_SORT;
    private static final SortField SENT_DATE_SORT_REVERSE;
    private static final SortField FIRST_TO_MAILBOX_DISPLAY_SORT;
    private static final SortField FIRST_TO_MAILBOX_DISPLAY_SORT_REVERSE;
    private static final SortField FIRST_FROM_MAILBOX_DISPLAY_SORT;
    private static final SortField FIRST_FROM_MAILBOX_DISPLAY_SORT_REVERSE;
    private final MailboxId.Factory mailboxIdFactory;
    private final MessageId.Factory messageIdFactory;
    private final IndexWriter writer;
    private final Directory directory;
    private int maxQueryResults = 100000;
    private boolean suffixMatch = false;

    @Inject
    public LuceneMessageSearchIndex(MessageMapperFactory factory, MailboxId.Factory mailboxIdFactory, Directory directory, MessageId.Factory messageIdFactory) throws IOException {
        this(factory, mailboxIdFactory, directory, false, true, messageIdFactory);
    }

    public LuceneMessageSearchIndex(MessageMapperFactory factory, MailboxId.Factory mailboxIdFactory, Directory directory, boolean dropIndexOnStart, boolean lenient, MessageId.Factory messageIdFactory) throws IOException {
        super(factory);
        this.mailboxIdFactory = mailboxIdFactory;
        this.messageIdFactory = messageIdFactory;
        this.directory = directory;
        this.writer = new IndexWriter(this.directory, this.createConfig(this.createAnalyzer(lenient), dropIndexOnStart));
    }

    @PreDestroy
    public void close() throws IOException {
        try {
            this.writer.close();
        }
        finally {
            if (IndexWriter.isLocked((Directory)this.directory)) {
                IndexWriter.unlock((Directory)this.directory);
            }
        }
    }

    public MailboxListener.ListenerType getType() {
        return MailboxListener.ListenerType.EACH_NODE;
    }

    public EnumSet<MailboxManager.SearchCapabilities> getSupportedCapabilities(EnumSet<MailboxManager.MessageCapabilities> messageCapabilities) {
        return EnumSet.of(MailboxManager.SearchCapabilities.MultimailboxSearch);
    }

    public void setMaxQueryResults(int maxQueryResults) {
        this.maxQueryResults = maxQueryResults;
    }

    protected IndexWriterConfig createConfig(Analyzer analyzer, boolean dropIndexOnStart) {
        IndexWriterConfig config = new IndexWriterConfig(Version.LUCENE_31, analyzer);
        if (dropIndexOnStart) {
            config.setOpenMode(IndexWriterConfig.OpenMode.CREATE);
        } else {
            config.setOpenMode(IndexWriterConfig.OpenMode.CREATE_OR_APPEND);
        }
        return config;
    }

    protected Analyzer createAnalyzer(boolean lenient) {
        if (lenient) {
            return new LenientImapSearchAnalyzer();
        }
        return new StrictImapSearchAnalyzer();
    }

    public void setEnableSuffixMatch(boolean suffixMatch) {
        this.suffixMatch = suffixMatch;
    }

    public Iterator<MessageUid> search(MailboxSession session, org.apache.james.mailbox.store.mail.model.Mailbox mailbox, SearchQuery searchQuery) throws MailboxException {
        Preconditions.checkArgument((session != null ? 1 : 0) != 0, (Object)"'session' is mandatory");
        return this.searchMultimap((Collection<MailboxId>)ImmutableList.of((Object)mailbox.getMailboxId()), searchQuery).stream().map(MessageSearchIndex.SearchResult::getMessageUid).iterator();
    }

    public List<MessageId> search(MailboxSession session, Collection<MailboxId> mailboxIds, SearchQuery searchQuery, long limit) throws MailboxException {
        Preconditions.checkArgument((session != null ? 1 : 0) != 0, (Object)"'session' is mandatory");
        if (mailboxIds.isEmpty()) {
            return ImmutableList.of();
        }
        return (List)this.searchMultimap(mailboxIds, searchQuery).stream().map(searchResult -> (MessageId)searchResult.getMessageId().get()).filter(SearchUtil.distinct()).limit(Long.valueOf(limit).intValue()).collect(Guavate.toImmutableList());
    }

    private List<MessageSearchIndex.SearchResult> searchMultimap(Collection<MailboxId> mailboxIds, SearchQuery searchQuery) throws MailboxException {
        ImmutableList.Builder results = ImmutableList.builder();
        Query inMailboxes = this.buildQueryFromMailboxes(mailboxIds);
        try (IndexSearcher searcher = new IndexSearcher(IndexReader.open((IndexWriter)this.writer, (boolean)true));){
            ScoreDoc[] sDocs;
            BooleanQuery query = new BooleanQuery();
            query.add(inMailboxes, BooleanClause.Occur.MUST);
            query.add((Query)new PrefixQuery(new Term(FLAGS_FIELD, "")), BooleanClause.Occur.MUST_NOT);
            List crits = searchQuery.getCriterias();
            for (SearchQuery.Criterion crit : crits) {
                query.add(this.createQuery(crit, inMailboxes, searchQuery.getRecentMessageUids()), BooleanClause.Occur.MUST);
            }
            TopFieldDocs docs = searcher.search((Query)query, null, this.maxQueryResults, this.createSort(searchQuery.getSorts()));
            for (ScoreDoc sDoc : sDocs = docs.scoreDocs) {
                Document doc = searcher.doc(sDoc.doc);
                MessageUid uid = MessageUid.of((long)Long.valueOf(doc.get(UID_FIELD)));
                MailboxId mailboxId = this.mailboxIdFactory.fromString(doc.get(MAILBOX_ID_FIELD));
                Optional<MessageId> messageId = this.toMessageId(Optional.ofNullable(doc.get(MESSAGE_ID_FIELD)));
                results.add((Object)new MessageSearchIndex.SearchResult(messageId, mailboxId, uid));
            }
        }
        catch (IOException e) {
            throw new MailboxException("Unable to search the mailbox", (Throwable)e);
        }
        return results.build();
    }

    private Optional<MessageId> toMessageId(Optional<String> messageIdField) {
        if (messageIdField.isPresent()) {
            return Optional.of(this.messageIdFactory.fromString(messageIdField.get()));
        }
        return Optional.empty();
    }

    private Query buildQueryFromMailboxes(Collection<MailboxId> mailboxIds) {
        BooleanQuery query = new BooleanQuery();
        for (MailboxId id : mailboxIds) {
            String idAsString = id.serialize();
            query.add((Query)new TermQuery(new Term(MAILBOX_ID_FIELD, idAsString)), BooleanClause.Occur.SHOULD);
        }
        return query;
    }

    private Document createMessageDocument(MailboxSession session, final MailboxMessage membership) throws MailboxException {
        final Document doc = new Document();
        doc.add((Fieldable)new org.apache.lucene.document.Field(USERS, session.getUser().getUserName().toUpperCase(Locale.US), Field.Store.YES, Field.Index.NOT_ANALYZED));
        doc.add((Fieldable)new org.apache.lucene.document.Field(MAILBOX_ID_FIELD, membership.getMailboxId().serialize().toUpperCase(Locale.US), Field.Store.YES, Field.Index.NOT_ANALYZED));
        doc.add((Fieldable)new NumericField(UID_FIELD, Field.Store.YES, true).setLongValue(membership.getUid().asLong()));
        doc.add((Fieldable)new org.apache.lucene.document.Field(HAS_ATTACHMENT_FIELD, Boolean.toString(LuceneMessageSearchIndex.hasAttachment(membership)), Field.Store.YES, Field.Index.NOT_ANALYZED));
        doc.add((Fieldable)new org.apache.lucene.document.Field(MESSAGE_ID_FIELD, SearchUtil.getSerializedMessageIdIfSupportedByUnderlyingStorageOrNull((MailboxMessage)membership), Field.Store.YES, Field.Index.NOT_ANALYZED));
        doc.add((Fieldable)new org.apache.lucene.document.Field(ID_FIELD, membership.getMailboxId().serialize().toUpperCase(Locale.US) + "-" + Long.toString(membership.getUid().asLong()), Field.Store.YES, Field.Index.NOT_ANALYZED));
        doc.add((Fieldable)new org.apache.lucene.document.Field(INTERNAL_DATE_FIELD_YEAR_RESOLUTION, DateTools.dateToString((Date)membership.getInternalDate(), (DateTools.Resolution)DateTools.Resolution.YEAR), Field.Store.NO, Field.Index.NOT_ANALYZED));
        doc.add((Fieldable)new org.apache.lucene.document.Field(INTERNAL_DATE_FIELD_MONTH_RESOLUTION, DateTools.dateToString((Date)membership.getInternalDate(), (DateTools.Resolution)DateTools.Resolution.MONTH), Field.Store.NO, Field.Index.NOT_ANALYZED));
        doc.add((Fieldable)new org.apache.lucene.document.Field(INTERNAL_DATE_FIELD_DAY_RESOLUTION, DateTools.dateToString((Date)membership.getInternalDate(), (DateTools.Resolution)DateTools.Resolution.DAY), Field.Store.NO, Field.Index.NOT_ANALYZED));
        doc.add((Fieldable)new org.apache.lucene.document.Field(INTERNAL_DATE_FIELD_HOUR_RESOLUTION, DateTools.dateToString((Date)membership.getInternalDate(), (DateTools.Resolution)DateTools.Resolution.HOUR), Field.Store.NO, Field.Index.NOT_ANALYZED));
        doc.add((Fieldable)new org.apache.lucene.document.Field(INTERNAL_DATE_FIELD_MINUTE_RESOLUTION, DateTools.dateToString((Date)membership.getInternalDate(), (DateTools.Resolution)DateTools.Resolution.MINUTE), Field.Store.NO, Field.Index.NOT_ANALYZED));
        doc.add((Fieldable)new org.apache.lucene.document.Field(INTERNAL_DATE_FIELD_SECOND_RESOLUTION, DateTools.dateToString((Date)membership.getInternalDate(), (DateTools.Resolution)DateTools.Resolution.SECOND), Field.Store.NO, Field.Index.NOT_ANALYZED));
        doc.add((Fieldable)new org.apache.lucene.document.Field(INTERNAL_DATE_FIELD_MILLISECOND_RESOLUTION, DateTools.dateToString((Date)membership.getInternalDate(), (DateTools.Resolution)DateTools.Resolution.MILLISECOND), Field.Store.NO, Field.Index.NOT_ANALYZED));
        doc.add((Fieldable)new NumericField(SIZE_FIELD, Field.Store.YES, true).setLongValue(membership.getFullContentOctets()));
        SimpleContentHandler handler = new SimpleContentHandler(){

            public void headers(Header header) {
                Date sentDate = null;
                String firstFromMailbox = "";
                String firstToMailbox = "";
                String firstCcMailbox = "";
                String firstFromDisplay = "";
                String firstToDisplay = "";
                for (Field f : header) {
                    String headerName = f.getName().toUpperCase(Locale.US);
                    String headerValue = f.getBody().toUpperCase(Locale.US);
                    String fullValue = f.toString().toUpperCase(Locale.US);
                    doc.add((Fieldable)new org.apache.lucene.document.Field(LuceneMessageSearchIndex.HEADERS_FIELD, fullValue, Field.Store.NO, Field.Index.ANALYZED));
                    doc.add((Fieldable)new org.apache.lucene.document.Field(LuceneMessageSearchIndex.PREFIX_HEADER_FIELD + headerName, headerValue, Field.Store.NO, Field.Index.ANALYZED));
                    if (f instanceof DateTimeField) {
                        try (StringReader reader = new StringReader(f.getBody());){
                            DateTime dateTime = new DateTimeParser((Reader)reader).parseAll();
                            Calendar cal = LuceneMessageSearchIndex.getGMT();
                            cal.set(dateTime.getYear(), dateTime.getMonth() - 1, dateTime.getDay(), dateTime.getHour(), dateTime.getMinute(), dateTime.getSecond());
                            sentDate = cal.getTime();
                        }
                        catch (ParseException e) {
                            LOGGER.debug("Unable to parse Date header for proper indexing", (Throwable)e);
                            sentDate = ((DateTimeField)f).getDate();
                        }
                        if (sentDate == null) {
                            sentDate = membership.getInternalDate();
                        }
                    }
                    String field = null;
                    if ("To".equalsIgnoreCase(headerName)) {
                        field = LuceneMessageSearchIndex.TO_FIELD;
                    } else if ("From".equalsIgnoreCase(headerName)) {
                        field = LuceneMessageSearchIndex.FROM_FIELD;
                    } else if ("Cc".equalsIgnoreCase(headerName)) {
                        field = LuceneMessageSearchIndex.CC_FIELD;
                    } else if ("Bcc".equalsIgnoreCase(headerName)) {
                        field = LuceneMessageSearchIndex.BCC_FIELD;
                    }
                    if (field != null) {
                        AddressList aList = LenientAddressParser.DEFAULT.parseAddressList((CharSequence)MimeUtil.unfold((String)f.getBody()));
                        for (int i = 0; i < aList.size(); ++i) {
                            Address address = aList.get(i);
                            if (address instanceof Mailbox) {
                                Mailbox mailbox = (Mailbox)address;
                                String value = AddressFormatter.DEFAULT.encode(mailbox).toUpperCase(Locale.US);
                                doc.add((Fieldable)new org.apache.lucene.document.Field(field, value, Field.Store.NO, Field.Index.ANALYZED));
                                if (i != 0) continue;
                                String mailboxAddress = SearchUtil.getMailboxAddress((Mailbox)mailbox);
                                String mailboxDisplay = SearchUtil.getDisplayAddress((Mailbox)mailbox);
                                if ("To".equalsIgnoreCase(headerName)) {
                                    firstToMailbox = mailboxAddress;
                                    firstToDisplay = mailboxDisplay;
                                    continue;
                                }
                                if ("From".equalsIgnoreCase(headerName)) {
                                    firstFromMailbox = mailboxAddress;
                                    firstFromDisplay = mailboxDisplay;
                                    continue;
                                }
                                if (!"Cc".equalsIgnoreCase(headerName)) continue;
                                firstCcMailbox = mailboxAddress;
                                continue;
                            }
                            if (!(address instanceof Group)) continue;
                            MailboxList mList = ((Group)address).getMailboxes();
                            for (int a = 0; a < mList.size(); ++a) {
                                Mailbox mailbox = mList.get(a);
                                String value = AddressFormatter.DEFAULT.encode(mailbox).toUpperCase(Locale.US);
                                doc.add((Fieldable)new org.apache.lucene.document.Field(field, value, Field.Store.NO, Field.Index.ANALYZED));
                                if (i != 0 || a != 0) continue;
                                String mailboxAddress = SearchUtil.getMailboxAddress((Mailbox)mailbox);
                                String mailboxDisplay = SearchUtil.getDisplayAddress((Mailbox)mailbox);
                                if ("To".equalsIgnoreCase(headerName)) {
                                    firstToMailbox = mailboxAddress;
                                    firstToDisplay = mailboxDisplay;
                                    continue;
                                }
                                if ("From".equalsIgnoreCase(headerName)) {
                                    firstFromMailbox = mailboxAddress;
                                    firstFromDisplay = mailboxDisplay;
                                    continue;
                                }
                                if (!"Cc".equalsIgnoreCase(headerName)) continue;
                                firstCcMailbox = mailboxAddress;
                            }
                        }
                        doc.add((Fieldable)new org.apache.lucene.document.Field(field, headerValue, Field.Store.NO, Field.Index.ANALYZED));
                        continue;
                    }
                    if (!headerName.equalsIgnoreCase("Subject")) continue;
                    doc.add((Fieldable)new org.apache.lucene.document.Field(LuceneMessageSearchIndex.BASE_SUBJECT_FIELD, SearchUtil.getBaseSubject((String)headerValue), Field.Store.YES, Field.Index.NOT_ANALYZED));
                }
                if (sentDate == null) {
                    sentDate = membership.getInternalDate();
                } else {
                    doc.add((Fieldable)new org.apache.lucene.document.Field(LuceneMessageSearchIndex.SENT_DATE_FIELD_YEAR_RESOLUTION, DateTools.dateToString((Date)sentDate, (DateTools.Resolution)DateTools.Resolution.YEAR), Field.Store.NO, Field.Index.NOT_ANALYZED));
                    doc.add((Fieldable)new org.apache.lucene.document.Field(LuceneMessageSearchIndex.SENT_DATE_FIELD_MONTH_RESOLUTION, DateTools.dateToString((Date)sentDate, (DateTools.Resolution)DateTools.Resolution.MONTH), Field.Store.NO, Field.Index.NOT_ANALYZED));
                    doc.add((Fieldable)new org.apache.lucene.document.Field(LuceneMessageSearchIndex.SENT_DATE_FIELD_DAY_RESOLUTION, DateTools.dateToString((Date)sentDate, (DateTools.Resolution)DateTools.Resolution.DAY), Field.Store.NO, Field.Index.NOT_ANALYZED));
                    doc.add((Fieldable)new org.apache.lucene.document.Field(LuceneMessageSearchIndex.SENT_DATE_FIELD_HOUR_RESOLUTION, DateTools.dateToString((Date)sentDate, (DateTools.Resolution)DateTools.Resolution.HOUR), Field.Store.NO, Field.Index.NOT_ANALYZED));
                    doc.add((Fieldable)new org.apache.lucene.document.Field(LuceneMessageSearchIndex.SENT_DATE_FIELD_MINUTE_RESOLUTION, DateTools.dateToString((Date)sentDate, (DateTools.Resolution)DateTools.Resolution.MINUTE), Field.Store.NO, Field.Index.NOT_ANALYZED));
                    doc.add((Fieldable)new org.apache.lucene.document.Field(LuceneMessageSearchIndex.SENT_DATE_FIELD_SECOND_RESOLUTION, DateTools.dateToString((Date)sentDate, (DateTools.Resolution)DateTools.Resolution.SECOND), Field.Store.NO, Field.Index.NOT_ANALYZED));
                    doc.add((Fieldable)new org.apache.lucene.document.Field(LuceneMessageSearchIndex.SENT_DATE_FIELD_MILLISECOND_RESOLUTION, DateTools.dateToString((Date)sentDate, (DateTools.Resolution)DateTools.Resolution.MILLISECOND), Field.Store.NO, Field.Index.NOT_ANALYZED));
                }
                doc.add((Fieldable)new org.apache.lucene.document.Field(LuceneMessageSearchIndex.SENT_DATE_SORT_FIELD_MILLISECOND_RESOLUTION, DateTools.dateToString((Date)sentDate, (DateTools.Resolution)DateTools.Resolution.MILLISECOND), Field.Store.NO, Field.Index.NOT_ANALYZED));
                doc.add((Fieldable)new org.apache.lucene.document.Field(LuceneMessageSearchIndex.FIRST_FROM_MAILBOX_NAME_FIELD, firstFromMailbox, Field.Store.YES, Field.Index.NOT_ANALYZED));
                doc.add((Fieldable)new org.apache.lucene.document.Field(LuceneMessageSearchIndex.FIRST_TO_MAILBOX_NAME_FIELD, firstToMailbox, Field.Store.YES, Field.Index.NOT_ANALYZED));
                doc.add((Fieldable)new org.apache.lucene.document.Field(LuceneMessageSearchIndex.FIRST_CC_MAILBOX_NAME_FIELD, firstCcMailbox, Field.Store.YES, Field.Index.NOT_ANALYZED));
                doc.add((Fieldable)new org.apache.lucene.document.Field(LuceneMessageSearchIndex.FIRST_FROM_MAILBOX_DISPLAY_FIELD, firstFromDisplay, Field.Store.YES, Field.Index.NOT_ANALYZED));
                doc.add((Fieldable)new org.apache.lucene.document.Field(LuceneMessageSearchIndex.FIRST_TO_MAILBOX_DISPLAY_FIELD, firstToDisplay, Field.Store.YES, Field.Index.NOT_ANALYZED));
            }

            public void body(BodyDescriptor desc, InputStream in) throws MimeException, IOException {
                String mediaType = desc.getMediaType();
                if (LuceneMessageSearchIndex.MEDIA_TYPE_TEXT.equalsIgnoreCase(mediaType) || LuceneMessageSearchIndex.MEDIA_TYPE_MESSAGE.equalsIgnoreCase(mediaType)) {
                    Charset charset;
                    String cset = desc.getCharset();
                    if (cset == null) {
                        cset = LuceneMessageSearchIndex.DEFAULT_ENCODING;
                    }
                    try {
                        charset = Charset.forName(cset);
                    }
                    catch (Exception e) {
                        charset = Charset.forName(LuceneMessageSearchIndex.DEFAULT_ENCODING);
                    }
                    try (BufferedReader bodyReader = new BufferedReader(new InputStreamReader(in, charset));){
                        String line = null;
                        while ((line = bodyReader.readLine()) != null) {
                            doc.add((Fieldable)new org.apache.lucene.document.Field(LuceneMessageSearchIndex.BODY_FIELD, line.toUpperCase(Locale.US), Field.Store.NO, Field.Index.ANALYZED));
                        }
                    }
                }
            }
        };
        MimeStreamParser parser = new MimeStreamParser(MimeConfig.PERMISSIVE);
        parser.setContentDecoding(true);
        parser.setContentHandler((ContentHandler)handler);
        try {
            parser.parse(membership.getFullContent());
        }
        catch (IOException | MimeException e) {
            throw new MailboxException("Unable to index content of message", e);
        }
        return doc;
    }

    private static boolean hasAttachment(MailboxMessage membership) {
        return membership.getProperties().stream().anyMatch(PropertyBuilder.isHasAttachmentProperty());
    }

    private String toSentDateField(SearchQuery.DateResolution res) {
        switch (res) {
            case Year: {
                return SENT_DATE_FIELD_YEAR_RESOLUTION;
            }
            case Month: {
                return SENT_DATE_FIELD_MONTH_RESOLUTION;
            }
            case Day: {
                return SENT_DATE_FIELD_DAY_RESOLUTION;
            }
            case Hour: {
                return SENT_DATE_FIELD_HOUR_RESOLUTION;
            }
            case Minute: {
                return SENT_DATE_FIELD_MINUTE_RESOLUTION;
            }
            case Second: {
                return SENT_DATE_FIELD_SECOND_RESOLUTION;
            }
        }
        return SENT_DATE_FIELD_MILLISECOND_RESOLUTION;
    }

    private static Calendar getGMT() {
        return Calendar.getInstance(TimeZone.getTimeZone("GMT"), Locale.US);
    }

    private String toInteralDateField(SearchQuery.DateResolution res) {
        switch (res) {
            case Year: {
                return INTERNAL_DATE_FIELD_YEAR_RESOLUTION;
            }
            case Month: {
                return INTERNAL_DATE_FIELD_MONTH_RESOLUTION;
            }
            case Day: {
                return INTERNAL_DATE_FIELD_DAY_RESOLUTION;
            }
            case Hour: {
                return INTERNAL_DATE_FIELD_HOUR_RESOLUTION;
            }
            case Minute: {
                return INTERNAL_DATE_FIELD_MINUTE_RESOLUTION;
            }
            case Second: {
                return INTERNAL_DATE_FIELD_SECOND_RESOLUTION;
            }
        }
        return INTERNAL_DATE_FIELD_MILLISECOND_RESOLUTION;
    }

    private Query createInternalDateQuery(SearchQuery.InternalDateCriterion crit) throws UnsupportedSearchException {
        SearchQuery.DateOperator dop = crit.getOperator();
        SearchQuery.DateResolution res = dop.getDateResultion();
        String field = this.toInteralDateField(res);
        return this.createQuery(field, dop);
    }

    private Query createSizeQuery(SearchQuery.SizeCriterion crit) throws UnsupportedSearchException {
        SearchQuery.NumericOperator op = crit.getOperator();
        switch (op.getType()) {
            case EQUALS: {
                return NumericRangeQuery.newLongRange((String)SIZE_FIELD, (Long)op.getValue(), (Long)op.getValue(), (boolean)true, (boolean)true);
            }
            case GREATER_THAN: {
                return NumericRangeQuery.newLongRange((String)SIZE_FIELD, (Long)op.getValue(), (Long)Long.MAX_VALUE, (boolean)false, (boolean)true);
            }
            case LESS_THAN: {
                return NumericRangeQuery.newLongRange((String)SIZE_FIELD, (Long)Long.MIN_VALUE, (Long)op.getValue(), (boolean)true, (boolean)false);
            }
        }
        throw new UnsupportedSearchException();
    }

    private Query createTermQuery(String fieldName, String value) {
        if (this.suffixMatch) {
            return new WildcardQuery(new Term(fieldName, "*" + value + "*"));
        }
        return new PrefixQuery(new Term(fieldName, value));
    }

    private Query createHeaderQuery(SearchQuery.HeaderCriterion crit) throws UnsupportedSearchException {
        SearchQuery.HeaderOperator op = crit.getOperator();
        String name = crit.getHeaderName().toUpperCase(Locale.US);
        String fieldName = PREFIX_HEADER_FIELD + name;
        if (op instanceof SearchQuery.ContainsOperator) {
            SearchQuery.ContainsOperator cop = (SearchQuery.ContainsOperator)op;
            return this.createTermQuery(fieldName, cop.getValue().toUpperCase(Locale.US));
        }
        if (op instanceof SearchQuery.ExistsOperator) {
            return new PrefixQuery(new Term(fieldName, ""));
        }
        if (op instanceof SearchQuery.DateOperator) {
            SearchQuery.DateOperator dop = (SearchQuery.DateOperator)op;
            String field = this.toSentDateField(dop.getDateResultion());
            return this.createQuery(field, dop);
        }
        if (op instanceof SearchQuery.AddressOperator) {
            String field = name.toLowerCase(Locale.US);
            return this.createTermQuery(field, ((SearchQuery.AddressOperator)op).getAddress().toUpperCase(Locale.US));
        }
        throw new UnsupportedSearchException();
    }

    private Query createQuery(String field, SearchQuery.DateOperator dop) throws UnsupportedSearchException {
        Date date = dop.getDate();
        SearchQuery.DateResolution res = dop.getDateResultion();
        DateTools.Resolution dRes = this.toResolution(res);
        String value = DateTools.dateToString((Date)date, (DateTools.Resolution)dRes);
        switch (dop.getType()) {
            case ON: {
                return new TermQuery(new Term(field, value));
            }
            case BEFORE: {
                return new TermRangeQuery(field, DateTools.dateToString((Date)MIN_DATE, (DateTools.Resolution)dRes), value, true, false);
            }
            case AFTER: {
                return new TermRangeQuery(field, value, DateTools.dateToString((Date)MAX_DATE, (DateTools.Resolution)dRes), false, true);
            }
        }
        throw new UnsupportedSearchException();
    }

    private DateTools.Resolution toResolution(SearchQuery.DateResolution res) {
        switch (res) {
            case Year: {
                return DateTools.Resolution.YEAR;
            }
            case Month: {
                return DateTools.Resolution.MONTH;
            }
            case Day: {
                return DateTools.Resolution.DAY;
            }
            case Hour: {
                return DateTools.Resolution.HOUR;
            }
            case Minute: {
                return DateTools.Resolution.MINUTE;
            }
            case Second: {
                return DateTools.Resolution.SECOND;
            }
        }
        return DateTools.Resolution.MILLISECOND;
    }

    private Query createUidQuery(SearchQuery.UidCriterion crit) throws UnsupportedSearchException {
        SearchQuery.UidRange[] ranges = crit.getOperator().getRange();
        if (ranges.length == 1) {
            SearchQuery.UidRange range = ranges[0];
            return NumericRangeQuery.newLongRange((String)UID_FIELD, (Long)range.getLowValue().asLong(), (Long)range.getHighValue().asLong(), (boolean)true, (boolean)true);
        }
        BooleanQuery rangesQuery = new BooleanQuery();
        for (SearchQuery.UidRange range : ranges) {
            rangesQuery.add((Query)NumericRangeQuery.newLongRange((String)UID_FIELD, (Long)range.getLowValue().asLong(), (Long)range.getHighValue().asLong(), (boolean)true, (boolean)true), BooleanClause.Occur.SHOULD);
        }
        return rangesQuery;
    }

    private Query createModSeqQuery(SearchQuery.ModSeqCriterion crit) throws UnsupportedSearchException {
        SearchQuery.NumericOperator op = crit.getOperator();
        switch (op.getType()) {
            case EQUALS: {
                return NumericRangeQuery.newLongRange((String)MODSEQ_FIELD, (Long)op.getValue(), (Long)op.getValue(), (boolean)true, (boolean)true);
            }
            case GREATER_THAN: {
                return NumericRangeQuery.newLongRange((String)MODSEQ_FIELD, (Long)op.getValue(), (Long)Long.MAX_VALUE, (boolean)false, (boolean)true);
            }
            case LESS_THAN: {
                return NumericRangeQuery.newLongRange((String)MODSEQ_FIELD, (Long)Long.MIN_VALUE, (Long)op.getValue(), (boolean)true, (boolean)false);
            }
        }
        throw new UnsupportedSearchException();
    }

    private Query createAttachmentQuery(boolean isSet) throws MailboxException {
        return new TermQuery(new Term(HAS_ATTACHMENT_FIELD, Boolean.toString(isSet)));
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private Query createFlagQuery(String flag, boolean isSet, Query inMailboxes, Collection<MessageUid> recentUids) throws MailboxException {
        BooleanQuery query = new BooleanQuery();
        if (isSet) {
            query.add((Query)new TermQuery(new Term(FLAGS_FIELD, flag)), BooleanClause.Occur.MUST);
        } else {
            BooleanQuery bQuery = new BooleanQuery();
            bQuery.add((Query)new PrefixQuery(new Term(FLAGS_FIELD, "")), BooleanClause.Occur.MUST);
            bQuery.add((Query)new TermQuery(new Term(FLAGS_FIELD, flag)), BooleanClause.Occur.MUST_NOT);
            query.add((Query)bQuery, BooleanClause.Occur.MUST);
        }
        query.add(inMailboxes, BooleanClause.Occur.MUST);
        try (IndexSearcher searcher = new IndexSearcher(IndexReader.open((IndexWriter)this.writer, (boolean)true));){
            ScoreDoc[] sDocs;
            HashSet<MessageUid> uids = new HashSet<MessageUid>();
            TopFieldDocs docs = searcher.search((Query)query, null, this.maxQueryResults, new Sort(UID_SORT));
            for (ScoreDoc sDoc : sDocs = docs.scoreDocs) {
                MessageUid uid = MessageUid.of((long)Long.valueOf(searcher.doc(sDoc.doc).get(UID_FIELD)));
                uids.add(uid);
            }
            if (flag.equalsIgnoreCase("\\RECENT")) {
                if (isSet) {
                    uids.addAll(recentUids);
                } else {
                    uids.removeAll(recentUids);
                }
            }
            List ranges = MessageRange.toRanges(new ArrayList(uids));
            SearchQuery.UidRange[] nRanges = new SearchQuery.UidRange[ranges.size()];
            for (int i = 0; i < ranges.size(); ++i) {
                MessageRange range = (MessageRange)ranges.get(i);
                nRanges[i] = new SearchQuery.UidRange(range.getUidFrom(), range.getUidTo());
            }
            Query query2 = this.createUidQuery((SearchQuery.UidCriterion)SearchQuery.uid((SearchQuery.UidRange[])nRanges));
            return query2;
        }
        catch (IOException e) {
            throw new MailboxException("Unable to search mailbox " + inMailboxes, (Throwable)e);
        }
    }

    private Sort createSort(List<SearchQuery.Sort> sorts) {
        ArrayList<SortField> fields = new ArrayList<SortField>();
        for (SearchQuery.Sort sort : sorts) {
            boolean reverse;
            SortField sortField = this.createSortField(sort, reverse = sort.isReverse());
            if (sortField == null) continue;
            fields.add(sortField);
            if (sortField == SENT_DATE_SORT) {
                fields.add(UID_SORT);
                continue;
            }
            if (sortField != SENT_DATE_SORT_REVERSE) continue;
            fields.add(UID_SORT_REVERSE);
        }
        fields.add(UID_SORT);
        Sort sort = new Sort();
        sort.setSort(fields.toArray(new SortField[0]));
        return sort;
    }

    private SortField createSortField(SearchQuery.Sort s, boolean reverse) {
        switch (s.getSortClause()) {
            case Arrival: {
                if (reverse) {
                    return ARRIVAL_MAILBOX_SORT_REVERSE;
                }
                return ARRIVAL_MAILBOX_SORT;
            }
            case SentDate: {
                if (reverse) {
                    return SENT_DATE_SORT_REVERSE;
                }
                return SENT_DATE_SORT;
            }
            case MailboxCc: {
                if (reverse) {
                    return FIRST_CC_MAILBOX_SORT_REVERSE;
                }
                return FIRST_CC_MAILBOX_SORT;
            }
            case MailboxFrom: {
                if (reverse) {
                    return FIRST_FROM_MAILBOX_SORT_REVERSE;
                }
                return FIRST_FROM_MAILBOX_SORT;
            }
            case Size: {
                if (reverse) {
                    return SIZE_SORT_REVERSE;
                }
                return SIZE_SORT;
            }
            case BaseSubject: {
                if (reverse) {
                    return BASE_SUBJECT_SORT_REVERSE;
                }
                return BASE_SUBJECT_SORT;
            }
            case MailboxTo: {
                if (reverse) {
                    return FIRST_TO_MAILBOX_SORT_REVERSE;
                }
                return FIRST_TO_MAILBOX_SORT;
            }
            case Uid: {
                if (reverse) {
                    return UID_SORT_REVERSE;
                }
                return UID_SORT;
            }
            case DisplayFrom: {
                if (reverse) {
                    return FIRST_FROM_MAILBOX_DISPLAY_SORT_REVERSE;
                }
                return FIRST_FROM_MAILBOX_DISPLAY_SORT;
            }
            case DisplayTo: {
                if (reverse) {
                    return FIRST_TO_MAILBOX_DISPLAY_SORT_REVERSE;
                }
                return FIRST_TO_MAILBOX_DISPLAY_SORT;
            }
        }
        return null;
    }

    private String toString(Flags.Flag flag) {
        if (Flags.Flag.ANSWERED.equals(flag)) {
            return "\\ANSWERED";
        }
        if (Flags.Flag.DELETED.equals(flag)) {
            return "\\DELETED";
        }
        if (Flags.Flag.DRAFT.equals(flag)) {
            return "\\DRAFT";
        }
        if (Flags.Flag.FLAGGED.equals(flag)) {
            return "\\FLAGGED";
        }
        if (Flags.Flag.RECENT.equals(flag)) {
            return "\\RECENT";
        }
        if (Flags.Flag.SEEN.equals(flag)) {
            return "\\FLAG";
        }
        return flag.toString();
    }

    private Query createTextQuery(SearchQuery.TextCriterion crit) throws UnsupportedSearchException {
        String value = crit.getOperator().getValue().toUpperCase(Locale.US);
        switch (crit.getType()) {
            case BODY: {
                return this.createTermQuery(BODY_FIELD, value);
            }
            case FULL: {
                BooleanQuery query = new BooleanQuery();
                query.add(this.createTermQuery(BODY_FIELD, value), BooleanClause.Occur.SHOULD);
                query.add(this.createTermQuery(HEADERS_FIELD, value), BooleanClause.Occur.SHOULD);
                return query;
            }
        }
        throw new UnsupportedSearchException();
    }

    private Query createAllQuery(SearchQuery.AllCriterion crit) throws UnsupportedSearchException {
        BooleanQuery query = new BooleanQuery();
        query.add(this.createQuery(MessageRange.all()), BooleanClause.Occur.MUST);
        query.add((Query)new PrefixQuery(new Term(FLAGS_FIELD, "")), BooleanClause.Occur.MUST_NOT);
        return query;
    }

    private Query createConjunctionQuery(SearchQuery.ConjunctionCriterion crit, Query inMailboxes, Collection<MessageUid> recentUids) throws UnsupportedSearchException, MailboxException {
        List crits = crit.getCriteria();
        BooleanQuery conQuery = new BooleanQuery();
        switch (crit.getType()) {
            case AND: {
                for (SearchQuery.Criterion criterion : crits) {
                    conQuery.add(this.createQuery(criterion, inMailboxes, recentUids), BooleanClause.Occur.MUST);
                }
                return conQuery;
            }
            case OR: {
                for (SearchQuery.Criterion criterion : crits) {
                    conQuery.add(this.createQuery(criterion, inMailboxes, recentUids), BooleanClause.Occur.SHOULD);
                }
                return conQuery;
            }
            case NOR: {
                BooleanQuery nor = new BooleanQuery();
                for (SearchQuery.Criterion criterion : crits) {
                    conQuery.add(this.createQuery(criterion, inMailboxes, recentUids), BooleanClause.Occur.SHOULD);
                }
                nor.add(inMailboxes, BooleanClause.Occur.MUST);
                nor.add((Query)conQuery, BooleanClause.Occur.MUST_NOT);
                return nor;
            }
        }
        throw new UnsupportedSearchException();
    }

    private Query createQuery(SearchQuery.Criterion criterion, Query inMailboxes, Collection<MessageUid> recentUids) throws MailboxException {
        if (criterion instanceof SearchQuery.InternalDateCriterion) {
            SearchQuery.InternalDateCriterion crit = (SearchQuery.InternalDateCriterion)criterion;
            return this.createInternalDateQuery(crit);
        }
        if (criterion instanceof SearchQuery.SizeCriterion) {
            SearchQuery.SizeCriterion crit = (SearchQuery.SizeCriterion)criterion;
            return this.createSizeQuery(crit);
        }
        if (criterion instanceof SearchQuery.HeaderCriterion) {
            SearchQuery.HeaderCriterion crit = (SearchQuery.HeaderCriterion)criterion;
            return this.createHeaderQuery(crit);
        }
        if (criterion instanceof SearchQuery.UidCriterion) {
            SearchQuery.UidCriterion crit = (SearchQuery.UidCriterion)criterion;
            return this.createUidQuery(crit);
        }
        if (criterion instanceof SearchQuery.FlagCriterion) {
            SearchQuery.FlagCriterion crit = (SearchQuery.FlagCriterion)criterion;
            return this.createFlagQuery(this.toString(crit.getFlag()), crit.getOperator().isSet(), inMailboxes, recentUids);
        }
        if (criterion instanceof SearchQuery.AttachmentCriterion) {
            SearchQuery.AttachmentCriterion crit = (SearchQuery.AttachmentCriterion)criterion;
            return this.createAttachmentQuery(crit.getOperator().isSet());
        }
        if (criterion instanceof SearchQuery.CustomFlagCriterion) {
            SearchQuery.CustomFlagCriterion crit = (SearchQuery.CustomFlagCriterion)criterion;
            return this.createFlagQuery(crit.getFlag(), crit.getOperator().isSet(), inMailboxes, recentUids);
        }
        if (criterion instanceof SearchQuery.TextCriterion) {
            SearchQuery.TextCriterion crit = (SearchQuery.TextCriterion)criterion;
            return this.createTextQuery(crit);
        }
        if (criterion instanceof SearchQuery.AllCriterion) {
            return this.createAllQuery((SearchQuery.AllCriterion)criterion);
        }
        if (criterion instanceof SearchQuery.ConjunctionCriterion) {
            SearchQuery.ConjunctionCriterion crit = (SearchQuery.ConjunctionCriterion)criterion;
            return this.createConjunctionQuery(crit, inMailboxes, recentUids);
        }
        if (criterion instanceof SearchQuery.ModSeqCriterion) {
            return this.createModSeqQuery((SearchQuery.ModSeqCriterion)criterion);
        }
        if (criterion instanceof SearchQuery.MimeMessageIDCriterion) {
            SearchQuery.MimeMessageIDCriterion mimeMessageIDCriterion = (SearchQuery.MimeMessageIDCriterion)criterion;
            return this.createHeaderQuery(mimeMessageIDCriterion.asHeaderCriterion());
        }
        throw new UnsupportedSearchException();
    }

    public void add(MailboxSession session, org.apache.james.mailbox.store.mail.model.Mailbox mailbox, MailboxMessage membership) throws MailboxException {
        Document doc = this.createMessageDocument(session, membership);
        Document flagsDoc = this.createFlagsDocument(membership);
        try {
            this.writer.addDocument(doc);
            this.writer.addDocument(flagsDoc);
        }
        catch (IOException e) {
            throw new MailboxException("Unable to add message to index", (Throwable)e);
        }
    }

    public void update(MailboxSession session, org.apache.james.mailbox.store.mail.model.Mailbox mailbox, List<UpdatedFlags> updatedFlagsList) throws MailboxException {
        for (UpdatedFlags updatedFlags : updatedFlagsList) {
            this.update(mailbox, updatedFlags.getUid(), updatedFlags.getNewFlags());
        }
    }

    private void update(org.apache.james.mailbox.store.mail.model.Mailbox mailbox, MessageUid uid, Flags f) throws MailboxException {
        try (IndexSearcher searcher = new IndexSearcher(IndexReader.open((IndexWriter)this.writer, (boolean)true));){
            ScoreDoc[] sDocs;
            BooleanQuery query = new BooleanQuery();
            query.add((Query)new TermQuery(new Term(MAILBOX_ID_FIELD, mailbox.getMailboxId().serialize())), BooleanClause.Occur.MUST);
            query.add(this.createQuery(MessageRange.one((MessageUid)uid)), BooleanClause.Occur.MUST);
            query.add((Query)new PrefixQuery(new Term(FLAGS_FIELD, "")), BooleanClause.Occur.MUST);
            TopDocs docs = searcher.search((Query)query, 100000);
            for (ScoreDoc sDoc : sDocs = docs.scoreDocs) {
                Document doc = searcher.doc(sDoc.doc);
                if (doc.getFieldable(FLAGS_FIELD) != null) continue;
                doc.removeFields(FLAGS_FIELD);
                this.indexFlags(doc, f);
                this.writer.updateDocument(new Term(ID_FIELD, doc.get(ID_FIELD)), doc);
            }
        }
        catch (IOException e) {
            throw new MailboxException("Unable to add messages in index", (Throwable)e);
        }
    }

    private Document createFlagsDocument(MailboxMessage message) {
        Document doc = new Document();
        doc.add((Fieldable)new org.apache.lucene.document.Field(ID_FIELD, "flags-" + message.getMailboxId().serialize() + "-" + Long.toString(message.getUid().asLong()), Field.Store.YES, Field.Index.NOT_ANALYZED));
        doc.add((Fieldable)new org.apache.lucene.document.Field(MAILBOX_ID_FIELD, message.getMailboxId().serialize(), Field.Store.YES, Field.Index.NOT_ANALYZED));
        doc.add((Fieldable)new NumericField(UID_FIELD, Field.Store.YES, true).setLongValue(message.getUid().asLong()));
        this.indexFlags(doc, message.createFlags());
        return doc;
    }

    private void indexFlags(Document doc, Flags f) {
        String[] userFlags;
        Flags.Flag[] flags;
        ArrayList<String> fString = new ArrayList<String>();
        for (Flags.Flag flag : flags = f.getSystemFlags()) {
            fString.add(this.toString(flag));
            doc.add((Fieldable)new org.apache.lucene.document.Field(FLAGS_FIELD, this.toString(flag), Field.Store.NO, Field.Index.NOT_ANALYZED));
        }
        for (String userFlag : userFlags = f.getUserFlags()) {
            doc.add((Fieldable)new org.apache.lucene.document.Field(FLAGS_FIELD, userFlag, Field.Store.NO, Field.Index.NOT_ANALYZED));
        }
        if (flags.length == 0 && userFlags.length == 0) {
            doc.add((Fieldable)new org.apache.lucene.document.Field(FLAGS_FIELD, "", Field.Store.NO, Field.Index.NOT_ANALYZED));
        }
    }

    private Query createQuery(MessageRange range) {
        switch (range.getType()) {
            case ONE: {
                return NumericRangeQuery.newLongRange((String)UID_FIELD, (Long)range.getUidFrom().asLong(), (Long)range.getUidTo().asLong(), (boolean)true, (boolean)true);
            }
            case FROM: {
                return NumericRangeQuery.newLongRange((String)UID_FIELD, (Long)range.getUidFrom().asLong(), (Long)MessageUid.MAX_VALUE.asLong(), (boolean)true, (boolean)true);
            }
        }
        return NumericRangeQuery.newLongRange((String)UID_FIELD, (Long)MessageUid.MIN_VALUE.asLong(), (Long)MessageUid.MAX_VALUE.asLong(), (boolean)true, (boolean)true);
    }

    public void delete(MailboxSession session, org.apache.james.mailbox.store.mail.model.Mailbox mailbox, List<MessageUid> expungedUids) throws MailboxException {
        List messageRanges = MessageRange.toRanges(expungedUids);
        for (MessageRange messageRange : messageRanges) {
            this.delete(mailbox, messageRange);
        }
    }

    public void deleteAll(MailboxSession session, org.apache.james.mailbox.store.mail.model.Mailbox mailbox) throws MailboxException {
        this.delete(mailbox, MessageRange.all());
    }

    public void delete(org.apache.james.mailbox.store.mail.model.Mailbox mailbox, MessageRange range) throws MailboxException {
        BooleanQuery query = new BooleanQuery();
        query.add((Query)new TermQuery(new Term(MAILBOX_ID_FIELD, mailbox.getMailboxId().serialize())), BooleanClause.Occur.MUST);
        query.add(this.createQuery(range), BooleanClause.Occur.MUST);
        try {
            this.writer.deleteDocuments((Query)query);
        }
        catch (IOException e) {
            throw new MailboxException("Unable to delete message from index", (Throwable)e);
        }
    }

    static {
        Calendar cal = Calendar.getInstance();
        cal.set(9999, 11, 31);
        MAX_DATE = cal.getTime();
        cal.set(0, 0, 1);
        MIN_DATE = cal.getTime();
        UID_SORT = new SortField(UID_FIELD, 6);
        UID_SORT_REVERSE = new SortField(UID_FIELD, 6, true);
        SIZE_SORT = new SortField(SIZE_FIELD, 6);
        SIZE_SORT_REVERSE = new SortField(SIZE_FIELD, 6, true);
        FIRST_CC_MAILBOX_SORT = new SortField(FIRST_CC_MAILBOX_NAME_FIELD, 3);
        FIRST_CC_MAILBOX_SORT_REVERSE = new SortField(FIRST_CC_MAILBOX_NAME_FIELD, 3, true);
        FIRST_TO_MAILBOX_SORT = new SortField(FIRST_TO_MAILBOX_NAME_FIELD, 3);
        FIRST_TO_MAILBOX_SORT_REVERSE = new SortField(FIRST_TO_MAILBOX_NAME_FIELD, 3, true);
        FIRST_FROM_MAILBOX_SORT = new SortField(FIRST_FROM_MAILBOX_NAME_FIELD, 3);
        FIRST_FROM_MAILBOX_SORT_REVERSE = new SortField(FIRST_FROM_MAILBOX_NAME_FIELD, 3, true);
        ARRIVAL_MAILBOX_SORT = new SortField(INTERNAL_DATE_FIELD_MILLISECOND_RESOLUTION, 6);
        ARRIVAL_MAILBOX_SORT_REVERSE = new SortField(INTERNAL_DATE_FIELD_MILLISECOND_RESOLUTION, 6, true);
        BASE_SUBJECT_SORT = new SortField(BASE_SUBJECT_FIELD, 3);
        BASE_SUBJECT_SORT_REVERSE = new SortField(BASE_SUBJECT_FIELD, 3, true);
        SENT_DATE_SORT = new SortField(SENT_DATE_SORT_FIELD_MILLISECOND_RESOLUTION, 6);
        SENT_DATE_SORT_REVERSE = new SortField(SENT_DATE_SORT_FIELD_MILLISECOND_RESOLUTION, 6, true);
        FIRST_TO_MAILBOX_DISPLAY_SORT = new SortField(FIRST_TO_MAILBOX_DISPLAY_FIELD, 3);
        FIRST_TO_MAILBOX_DISPLAY_SORT_REVERSE = new SortField(FIRST_TO_MAILBOX_DISPLAY_FIELD, 3, true);
        FIRST_FROM_MAILBOX_DISPLAY_SORT = new SortField(FIRST_FROM_MAILBOX_DISPLAY_FIELD, 3);
        FIRST_FROM_MAILBOX_DISPLAY_SORT_REVERSE = new SortField(FIRST_FROM_MAILBOX_DISPLAY_FIELD, 3, true);
    }
}

