/*
 * Decompiled with CFR 0.152.
 */
package org.apache.ratis.util;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Consumer;
import java.util.function.Supplier;
import org.apache.ratis.util.LeakDetector;
import org.apache.ratis.util.MemoizedSupplier;
import org.apache.ratis.util.Preconditions;
import org.apache.ratis.util.ReferenceCountedObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class ReferenceCountedLeakDetector {
    private static final Logger LOG = LoggerFactory.getLogger(ReferenceCountedLeakDetector.class);
    private static final AtomicReference<Mode> FACTORY = new AtomicReference<Mode>(Mode.NONE);
    private static final Supplier<LeakDetector> SUPPLIER = MemoizedSupplier.valueOf(() -> new LeakDetector(FACTORY.get().name()).start());

    static Factory getFactory() {
        return FACTORY.get();
    }

    public static LeakDetector getLeakDetector() {
        return SUPPLIER.get();
    }

    private ReferenceCountedLeakDetector() {
    }

    public static synchronized void enable(boolean advanced) {
        FACTORY.set(advanced ? Mode.ADVANCED : Mode.SIMPLE);
    }

    private static class AdvancedTracing<T>
    extends SimpleTracing<T> {
        private final List<TraceInfo> traceInfos = new ArrayList<TraceInfo>();
        private TraceInfo previous;

        AdvancedTracing(T value, Runnable retainMethod, Consumer<Boolean> releaseMethod, LeakDetector leakDetector) {
            super(value, retainMethod, releaseMethod, leakDetector);
            this.addTraceInfo(Op.CREATION, -1);
        }

        private synchronized TraceInfo addTraceInfo(Op op, int previousRefCount) {
            TraceInfo current = new TraceInfo(this.traceInfos.size(), op, this.previous, previousRefCount);
            this.traceInfos.add(current);
            this.previous = current;
            return current;
        }

        @Override
        public synchronized T retain() {
            int previousRefCount = this.getCount();
            Object retained = super.retain();
            TraceInfo info = this.addTraceInfo(Op.RETAIN, previousRefCount);
            Preconditions.assertSame(this.getCount(), info.counts.refCount, "refCount");
            return retained;
        }

        @Override
        public synchronized boolean release() {
            int previousRefCount = this.getCount();
            boolean released = super.release();
            TraceInfo info = this.addTraceInfo(Op.RELEASE, previousRefCount);
            int count = this.getCount();
            int expected = count == -1 ? 0 : count;
            Preconditions.assertSame(expected, info.counts.refCount, "refCount");
            return released;
        }

        @Override
        synchronized String getTraceString(int count) {
            return super.getTraceString(count) + this.getTraceInfosString();
        }

        private String getTraceInfosString() {
            int n = this.traceInfos.size();
            StringBuilder b = new StringBuilder(n << 10).append(" #TraceInfos=").append(n);
            TraceInfo last = null;
            for (TraceInfo info : this.traceInfos) {
                info.appendTo(b.append("\n"));
                last = info;
            }
            TraceInfo current = new TraceInfo(n, Op.CURRENT, last, this.getCount());
            current.appendTo(b.append("\n"));
            return b.toString();
        }

        static class TraceInfo {
            private final int id;
            private final Op op;
            private final int previousRefCount;
            private final Counts counts;
            private final String threadInfo;
            private final StackTraceElement[] stackTraces;
            private final int newTraceElementIndex;

            TraceInfo(int id, Op op, TraceInfo previous, int previousRefCount) {
                this.id = id;
                this.op = op;
                this.previousRefCount = previousRefCount;
                this.counts = previous == null ? new Counts() : (op == Op.CURRENT ? previous.counts : new Counts(op, previous.counts));
                Thread thread = Thread.currentThread();
                this.threadInfo = "Thread_" + thread.getId() + ":" + thread.getName();
                this.stackTraces = thread.getStackTrace();
                this.newTraceElementIndex = previous == null ? this.stackTraces.length - 1 : TraceInfo.findFirstUnequalFromTail(this.stackTraces, previous.stackTraces);
            }

            static <T> int findFirstUnequalFromTail(T[] current, T[] previous) {
                int c = current.length == 0 ? 0 : current.length - 1;
                int p = previous.length - 1;
                while (p >= 0) {
                    if (!previous[p].equals(current[c])) {
                        return c;
                    }
                    --p;
                    --c;
                }
                return -1;
            }

            private StringBuilder appendTo(StringBuilder b) {
                int line;
                b.append((Object)this.op).append("_").append(this.id).append(": previousRefCount=").append(this.previousRefCount).append(", ").append(this.counts).append(", ").append(this.threadInfo).append("\n");
                int n = this.newTraceElementIndex + 1;
                for (line = 3; line <= n && line < this.stackTraces.length; ++line) {
                    b.append("    ").append(this.stackTraces[line]).append("\n");
                }
                if (line < this.stackTraces.length) {
                    b.append("    ...\n");
                }
                return b;
            }

            public String toString() {
                return this.appendTo(new StringBuilder()).toString();
            }
        }

        static class Counts {
            private final int refCount;
            private final int retainCount;
            private final int releaseCount;

            Counts() {
                this.refCount = 0;
                this.retainCount = 0;
                this.releaseCount = 0;
            }

            Counts(Op op, Counts previous) {
                if (op == Op.RETAIN) {
                    this.refCount = previous.refCount + 1;
                    this.retainCount = previous.retainCount + 1;
                    this.releaseCount = previous.releaseCount;
                } else if (op == Op.RELEASE) {
                    this.refCount = previous.refCount - 1;
                    this.retainCount = previous.retainCount;
                    this.releaseCount = previous.releaseCount + 1;
                } else {
                    throw new IllegalStateException("Unexpected op: " + (Object)((Object)op));
                }
            }

            public String toString() {
                return "refCount=" + this.refCount + ", retainCount=" + this.retainCount + ", releaseCount=" + this.releaseCount;
            }
        }

        static enum Op {
            CREATION,
            RETAIN,
            RELEASE,
            CURRENT;

        }
    }

    private static class SimpleTracing<T>
    extends Impl<T> {
        private final LeakDetector leakDetector;
        private final Class<?> valueClass;
        private String valueString = null;
        private Runnable removeMethod = null;

        SimpleTracing(T value, Runnable retainMethod, Consumer<Boolean> releaseMethod, LeakDetector leakDetector) {
            super(value, retainMethod, releaseMethod);
            this.valueClass = value.getClass();
            this.leakDetector = leakDetector;
        }

        String getTraceString(int count) {
            return "(" + this.valueClass + ", count=" + count + ", value=" + this.valueString + ")";
        }

        String logLeakMessage() {
            int count = this.getCount();
            if (count == 0) {
                return null;
            }
            String message = "LEAK: " + this.getTraceString(count);
            LOG.warn(message);
            return message;
        }

        @Override
        public synchronized T get() {
            try {
                return (T)super.get();
            }
            catch (Exception e) {
                throw new IllegalStateException("Failed to get: " + this.getTraceString(this.getCount()), e);
            }
        }

        @Override
        public synchronized T retain() {
            Object value;
            try {
                value = super.retain();
            }
            catch (Exception e) {
                throw new IllegalStateException("Failed to retain: " + this.getTraceString(this.getCount()), e);
            }
            if (this.getCount() == 1) {
                this.removeMethod = this.leakDetector.track(this, this::logLeakMessage);
                this.valueString = value.toString();
            }
            return (T)value;
        }

        @Override
        public synchronized boolean release() {
            boolean released;
            try {
                released = super.release();
            }
            catch (Exception e) {
                throw new IllegalStateException("Failed to release: " + this.getTraceString(this.getCount()), e);
            }
            if (released) {
                Preconditions.assertNotNull(this.removeMethod, () -> "Not yet retained (removeMethod == null): " + this.valueClass);
                this.removeMethod.run();
            }
            return released;
        }
    }

    private static class Impl<V>
    implements ReferenceCountedObject<V> {
        private final AtomicInteger count;
        private final V value;
        private final Runnable retainMethod;
        private final Consumer<Boolean> releaseMethod;

        Impl(V value, Runnable retainMethod, Consumer<Boolean> releaseMethod) {
            this.value = value;
            this.retainMethod = retainMethod;
            this.releaseMethod = releaseMethod;
            this.count = new AtomicInteger();
        }

        @Override
        public V get() {
            int previous = this.count.get();
            if (previous < 0) {
                throw new IllegalStateException("Failed to get: object has already been completely released.");
            }
            if (previous == 0) {
                throw new IllegalStateException("Failed to get: object has not yet been retained.");
            }
            return this.value;
        }

        final int getCount() {
            return this.count.get();
        }

        @Override
        public V retain() {
            if (this.count.getAndUpdate(n -> n < 0 ? n : n + 1) < 0) {
                throw new IllegalStateException("Failed to retain: object has already been completely released.");
            }
            this.retainMethod.run();
            return this.value;
        }

        @Override
        public boolean release() {
            int previous = this.count.getAndUpdate(n -> n <= 1 ? -1 : n - 1);
            if (previous < 0) {
                throw new IllegalStateException("Failed to release: object has already been completely released.");
            }
            if (previous == 0) {
                throw new IllegalStateException("Failed to release: object has not yet been retained.");
            }
            boolean completedReleased = previous == 1;
            this.releaseMethod.accept(completedReleased);
            return completedReleased;
        }
    }

    private static enum Mode implements Factory
    {
        NONE{

            @Override
            public <V> ReferenceCountedObject<V> create(V value, Runnable retainMethod, Consumer<Boolean> releaseMethod) {
                return new Impl<V>(value, retainMethod, releaseMethod);
            }
        }
        ,
        SIMPLE{

            @Override
            public <V> ReferenceCountedObject<V> create(V value, Runnable retainMethod, Consumer<Boolean> releaseMethod) {
                return new SimpleTracing<V>(value, retainMethod, releaseMethod, ReferenceCountedLeakDetector.getLeakDetector());
            }
        }
        ,
        ADVANCED{

            @Override
            public <V> ReferenceCountedObject<V> create(V value, Runnable retainMethod, Consumer<Boolean> releaseMethod) {
                return new AdvancedTracing<V>(value, retainMethod, releaseMethod, ReferenceCountedLeakDetector.getLeakDetector());
            }
        };

    }

    static interface Factory {
        public <V> ReferenceCountedObject<V> create(V var1, Runnable var2, Consumer<Boolean> var3);
    }
}

