/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iotdb.session.subscription.consumer;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.SortedMap;
import java.util.concurrent.ConcurrentSkipListMap;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.stream.Collectors;
import org.apache.iotdb.common.rpc.thrift.TEndPoint;
import org.apache.iotdb.rpc.IoTDBConnectionException;
import org.apache.iotdb.rpc.subscription.exception.SubscriptionConnectionException;
import org.apache.iotdb.rpc.subscription.exception.SubscriptionException;
import org.apache.iotdb.session.subscription.consumer.SubscriptionConsumer;
import org.apache.iotdb.session.subscription.consumer.SubscriptionProvider;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

final class SubscriptionProviders {
    private static final Logger LOGGER = LoggerFactory.getLogger(SubscriptionProviders.class);
    private final SortedMap<Integer, SubscriptionProvider> subscriptionProviders = new ConcurrentSkipListMap<Integer, SubscriptionProvider>();
    private int nextDataNodeId = -1;
    private final ReentrantReadWriteLock subscriptionProvidersLock = new ReentrantReadWriteLock(true);
    private final Set<TEndPoint> initialEndpoints;

    SubscriptionProviders(Set<TEndPoint> initialEndpoints) {
        this.initialEndpoints = initialEndpoints;
    }

    void acquireReadLock() {
        this.subscriptionProvidersLock.readLock().lock();
    }

    void releaseReadLock() {
        this.subscriptionProvidersLock.readLock().unlock();
    }

    void acquireWriteLock() {
        this.subscriptionProvidersLock.writeLock().lock();
    }

    void releaseWriteLock() {
        this.subscriptionProvidersLock.writeLock().unlock();
    }

    void openProviders(SubscriptionConsumer consumer) throws SubscriptionException {
        this.closeProviders();
        for (TEndPoint endPoint : this.initialEndpoints) {
            Map<Integer, TEndPoint> allEndPoints;
            SubscriptionProvider defaultProvider;
            try {
                defaultProvider = consumer.constructProviderAndHandshake(endPoint);
            }
            catch (Exception e) {
                LOGGER.warn("Failed to create connection with {}", (Object)endPoint, (Object)e);
                continue;
            }
            int defaultDataNodeId = defaultProvider.getDataNodeId();
            this.addProvider(defaultDataNodeId, defaultProvider);
            try {
                allEndPoints = defaultProvider.getSessionConnection().fetchAllEndPoints();
            }
            catch (Exception e) {
                LOGGER.warn("Failed to fetch all endpoints from {}, will retry later...", (Object)endPoint, (Object)e);
                break;
            }
            for (Map.Entry<Integer, TEndPoint> entry : allEndPoints.entrySet()) {
                SubscriptionProvider provider;
                if (defaultDataNodeId == entry.getKey()) continue;
                try {
                    provider = consumer.constructProviderAndHandshake(entry.getValue());
                }
                catch (Exception e) {
                    LOGGER.warn("Failed to create connection with {}, will retry later...", (Object)entry.getValue(), (Object)e);
                    continue;
                }
                this.addProvider(entry.getKey(), provider);
            }
        }
        if (this.hasNoAvailableProviders()) {
            throw new SubscriptionConnectionException(String.format("Cluster has no available subscription providers to connect with initial endpoints %s", this.initialEndpoints));
        }
        this.nextDataNodeId = this.subscriptionProviders.firstKey();
    }

    void closeProviders() {
        for (SubscriptionProvider provider : this.getAllProviders()) {
            try {
                provider.close();
            }
            catch (Exception e) {
                LOGGER.warn(e.getMessage());
            }
        }
        this.subscriptionProviders.clear();
    }

    void addProvider(int dataNodeId, SubscriptionProvider provider) {
        LOGGER.info("add new subscription provider {}", (Object)provider);
        this.subscriptionProviders.put(dataNodeId, provider);
    }

    void closeAndRemoveProvider(int dataNodeId) throws SubscriptionException, IoTDBConnectionException {
        if (!this.containsProvider(dataNodeId)) {
            return;
        }
        SubscriptionProvider provider = (SubscriptionProvider)this.subscriptionProviders.get(dataNodeId);
        try {
            provider.close();
        }
        finally {
            LOGGER.info("close and remove stale subscription provider {}", (Object)provider);
            this.subscriptionProviders.remove(dataNodeId);
        }
    }

    boolean hasNoProviders() {
        return this.subscriptionProviders.isEmpty();
    }

    List<SubscriptionProvider> getAllProviders() {
        return new ArrayList<SubscriptionProvider>(this.subscriptionProviders.values());
    }

    SubscriptionProvider getProvider(int dataNodeId) {
        return this.containsProvider(dataNodeId) ? (SubscriptionProvider)this.subscriptionProviders.get(dataNodeId) : null;
    }

    boolean hasNoAvailableProviders() {
        return this.subscriptionProviders.values().stream().noneMatch(SubscriptionProvider::isAvailable);
    }

    boolean containsProvider(int dataNodeId) {
        return this.subscriptionProviders.containsKey(dataNodeId);
    }

    List<SubscriptionProvider> getAllAvailableProviders() {
        return this.subscriptionProviders.values().stream().filter(SubscriptionProvider::isAvailable).collect(Collectors.toList());
    }

    void updateNextDataNodeId() {
        SortedMap<Integer, SubscriptionProvider> subProviders = this.subscriptionProviders.tailMap(this.nextDataNodeId + 1);
        this.nextDataNodeId = subProviders.isEmpty() ? this.subscriptionProviders.firstKey() : subProviders.firstKey();
    }

    SubscriptionProvider getNextAvailableProvider() {
        if (this.hasNoAvailableProviders()) {
            return null;
        }
        SubscriptionProvider provider = this.getProvider(this.nextDataNodeId);
        while (Objects.isNull(provider) || !provider.isAvailable()) {
            this.updateNextDataNodeId();
            provider = this.getProvider(this.nextDataNodeId);
        }
        this.updateNextDataNodeId();
        return provider;
    }

    void heartbeat(SubscriptionConsumer consumer) {
        if (consumer.isClosed()) {
            return;
        }
        this.acquireWriteLock();
        try {
            this.heartbeatInternal(consumer);
        }
        finally {
            this.releaseWriteLock();
        }
    }

    private void heartbeatInternal(SubscriptionConsumer consumer) {
        for (SubscriptionProvider provider : this.getAllProviders()) {
            try {
                consumer.subscribedTopics = provider.heartbeat();
                provider.setAvailable();
            }
            catch (Exception e) {
                LOGGER.warn("something unexpected happened when sending heartbeat to subscription provider {}, set subscription provider unavailable", (Object)provider, (Object)e);
                provider.setUnavailable();
            }
        }
    }

    void sync(SubscriptionConsumer consumer) {
        if (consumer.isClosed()) {
            return;
        }
        this.acquireWriteLock();
        try {
            this.syncInternal(consumer);
        }
        finally {
            this.releaseWriteLock();
        }
    }

    private void syncInternal(SubscriptionConsumer consumer) {
        Map<Integer, TEndPoint> allEndPoints;
        if (this.hasNoAvailableProviders()) {
            try {
                this.openProviders(consumer);
            }
            catch (Exception e) {
                LOGGER.warn("something unexpected happened when syncing subscription endpoints...", (Throwable)e);
                return;
            }
        }
        try {
            allEndPoints = consumer.fetchAllEndPointsWithRedirection();
        }
        catch (Exception e) {
            LOGGER.warn("Failed to fetch all endpoints, will retry later...", (Throwable)e);
            return;
        }
        for (Map.Entry<Integer, TEndPoint> entry : allEndPoints.entrySet()) {
            SubscriptionProvider provider = this.getProvider(entry.getKey());
            if (Objects.isNull(provider)) {
                SubscriptionProvider newProvider;
                TEndPoint endPoint = entry.getValue();
                try {
                    newProvider = consumer.constructProviderAndHandshake(endPoint);
                }
                catch (Exception e) {
                    LOGGER.warn("Failed to create connection with endpoint {}, will retry later...", (Object)endPoint, (Object)e);
                    continue;
                }
                this.addProvider(entry.getKey(), newProvider);
                continue;
            }
            try {
                consumer.subscribedTopics = provider.heartbeat();
                provider.setAvailable();
            }
            catch (Exception e) {
                LOGGER.warn("something unexpected happened when sending heartbeat to subscription provider {}, set subscription provider unavailable", (Object)provider, (Object)e);
                provider.setUnavailable();
            }
            if (provider.isAvailable()) continue;
            try {
                this.closeAndRemoveProvider(entry.getKey());
            }
            catch (Exception e) {
                LOGGER.warn("Exception occurred when closing and removing subscription provider with data node id {}", (Object)entry.getKey(), (Object)e);
            }
        }
        for (SubscriptionProvider provider : this.getAllProviders()) {
            int dataNodeId = provider.getDataNodeId();
            if (allEndPoints.containsKey(dataNodeId)) continue;
            try {
                this.closeAndRemoveProvider(dataNodeId);
            }
            catch (Exception e) {
                LOGGER.warn("Exception occurred when closing and removing subscription provider with data node id {}", (Object)dataNodeId, (Object)e);
            }
        }
    }

    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("SubscriptionProviders{");
        for (Map.Entry<Integer, SubscriptionProvider> entry : this.subscriptionProviders.entrySet()) {
            sb.append(entry.getValue().toString()).append(", ");
        }
        if (!this.subscriptionProviders.isEmpty()) {
            sb.delete(sb.length() - 2, sb.length());
        }
        sb.append("}");
        return sb.toString();
    }
}

