/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iotdb.consensus.iot;

import com.google.common.collect.ImmutableList;
import java.io.DataOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.file.CopyOption;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.PriorityQueue;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.commons.io.FileUtils;
import org.apache.iotdb.common.rpc.thrift.TEndPoint;
import org.apache.iotdb.common.rpc.thrift.TSStatus;
import org.apache.iotdb.commons.client.IClientManager;
import org.apache.iotdb.commons.client.exception.ClientManagerException;
import org.apache.iotdb.commons.consensus.index.ComparableConsensusRequest;
import org.apache.iotdb.commons.consensus.index.ProgressIndex;
import org.apache.iotdb.commons.consensus.index.impl.IoTProgressIndex;
import org.apache.iotdb.commons.service.metric.MetricService;
import org.apache.iotdb.commons.service.metric.PerformanceOverviewMetrics;
import org.apache.iotdb.commons.utils.CommonDateTimeUtils;
import org.apache.iotdb.commons.utils.KillPoint.DataNodeKillPoints;
import org.apache.iotdb.commons.utils.KillPoint.KillPoint;
import org.apache.iotdb.consensus.IStateMachine;
import org.apache.iotdb.consensus.common.DataSet;
import org.apache.iotdb.consensus.common.Peer;
import org.apache.iotdb.consensus.common.request.DeserializedBatchIndexedConsensusRequest;
import org.apache.iotdb.consensus.common.request.IConsensusRequest;
import org.apache.iotdb.consensus.common.request.IndexedConsensusRequest;
import org.apache.iotdb.consensus.config.IoTConsensusConfig;
import org.apache.iotdb.consensus.exception.ConsensusGroupModifyPeerException;
import org.apache.iotdb.consensus.iot.IoTConsensusServerMetrics;
import org.apache.iotdb.consensus.iot.client.AsyncIoTConsensusServiceClient;
import org.apache.iotdb.consensus.iot.client.SyncIoTConsensusServiceClient;
import org.apache.iotdb.consensus.iot.log.ConsensusReqReader;
import org.apache.iotdb.consensus.iot.log.GetConsensusReqReaderPlan;
import org.apache.iotdb.consensus.iot.logdispatcher.LogDispatcher;
import org.apache.iotdb.consensus.iot.snapshot.IoTConsensusRateLimiter;
import org.apache.iotdb.consensus.iot.snapshot.SnapshotFragmentReader;
import org.apache.iotdb.consensus.iot.thrift.TActivatePeerReq;
import org.apache.iotdb.consensus.iot.thrift.TActivatePeerRes;
import org.apache.iotdb.consensus.iot.thrift.TBuildSyncLogChannelReq;
import org.apache.iotdb.consensus.iot.thrift.TBuildSyncLogChannelRes;
import org.apache.iotdb.consensus.iot.thrift.TCleanupTransferredSnapshotReq;
import org.apache.iotdb.consensus.iot.thrift.TCleanupTransferredSnapshotRes;
import org.apache.iotdb.consensus.iot.thrift.TInactivatePeerReq;
import org.apache.iotdb.consensus.iot.thrift.TInactivatePeerRes;
import org.apache.iotdb.consensus.iot.thrift.TRemoveSyncLogChannelReq;
import org.apache.iotdb.consensus.iot.thrift.TRemoveSyncLogChannelRes;
import org.apache.iotdb.consensus.iot.thrift.TSendSnapshotFragmentReq;
import org.apache.iotdb.consensus.iot.thrift.TSendSnapshotFragmentRes;
import org.apache.iotdb.consensus.iot.thrift.TTriggerSnapshotLoadReq;
import org.apache.iotdb.consensus.iot.thrift.TTriggerSnapshotLoadRes;
import org.apache.iotdb.consensus.iot.thrift.TWaitSyncLogCompleteReq;
import org.apache.iotdb.consensus.iot.thrift.TWaitSyncLogCompleteRes;
import org.apache.iotdb.metrics.metricsets.IMetricSet;
import org.apache.iotdb.rpc.RpcUtils;
import org.apache.iotdb.rpc.TSStatusCode;
import org.apache.thrift.TException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class IoTConsensusServerImpl {
    private static final String CONFIGURATION_FILE_NAME = "configuration.dat";
    private static final String CONFIGURATION_TMP_FILE_NAME = "configuration.dat.tmp";
    public static final String SNAPSHOT_DIR_NAME = "snapshot";
    private static final Pattern SNAPSHOT_INDEX_PATTEN = Pattern.compile(".*[^\\d](?=(\\d+))");
    private static final PerformanceOverviewMetrics PERFORMANCE_OVERVIEW_METRICS = PerformanceOverviewMetrics.getInstance();
    private final Logger logger = LoggerFactory.getLogger(IoTConsensusServerImpl.class);
    private final Peer thisNode;
    private final IStateMachine stateMachine;
    private final ConcurrentHashMap<Integer, SyncLogCacheQueue> cacheQueueMap;
    private final Lock stateMachineLock = new ReentrantLock();
    private final Condition stateMachineCondition = this.stateMachineLock.newCondition();
    private final String storageDir;
    private final List<Peer> configuration;
    private final AtomicLong searchIndex;
    private final LogDispatcher logDispatcher;
    private final IoTConsensusConfig config;
    private final ConsensusReqReader consensusReqReader;
    private volatile boolean active = true;
    private String newSnapshotDirName;
    private final IClientManager<TEndPoint, SyncIoTConsensusServiceClient> syncClientManager;
    private final IoTConsensusServerMetrics ioTConsensusServerMetrics;
    private final String consensusGroupId;
    private final ScheduledExecutorService backgroundTaskService;
    private final IoTConsensusRateLimiter ioTConsensusRateLimiter = IoTConsensusRateLimiter.getInstance();
    private volatile long lastPinnedSearchIndexForMigration = -1L;

    public IoTConsensusServerImpl(String storageDir, Peer thisNode, List<Peer> configuration, IStateMachine stateMachine, ScheduledExecutorService backgroundTaskService, IClientManager<TEndPoint, AsyncIoTConsensusServiceClient> clientManager, IClientManager<TEndPoint, SyncIoTConsensusServiceClient> syncClientManager, IoTConsensusConfig config) {
        this.storageDir = storageDir;
        this.thisNode = thisNode;
        this.stateMachine = stateMachine;
        this.cacheQueueMap = new ConcurrentHashMap();
        this.syncClientManager = syncClientManager;
        this.configuration = configuration;
        if (configuration.isEmpty()) {
            this.recoverConfiguration();
        } else {
            this.persistConfiguration();
        }
        this.backgroundTaskService = backgroundTaskService;
        this.config = config;
        this.consensusGroupId = thisNode.getGroupId().toString();
        this.consensusReqReader = (ConsensusReqReader)((Object)stateMachine.read(new GetConsensusReqReaderPlan()));
        this.searchIndex = new AtomicLong(this.consensusReqReader.getCurrentSearchIndex());
        this.ioTConsensusServerMetrics = new IoTConsensusServerMetrics(this);
        this.logDispatcher = new LogDispatcher(this, clientManager);
        this.checkAndUpdateSafeDeletedSearchIndex();
        this.checkAndUpdateSearchIndex();
    }

    public IStateMachine getStateMachine() {
        return this.stateMachine;
    }

    public void start() {
        MetricService.getInstance().addMetricSet((IMetricSet)this.ioTConsensusServerMetrics);
        this.stateMachine.start();
        this.logDispatcher.start();
    }

    public void stop() {
        this.logDispatcher.stop();
        this.stateMachine.stop();
        MetricService.getInstance().removeMetricSet((IMetricSet)this.ioTConsensusServerMetrics);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public TSStatus write(IConsensusRequest request) {
        long consensusWriteStartTime = System.nanoTime();
        this.stateMachineLock.lock();
        try {
            AtomicLong atomicLong;
            long getStateMachineLockTime = System.nanoTime();
            this.ioTConsensusServerMetrics.recordGetStateMachineLockTime(getStateMachineLockTime - consensusWriteStartTime);
            if (this.needBlockWrite()) {
                this.logger.info("[Throttle Down] index:{}, safeIndex:{}", (Object)this.getSearchIndex(), (Object)this.getMinSyncIndex());
                try {
                    boolean timeout;
                    boolean bl = timeout = !this.stateMachineCondition.await(this.config.getReplication().getThrottleTimeOutMs(), TimeUnit.MILLISECONDS);
                    if (timeout) {
                        TSStatus tSStatus = RpcUtils.getStatus((TSStatusCode)TSStatusCode.WRITE_PROCESS_REJECT, (String)String.format("The write is rejected because the wal directory size has reached the threshold %d bytes. You may need to adjust the flush policy of the storage storageengine or the IoTConsensus synchronization parameter", this.config.getReplication().getWalThrottleThreshold()));
                        return tSStatus;
                    }
                }
                catch (InterruptedException e) {
                    this.logger.error("Failed to throttle down because ", (Throwable)e);
                    Thread.currentThread().interrupt();
                }
            }
            long writeToStateMachineStartTime = System.nanoTime();
            this.ioTConsensusServerMetrics.recordCheckingBeforeWriteTime(writeToStateMachineStartTime - getStateMachineLockTime);
            IndexedConsensusRequest indexedConsensusRequest = this.buildIndexedConsensusRequestForLocalRequest(request);
            if (indexedConsensusRequest.getSearchIndex() % 100000L == 0L) {
                this.logger.info("DataRegion[{}]: index after build: safeIndex:{}, searchIndex: {}", new Object[]{this.thisNode.getGroupId(), this.getMinSyncIndex(), indexedConsensusRequest.getSearchIndex()});
            }
            IConsensusRequest planNode = this.stateMachine.deserializeRequest(indexedConsensusRequest);
            long startWriteTime = System.nanoTime();
            TSStatus result = this.stateMachine.write(planNode);
            PERFORMANCE_OVERVIEW_METRICS.recordEngineCost(System.nanoTime() - startWriteTime);
            long writeToStateMachineEndTime = System.nanoTime();
            this.ioTConsensusServerMetrics.recordWriteStateMachineTime(writeToStateMachineEndTime - writeToStateMachineStartTime);
            if (result.getCode() == TSStatusCode.SUCCESS_STATUS.getStatusCode()) {
                atomicLong = this.searchIndex;
                synchronized (atomicLong) {
                    this.logDispatcher.offer(indexedConsensusRequest);
                    this.searchIndex.incrementAndGet();
                }
                this.ioTConsensusServerMetrics.recordOfferRequestToQueueTime(System.nanoTime() - writeToStateMachineEndTime);
            } else {
                this.logger.debug("{}: write operation failed. searchIndex: {}. Code: {}", new Object[]{this.thisNode.getGroupId(), indexedConsensusRequest.getSearchIndex(), result.getCode()});
            }
            this.ioTConsensusServerMetrics.recordConsensusWriteTime(System.nanoTime() - consensusWriteStartTime);
            atomicLong = result;
            return atomicLong;
        }
        finally {
            this.stateMachineLock.unlock();
        }
    }

    public DataSet read(IConsensusRequest request) {
        return this.stateMachine.read(request);
    }

    public void takeSnapshot() throws ConsensusGroupModifyPeerException {
        try {
            long newSnapshotIndex = this.getLatestSnapshotIndex() + 1L;
            this.newSnapshotDirName = String.format("%s_%s_%d", SNAPSHOT_DIR_NAME, this.thisNode.getGroupId().getId(), newSnapshotIndex);
            File snapshotDir = new File(this.storageDir, this.newSnapshotDirName);
            if (snapshotDir.exists()) {
                FileUtils.deleteDirectory((File)snapshotDir);
            }
            if (!snapshotDir.mkdirs()) {
                throw new ConsensusGroupModifyPeerException(String.format("%s: cannot mkdir for snapshot", this.thisNode.getGroupId()));
            }
            if (!this.stateMachine.takeSnapshot(snapshotDir)) {
                throw new ConsensusGroupModifyPeerException("unknown error when taking snapshot");
            }
            this.clearOldSnapshot();
        }
        catch (IOException e) {
            throw new ConsensusGroupModifyPeerException("error when taking snapshot", e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void transmitSnapshot(Peer targetPeer) throws ConsensusGroupModifyPeerException {
        File snapshotDir = new File(this.storageDir, this.newSnapshotDirName);
        List<Path> snapshotPaths = this.stateMachine.getSnapshotFiles(snapshotDir);
        AtomicLong snapshotSizeSumAtomic = new AtomicLong();
        StringBuilder allFilesStr = new StringBuilder();
        snapshotPaths.forEach(path -> {
            try {
                long fileSize = Files.size(path);
                snapshotSizeSumAtomic.addAndGet(fileSize);
                allFilesStr.append("\n").append(path).append(" ").append(org.apache.iotdb.commons.utils.FileUtils.humanReadableByteCountSI((long)fileSize));
            }
            catch (IOException e) {
                this.logger.error("[SNAPSHOT TRANSMISSION] Calculate snapshot file's size fail: {}", path, (Object)e);
            }
        });
        long snapshotSizeSum = snapshotSizeSumAtomic.get();
        long transitedSnapshotSizeSum = 0L;
        long transitedFilesNum = 0L;
        long startTime = System.nanoTime();
        this.logger.info("[SNAPSHOT TRANSMISSION] Start to transmit snapshots ({} files, total size {}) from dir {}", new Object[]{snapshotPaths.size(), org.apache.iotdb.commons.utils.FileUtils.humanReadableByteCountSI((long)snapshotSizeSum), snapshotDir});
        this.logger.info("[SNAPSHOT TRANSMISSION] All the files below shell be transmitted: {}", (Object)allFilesStr);
        try (SyncIoTConsensusServiceClient client = (SyncIoTConsensusServiceClient)this.syncClientManager.borrowClient((Object)targetPeer.getEndpoint());){
            for (Path path2 : snapshotPaths) {
                try (SnapshotFragmentReader reader = new SnapshotFragmentReader(this.newSnapshotDirName, path2);){
                    while (reader.hasNext()) {
                        TSendSnapshotFragmentReq req = reader.next().toTSendSnapshotFragmentReq();
                        req.setConsensusGroupId(targetPeer.getGroupId().convertToTConsensusGroupId());
                        this.ioTConsensusRateLimiter.acquireTransitDataSizeWithRateLimiter(req.getChunkLength());
                        TSendSnapshotFragmentRes res = client.sendSnapshotFragment(req);
                        if (this.isSuccess(res.getStatus())) continue;
                        throw new ConsensusGroupModifyPeerException(String.format("[SNAPSHOT TRANSMISSION] Error when transmitting snapshot fragment to %s", targetPeer));
                    }
                    this.logger.info("[SNAPSHOT TRANSMISSION] The overall progress for dir {}: files {}/{} done, size {}/{} done, time {} passed. File {} done.", new Object[]{this.newSnapshotDirName, ++transitedFilesNum, snapshotPaths.size(), org.apache.iotdb.commons.utils.FileUtils.humanReadableByteCountSI((long)(transitedSnapshotSizeSum += reader.getTotalReadSize())), org.apache.iotdb.commons.utils.FileUtils.humanReadableByteCountSI((long)snapshotSizeSum), CommonDateTimeUtils.convertMillisecondToDurationStr((long)((System.nanoTime() - startTime) / 1000000L)), path2});
                }
            }
        }
        catch (Exception e) {
            throw new ConsensusGroupModifyPeerException(String.format("[SNAPSHOT TRANSMISSION] Error when send snapshot file to %s", targetPeer), e);
        }
        this.logger.info("[SNAPSHOT TRANSMISSION] After {}, successfully transmit all snapshots from dir {}", (Object)CommonDateTimeUtils.convertMillisecondToDurationStr((long)((System.nanoTime() - startTime) / 1000000L)), (Object)snapshotDir);
    }

    public void receiveSnapshotFragment(String snapshotId, String originalFilePath, ByteBuffer fileChunk) throws ConsensusGroupModifyPeerException {
        try {
            String targetFilePath = this.calculateSnapshotPath(snapshotId, originalFilePath);
            File targetFile = new File(this.storageDir, targetFilePath);
            Path parentDir = Paths.get(targetFile.getParent(), new String[0]);
            if (!Files.exists(parentDir, new LinkOption[0])) {
                Files.createDirectories(parentDir, new FileAttribute[0]);
            }
            try (FileOutputStream fos = new FileOutputStream(targetFile.getAbsolutePath(), true);
                 FileChannel channel = fos.getChannel();){
                channel.write(fileChunk.slice());
            }
        }
        catch (IOException e) {
            throw new ConsensusGroupModifyPeerException(String.format("error when receiving snapshot %s", snapshotId), e);
        }
    }

    private String calculateSnapshotPath(String snapshotId, String originalFilePath) throws ConsensusGroupModifyPeerException {
        if (!originalFilePath.contains(snapshotId)) {
            throw new ConsensusGroupModifyPeerException(String.format("invalid snapshot file. snapshotId: %s, filePath: %s", snapshotId, originalFilePath));
        }
        return originalFilePath.substring(originalFilePath.indexOf(snapshotId));
    }

    private long getLatestSnapshotIndex() {
        long snapShotIndex = 0L;
        File directory = new File(this.storageDir);
        File[] versionFiles = directory.listFiles((dir, name) -> name.startsWith(SNAPSHOT_DIR_NAME));
        if (versionFiles == null || versionFiles.length == 0) {
            return snapShotIndex;
        }
        for (File file : versionFiles) {
            snapShotIndex = Math.max(snapShotIndex, Long.parseLong(SNAPSHOT_INDEX_PATTEN.matcher(file.getName()).replaceAll("")));
        }
        return snapShotIndex;
    }

    private void clearOldSnapshot() {
        File directory = new File(this.storageDir);
        File[] versionFiles = directory.listFiles((dir, name) -> name.startsWith(SNAPSHOT_DIR_NAME));
        if (versionFiles == null || versionFiles.length == 0) {
            this.logger.error("Can not find any snapshot dir after build a new snapshot for group {}", (Object)this.thisNode.getGroupId());
            return;
        }
        for (File file : versionFiles) {
            if (file.getName().equals(this.newSnapshotDirName)) continue;
            try {
                FileUtils.deleteDirectory((File)file);
            }
            catch (IOException e) {
                this.logger.error("Delete old snapshot dir {} failed", (Object)file.getAbsolutePath(), (Object)e);
            }
        }
    }

    public void loadSnapshot(String snapshotId) {
        this.stateMachine.loadSnapshot(new File(this.storageDir, snapshotId));
    }

    public void inactivePeer(Peer peer, boolean forDeletionPurpose) throws ConsensusGroupModifyPeerException {
        try (SyncIoTConsensusServiceClient client = (SyncIoTConsensusServiceClient)this.syncClientManager.borrowClient((Object)peer.getEndpoint());){
            try {
                TInactivatePeerRes res = client.inactivatePeer(new TInactivatePeerReq(peer.getGroupId().convertToTConsensusGroupId()).setForDeletionPurpose(forDeletionPurpose));
                if (!this.isSuccess(res.status)) {
                    throw new ConsensusGroupModifyPeerException(String.format("error when inactivating %s. %s", peer, res.getStatus()));
                }
            }
            catch (Exception e) {
                throw new ConsensusGroupModifyPeerException(String.format("error when inactivating %s", peer), e);
            }
        }
        catch (ClientManagerException e) {
            throw new ConsensusGroupModifyPeerException(e);
        }
    }

    public void triggerSnapshotLoad(Peer peer) throws ConsensusGroupModifyPeerException {
        try (SyncIoTConsensusServiceClient client = (SyncIoTConsensusServiceClient)this.syncClientManager.borrowClient((Object)peer.getEndpoint());){
            TTriggerSnapshotLoadRes res = client.triggerSnapshotLoad(new TTriggerSnapshotLoadReq(this.thisNode.getGroupId().convertToTConsensusGroupId(), this.newSnapshotDirName));
            if (!this.isSuccess(res.status)) {
                throw new ConsensusGroupModifyPeerException(String.format("error when triggering snapshot load %s. %s", peer, res.getStatus()));
            }
        }
        catch (Exception e) {
            throw new ConsensusGroupModifyPeerException(String.format("error when activating %s", peer), e);
        }
    }

    public void activePeer(Peer peer) throws ConsensusGroupModifyPeerException {
        try (SyncIoTConsensusServiceClient client = (SyncIoTConsensusServiceClient)this.syncClientManager.borrowClient((Object)peer.getEndpoint());){
            TActivatePeerRes res = client.activatePeer(new TActivatePeerReq(peer.getGroupId().convertToTConsensusGroupId()));
            if (!this.isSuccess(res.status)) {
                throw new ConsensusGroupModifyPeerException(String.format("error when activating %s. %s", peer, res.getStatus()));
            }
        }
        catch (Exception e) {
            throw new ConsensusGroupModifyPeerException(String.format("error when activating %s", peer), e);
        }
    }

    public void notifyPeersToBuildSyncLogChannel(Peer targetPeer) throws ConsensusGroupModifyPeerException {
        ArrayList<Peer> currentMembers = new ArrayList<Peer>(this.configuration);
        this.logger.info("[IoTConsensus] notify current peers to build sync log. group member: {}, target: {}", currentMembers, (Object)targetPeer);
        for (Peer peer : currentMembers) {
            this.logger.info("[IoTConsensus] build sync log channel from {}", (Object)peer);
            if (peer.equals(this.thisNode)) {
                this.buildSyncLogChannel(targetPeer, this.lastPinnedSearchIndexForMigration);
                continue;
            }
            try {
                SyncIoTConsensusServiceClient client = (SyncIoTConsensusServiceClient)this.syncClientManager.borrowClient((Object)peer.getEndpoint());
                try {
                    TBuildSyncLogChannelRes res = client.buildSyncLogChannel(new TBuildSyncLogChannelReq(targetPeer.getGroupId().convertToTConsensusGroupId(), targetPeer.getEndpoint(), targetPeer.getNodeId()));
                    if (this.isSuccess(res.status)) continue;
                    throw new ConsensusGroupModifyPeerException(String.format("build sync log channel failed from %s to %s", peer, targetPeer));
                }
                finally {
                    if (client == null) continue;
                    client.close();
                }
            }
            catch (Exception e) {
                this.logger.error("cannot notify {} to build sync log channel. Please check the status of this node manually", (Object)peer, (Object)e);
            }
        }
    }

    public void notifyPeersToRemoveSyncLogChannel(Peer targetPeer) throws ConsensusGroupModifyPeerException {
        ImmutableList currentMembers = ImmutableList.copyOf(this.configuration);
        this.removeSyncLogChannel(targetPeer);
        for (Peer peer : currentMembers) {
            if (peer.equals(targetPeer) || peer.equals(this.thisNode)) continue;
            try {
                SyncIoTConsensusServiceClient client = (SyncIoTConsensusServiceClient)this.syncClientManager.borrowClient((Object)peer.getEndpoint());
                try {
                    TRemoveSyncLogChannelRes res = client.removeSyncLogChannel(new TRemoveSyncLogChannelReq(targetPeer.getGroupId().convertToTConsensusGroupId(), targetPeer.getEndpoint(), targetPeer.getNodeId()));
                    if (this.isSuccess(res.status)) continue;
                    throw new ConsensusGroupModifyPeerException(String.format("remove sync log channel failed from %s to %s", peer, targetPeer));
                }
                finally {
                    if (client == null) continue;
                    client.close();
                }
            }
            catch (Exception e) {
                throw new ConsensusGroupModifyPeerException(String.format("error when removing sync log channel to %s", peer), e);
            }
        }
    }

    public void waitTargetPeerUntilSyncLogCompleted(Peer targetPeer) throws ConsensusGroupModifyPeerException {
        long checkIntervalInMs = 10000L;
        try (SyncIoTConsensusServiceClient client = (SyncIoTConsensusServiceClient)this.syncClientManager.borrowClient((Object)targetPeer.getEndpoint());){
            while (true) {
                TWaitSyncLogCompleteRes res = client.waitSyncLogComplete(new TWaitSyncLogCompleteReq(targetPeer.getGroupId().convertToTConsensusGroupId()));
                if (res.complete) {
                    this.logger.info("[WAIT LOG SYNC] {} SyncLog is completed. TargetIndex: {}, CurrentSyncIndex: {}", new Object[]{targetPeer, res.searchIndex, res.safeIndex});
                    return;
                }
                this.logger.info("[WAIT LOG SYNC] {} SyncLog is still in progress. TargetIndex: {}, CurrentSyncIndex: {}", new Object[]{targetPeer, res.searchIndex, res.safeIndex});
                Thread.sleep(checkIntervalInMs);
            }
        }
        catch (ClientManagerException | TException e) {
            throw new ConsensusGroupModifyPeerException(String.format("error when waiting %s to complete SyncLog. %s", targetPeer, e.getMessage()), e);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new ConsensusGroupModifyPeerException(String.format("thread interrupted when waiting %s to complete SyncLog. %s", targetPeer, e.getMessage()), e);
        }
    }

    private boolean isSuccess(TSStatus status) {
        return status.getCode() == TSStatusCode.SUCCESS_STATUS.getStatusCode();
    }

    public void buildSyncLogChannel(Peer targetPeer) throws ConsensusGroupModifyPeerException {
        this.buildSyncLogChannel(targetPeer, this.getMinSyncIndex());
    }

    public void buildSyncLogChannel(Peer targetPeer, long initialSyncIndex) throws ConsensusGroupModifyPeerException {
        KillPoint.setKillPoint((Enum)DataNodeKillPoints.ORIGINAL_ADD_PEER_DONE);
        this.logger.info("[IoTConsensus] build sync log channel to {} with initialSyncIndex {}", (Object)targetPeer, (Object)initialSyncIndex);
        this.logDispatcher.addLogDispatcherThread(targetPeer, initialSyncIndex);
        this.configuration.add(targetPeer);
        this.persistConfiguration();
        this.logger.info("[IoTConsensus] persist new configuration: {}", this.configuration);
    }

    public void removeSyncLogChannel(Peer targetPeer) throws ConsensusGroupModifyPeerException {
        try {
            this.logDispatcher.removeLogDispatcherThread(targetPeer);
            this.logger.info("[IoTConsensus] log dispatcher to {} removed and cleanup", (Object)targetPeer);
            this.configuration.remove(targetPeer);
            this.checkAndUpdateSafeDeletedSearchIndex();
            this.persistConfiguration();
            this.logger.info("[IoTConsensus] configuration updated to {}", this.configuration);
        }
        catch (IOException e) {
            throw new ConsensusGroupModifyPeerException("error when remove LogDispatcherThread", e);
        }
    }

    public void persistConfiguration() {
        try {
            this.removeDuplicateConfiguration();
            this.renameTmpConfigurationFileToRemoveSuffix();
            this.serializeConfigurationAndFsyncToDisk();
            this.deleteConfiguration();
            this.renameTmpConfigurationFileToRemoveSuffix();
        }
        catch (IOException e) {
            this.logger.error("Unexpected error occurs when persisting configuration", (Throwable)e);
        }
    }

    public void recoverConfiguration() {
        try {
            Path tmpConfigurationPath = Paths.get(new File(this.storageDir, CONFIGURATION_TMP_FILE_NAME).getAbsolutePath(), new String[0]);
            Path configurationPath = Paths.get(new File(this.storageDir, CONFIGURATION_FILE_NAME).getAbsolutePath(), new String[0]);
            if (Files.exists(tmpConfigurationPath, new LinkOption[0])) {
                Files.deleteIfExists(configurationPath);
                Files.move(tmpConfigurationPath, configurationPath, new CopyOption[0]);
            }
            if (Files.exists(configurationPath, new LinkOption[0])) {
                this.recoverFromOldConfigurationFile(configurationPath);
            } else {
                Path dirPath = Paths.get(this.storageDir, new String[0]);
                List<Peer> tmpPeerList = this.getConfiguration(dirPath, CONFIGURATION_TMP_FILE_NAME);
                this.configuration.addAll(tmpPeerList);
                List<Peer> peerList = this.getConfiguration(dirPath, CONFIGURATION_FILE_NAME);
                for (Peer peer : peerList) {
                    if (this.configuration.contains(peer)) continue;
                    this.configuration.add(peer);
                }
                this.persistConfiguration();
            }
            this.logger.info("Recover IoTConsensus server Impl, configuration: {}", this.configuration);
        }
        catch (IOException e) {
            this.logger.error("Unexpected error occurs when recovering configuration", (Throwable)e);
        }
    }

    private void recoverFromOldConfigurationFile(Path oldConfigurationPath) throws IOException {
        ByteBuffer buffer = ByteBuffer.wrap(Files.readAllBytes(oldConfigurationPath));
        int size = buffer.getInt();
        for (int i = 0; i < size; ++i) {
            this.configuration.add(Peer.deserialize(buffer));
        }
        this.persistConfiguration();
    }

    public static String generateConfigurationDatFileName(int nodeId, String suffix) {
        return nodeId + "_" + suffix;
    }

    private List<Peer> getConfiguration(Path dirPath, String configurationFileName) throws IOException {
        Path[] files;
        ArrayList<Peer> tmpConfiguration = new ArrayList<Peer>();
        for (Path file : files = (Path[])Files.walk(dirPath, new FileVisitOption[0]).filter(x$0 -> Files.isRegularFile(x$0, new LinkOption[0])).filter(filePath -> filePath.getFileName().toString().contains(configurationFileName)).toArray(Path[]::new)) {
            ByteBuffer buffer = ByteBuffer.wrap(Files.readAllBytes(file));
            tmpConfiguration.add(Peer.deserialize(buffer));
        }
        return tmpConfiguration;
    }

    public IndexedConsensusRequest buildIndexedConsensusRequestForLocalRequest(IConsensusRequest request) {
        if (request instanceof ComparableConsensusRequest) {
            IoTProgressIndex iotProgressIndex = new IoTProgressIndex(Integer.valueOf(this.thisNode.getNodeId()), Long.valueOf(this.searchIndex.get() + 1L));
            ((ComparableConsensusRequest)request).setProgressIndex((ProgressIndex)iotProgressIndex);
        }
        return new IndexedConsensusRequest(this.searchIndex.get() + 1L, Collections.singletonList(request));
    }

    public IndexedConsensusRequest buildIndexedConsensusRequestForRemoteRequest(long syncIndex, List<IConsensusRequest> requests) {
        return new IndexedConsensusRequest(-1L, syncIndex, requests);
    }

    public long getMinSyncIndex() {
        return this.logDispatcher.getMinSyncIndex().orElseGet(this.searchIndex::get);
    }

    public long getMinFlushedSyncIndex() {
        return this.lastPinnedSearchIndexForMigration == -1L ? this.logDispatcher.getMinFlushedSyncIndex().orElseGet(this.searchIndex::get) : this.lastPinnedSearchIndexForMigration;
    }

    public String getStorageDir() {
        return this.storageDir;
    }

    public Peer getThisNode() {
        return this.thisNode;
    }

    public List<Peer> getConfiguration() {
        return this.configuration;
    }

    public long getSearchIndex() {
        return this.searchIndex.get();
    }

    public long getSyncLag() {
        long minSyncIndex = this.getMinSyncIndex();
        return this.getSearchIndex() - minSyncIndex;
    }

    public IoTConsensusConfig getConfig() {
        return this.config;
    }

    public long getLogEntriesFromWAL() {
        return this.logDispatcher.getLogEntriesFromWAL();
    }

    public long getLogEntriesFromQueue() {
        return this.logDispatcher.getLogEntriesFromQueue();
    }

    public boolean needBlockWrite() {
        return this.consensusReqReader.getTotalSize() > this.config.getReplication().getWalThrottleThreshold();
    }

    public boolean unblockWrite() {
        return this.consensusReqReader.getTotalSize() < this.config.getReplication().getWalThrottleThreshold();
    }

    public void signal() {
        this.stateMachineLock.lock();
        try {
            this.stateMachineCondition.signalAll();
        }
        finally {
            this.stateMachineLock.unlock();
        }
    }

    public AtomicLong getIndexObject() {
        return this.searchIndex;
    }

    public ScheduledExecutorService getBackgroundTaskService() {
        return this.backgroundTaskService;
    }

    public LogDispatcher getLogDispatcher() {
        return this.logDispatcher;
    }

    public IoTConsensusServerMetrics getIoTConsensusServerMetrics() {
        return this.ioTConsensusServerMetrics;
    }

    public boolean isReadOnly() {
        return this.stateMachine.isReadOnly();
    }

    public boolean isActive() {
        return this.active;
    }

    public void setActive(boolean active) {
        this.logger.info("set {} active status to {}", (Object)this.thisNode, (Object)active);
        this.active = active;
    }

    public void cleanupRemoteSnapshot(Peer targetPeer) throws ConsensusGroupModifyPeerException {
        try (SyncIoTConsensusServiceClient client = (SyncIoTConsensusServiceClient)this.syncClientManager.borrowClient((Object)targetPeer.getEndpoint());){
            TCleanupTransferredSnapshotReq req = new TCleanupTransferredSnapshotReq(targetPeer.getGroupId().convertToTConsensusGroupId(), this.newSnapshotDirName);
            TCleanupTransferredSnapshotRes res = client.cleanupTransferredSnapshot(req);
            if (!this.isSuccess(res.getStatus())) {
                throw new ConsensusGroupModifyPeerException(String.format("cleanup remote snapshot failed of %s ,status is %s", targetPeer, res.getStatus()));
            }
        }
        catch (Exception e) {
            throw new ConsensusGroupModifyPeerException(String.format("cleanup remote snapshot failed of %s", targetPeer), e);
        }
    }

    public void cleanupTransferredSnapshot(String snapshotId) throws ConsensusGroupModifyPeerException {
        File snapshotDir = new File(this.storageDir, snapshotId);
        if (snapshotDir.exists()) {
            try {
                FileUtils.deleteDirectory((File)snapshotDir);
            }
            catch (IOException e) {
                throw new ConsensusGroupModifyPeerException(e);
            }
        }
    }

    public void checkAndLockSafeDeletedSearchIndex() {
        this.lastPinnedSearchIndexForMigration = this.searchIndex.get();
        this.consensusReqReader.setSafelyDeletedSearchIndex(this.getMinFlushedSyncIndex());
    }

    public void checkAndUnlockSafeDeletedSearchIndex() {
        this.lastPinnedSearchIndexForMigration = -1L;
        this.checkAndUpdateSafeDeletedSearchIndex();
    }

    public void checkAndUpdateSafeDeletedSearchIndex() {
        if (this.configuration.size() == 1) {
            this.consensusReqReader.setSafelyDeletedSearchIndex(Long.MAX_VALUE);
        } else {
            this.consensusReqReader.setSafelyDeletedSearchIndex(this.getMinFlushedSyncIndex());
        }
    }

    public void checkAndUpdateSearchIndex() {
        long safelyDeletedSearchIndex;
        long currentSearchIndex = this.searchIndex.get();
        if (currentSearchIndex < (safelyDeletedSearchIndex = this.getMinFlushedSyncIndex())) {
            this.logger.warn("The searchIndex for this region({}) is smaller than the safelyDeletedSearchIndex when the node is restarted, which means that the data of the current region is not flushed by the wal, but has been synchronized to other nodes. At this point, different replicas have been inconsistent and cannot be automatically recovered. To prevent subsequent logs from marking smaller searchIndex and exacerbating the inconsistency, we manually set the searchIndex({}) to safelyDeletedSearchIndex({}) here to reduce the impact of this problem in the future", new Object[]{this.consensusGroupId, currentSearchIndex, safelyDeletedSearchIndex});
            this.searchIndex.set(safelyDeletedSearchIndex);
        }
    }

    public TSStatus syncLog(int sourcePeerId, IConsensusRequest request) {
        return this.cacheQueueMap.computeIfAbsent(sourcePeerId, x$0 -> new SyncLogCacheQueue((int)x$0)).cacheAndInsertLatestNode((DeserializedBatchIndexedConsensusRequest)request);
    }

    public String getConsensusGroupId() {
        return this.consensusGroupId;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void serializeConfigurationAndFsyncToDisk() throws IOException {
        for (Peer peer : this.configuration) {
            String peerConfigurationFileName = IoTConsensusServerImpl.generateConfigurationDatFileName(peer.getNodeId(), CONFIGURATION_TMP_FILE_NAME);
            FileOutputStream fileOutputStream = new FileOutputStream(new File(this.storageDir, peerConfigurationFileName));
            try (DataOutputStream outputStream = new DataOutputStream(fileOutputStream);){
                peer.serialize(outputStream);
            }
            finally {
                try {
                    fileOutputStream.flush();
                    fileOutputStream.getFD().sync();
                }
                catch (IOException iOException) {}
            }
        }
    }

    private void renameTmpConfigurationFileToRemoveSuffix() throws IOException {
        try (Stream<Path> stream = Files.list(Paths.get(this.storageDir, new String[0]));){
            List paths = stream.filter(x$0 -> Files.isRegularFile(x$0, new LinkOption[0])).filter(filePath -> filePath.getFileName().toString().endsWith(CONFIGURATION_TMP_FILE_NAME)).collect(Collectors.toList());
            for (Path filePath2 : paths) {
                String targetPath = filePath2.toString().replace(CONFIGURATION_TMP_FILE_NAME, CONFIGURATION_FILE_NAME);
                File targetFile = new File(targetPath);
                if (targetFile.exists()) {
                    try {
                        Files.delete(targetFile.toPath());
                    }
                    catch (IOException e) {
                        this.logger.error("Unexpected error occurs when delete file: {}", (Object)targetPath, (Object)e);
                    }
                }
                if (filePath2.toFile().renameTo(targetFile)) continue;
                this.logger.error("Unexpected error occurs when rename file: {} -> {}", (Object)filePath2, (Object)targetPath);
            }
        }
        catch (UncheckedIOException e) {
            throw e.getCause();
        }
    }

    private void deleteConfiguration() throws IOException {
        try (Stream<Path> stream = Files.list(Paths.get(this.storageDir, new String[0]));){
            stream.filter(x$0 -> Files.isRegularFile(x$0, new LinkOption[0])).filter(filePath -> filePath.getFileName().toString().endsWith(CONFIGURATION_FILE_NAME)).forEach(filePath -> {
                try {
                    Files.delete(filePath);
                }
                catch (IOException e) {
                    this.logger.error("Unexpected error occurs when deleting old configuration file {}", filePath, (Object)e);
                }
            });
        }
        catch (UncheckedIOException e) {
            throw e.getCause();
        }
    }

    public void removeDuplicateConfiguration() {
        HashSet<Peer> seen = new HashSet<Peer>();
        Iterator<Peer> it = this.configuration.iterator();
        while (it.hasNext()) {
            Peer peer = it.next();
            if (seen.add(peer)) continue;
            it.remove();
        }
    }

    private class SyncLogCacheQueue {
        private final int sourcePeerId;
        private final Lock queueLock = new ReentrantLock();
        private final Condition queueSortCondition = this.queueLock.newCondition();
        private final PriorityQueue<DeserializedBatchIndexedConsensusRequest> requestCache;
        private long nextSyncIndex = -1L;

        public SyncLogCacheQueue(int sourcePeerId) {
            this.sourcePeerId = sourcePeerId;
            this.requestCache = new PriorityQueue();
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private TSStatus cacheAndInsertLatestNode(DeserializedBatchIndexedConsensusRequest request) {
            IoTConsensusServerImpl.this.logger.debug("cacheAndInsert start: source = {}, region = {}, queue size {}, startSyncIndex = {}, endSyncIndex = {}", new Object[]{this.sourcePeerId, IoTConsensusServerImpl.this.consensusGroupId, this.requestCache.size(), request.getStartSyncIndex(), request.getEndSyncIndex()});
            this.queueLock.lock();
            try {
                long insertStartTime = System.nanoTime();
                this.requestCache.add(request);
                if (this.requestCache.size() == IoTConsensusServerImpl.this.config.getReplication().getMaxPendingBatchesNum() && this.requestCache.peek() != null && this.requestCache.peek().getStartSyncIndex() != request.getStartSyncIndex()) {
                    this.queueSortCondition.signalAll();
                }
                while (true) {
                    if (request.getStartSyncIndex() == this.nextSyncIndex) {
                        this.requestCache.remove(request);
                        this.nextSyncIndex = request.getEndSyncIndex() + 1L;
                        break;
                    }
                    if (this.requestCache.size() == IoTConsensusServerImpl.this.config.getReplication().getMaxPendingBatchesNum() && this.requestCache.peek() != null && this.requestCache.peek().getStartSyncIndex() == request.getStartSyncIndex()) {
                        this.requestCache.remove();
                        this.nextSyncIndex = request.getEndSyncIndex() + 1L;
                        break;
                    }
                    try {
                        boolean timeout = !this.queueSortCondition.await(IoTConsensusServerImpl.this.config.getReplication().getMaxWaitingTimeForWaitBatchInMs(), TimeUnit.MILLISECONDS);
                        if (!timeout || this.requestCache.peek() == null || this.requestCache.peek().getStartSyncIndex() != request.getStartSyncIndex()) continue;
                        IoTConsensusServerImpl.this.logger.info("waiting target request timeout. current index: {}, target index: {}", (Object)request.getStartSyncIndex(), (Object)this.nextSyncIndex);
                        this.requestCache.remove(request);
                        this.nextSyncIndex = Math.max(this.nextSyncIndex, request.getEndSyncIndex() + 1L);
                    }
                    catch (InterruptedException e) {
                        IoTConsensusServerImpl.this.logger.warn("current waiting is interrupted. SyncIndex: {}. Exception: ", (Object)request.getStartSyncIndex(), (Object)e);
                        Thread.currentThread().interrupt();
                        continue;
                    }
                    break;
                }
                long sortTime = System.nanoTime();
                IoTConsensusServerImpl.this.ioTConsensusServerMetrics.recordSortCost(sortTime - insertStartTime);
                LinkedList<TSStatus> subStatus = new LinkedList<TSStatus>();
                for (IConsensusRequest insertNode : request.getInsertNodes()) {
                    subStatus.add(IoTConsensusServerImpl.this.stateMachine.write(insertNode));
                }
                long applyTime = System.nanoTime();
                IoTConsensusServerImpl.this.ioTConsensusServerMetrics.recordApplyCost(applyTime - sortTime);
                this.queueSortCondition.signalAll();
                IoTConsensusServerImpl.this.logger.debug("cacheAndInsert end: source = {}, region = {}, queue size {}, startSyncIndex = {}, endSyncIndex = {}, sortTime = {}ms, applyTime = {}ms", new Object[]{this.sourcePeerId, IoTConsensusServerImpl.this.consensusGroupId, this.requestCache.size(), request.getStartSyncIndex(), request.getEndSyncIndex(), TimeUnit.NANOSECONDS.toMillis(sortTime - insertStartTime), TimeUnit.NANOSECONDS.toMillis(applyTime - sortTime)});
                TSStatus tSStatus = new TSStatus().setSubStatus(subStatus);
                return tSStatus;
            }
            finally {
                this.queueLock.unlock();
            }
        }
    }

    @FunctionalInterface
    public static interface ThrowableFunction<T, R> {
        public R apply(T var1) throws Exception;
    }
}

