/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iotdb.tsfile.write.chunk;

import java.io.IOException;
import java.io.Serializable;
import java.nio.ByteBuffer;
import java.nio.channels.Channels;
import java.nio.channels.WritableByteChannel;
import org.apache.iotdb.tsfile.common.conf.TSFileDescriptor;
import org.apache.iotdb.tsfile.compress.ICompressor;
import org.apache.iotdb.tsfile.encoding.encoder.SDTEncoder;
import org.apache.iotdb.tsfile.exception.write.PageException;
import org.apache.iotdb.tsfile.file.header.ChunkHeader;
import org.apache.iotdb.tsfile.file.header.PageHeader;
import org.apache.iotdb.tsfile.file.metadata.enums.TSDataType;
import org.apache.iotdb.tsfile.file.metadata.statistics.Statistics;
import org.apache.iotdb.tsfile.utils.Binary;
import org.apache.iotdb.tsfile.utils.PublicBAOS;
import org.apache.iotdb.tsfile.utils.ReadWriteForEncodingUtils;
import org.apache.iotdb.tsfile.write.chunk.IChunkWriter;
import org.apache.iotdb.tsfile.write.page.PageWriter;
import org.apache.iotdb.tsfile.write.schema.IMeasurementSchema;
import org.apache.iotdb.tsfile.write.writer.TsFileIOWriter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ChunkWriterImpl
implements IChunkWriter {
    private static final Logger logger = LoggerFactory.getLogger(ChunkWriterImpl.class);
    private final IMeasurementSchema measurementSchema;
    private final ICompressor compressor;
    private final PublicBAOS pageBuffer;
    private int numOfPages;
    private PageWriter pageWriter;
    private final long pageSizeThreshold;
    private final int maxNumberOfPointsInPage;
    private int valueCountInOnePageForNextCheck;
    private static final int MINIMUM_RECORD_COUNT_FOR_CHECK = 1500;
    private Statistics statistics;
    private boolean isSdtEncoding;
    private boolean isLastPoint;
    private boolean isMerging;
    private SDTEncoder sdtEncoder;
    private static final String LOSS = "loss";
    private static final String SDT = "sdt";
    private static final String SDT_COMP_DEV = "compdev";
    private static final String SDT_COMP_MIN_TIME = "compmintime";
    private static final String SDT_COMP_MAX_TIME = "compmaxtime";
    private int sizeWithoutStatistic;
    private Statistics<?> firstPageStatistics;

    public ChunkWriterImpl(IMeasurementSchema schema) {
        this.measurementSchema = schema;
        this.compressor = ICompressor.getCompressor(schema.getCompressor());
        this.pageBuffer = new PublicBAOS();
        this.pageSizeThreshold = TSFileDescriptor.getInstance().getConfig().getPageSizeInByte();
        this.maxNumberOfPointsInPage = TSFileDescriptor.getInstance().getConfig().getMaxNumberOfPointsInPage();
        this.valueCountInOnePageForNextCheck = 1500;
        this.statistics = Statistics.getStatsByType(this.measurementSchema.getType());
        this.pageWriter = new PageWriter(this.measurementSchema);
        this.pageWriter.setTimeEncoder(this.measurementSchema.getTimeEncoder());
        this.pageWriter.setValueEncoder(this.measurementSchema.getValueEncoder());
        this.checkSdtEncoding();
    }

    public ChunkWriterImpl(IMeasurementSchema schema, boolean isMerging) {
        this(schema);
        this.isMerging = isMerging;
    }

    private void checkSdtEncoding() {
        if (this.measurementSchema.getProps() != null && !this.isMerging) {
            if (this.measurementSchema.getProps().getOrDefault(LOSS, "").equals(SDT)) {
                this.isSdtEncoding = true;
                this.sdtEncoder = new SDTEncoder();
            }
            if (this.isSdtEncoding && this.measurementSchema.getProps().containsKey(SDT_COMP_DEV)) {
                this.sdtEncoder.setCompDeviation(Double.parseDouble(this.measurementSchema.getProps().get(SDT_COMP_DEV)));
            }
            if (this.isSdtEncoding && this.measurementSchema.getProps().containsKey(SDT_COMP_MIN_TIME)) {
                this.sdtEncoder.setCompMinTime(Long.parseLong(this.measurementSchema.getProps().get(SDT_COMP_MIN_TIME)));
            }
            if (this.isSdtEncoding && this.measurementSchema.getProps().containsKey(SDT_COMP_MAX_TIME)) {
                this.sdtEncoder.setCompMaxTime(Long.parseLong(this.measurementSchema.getProps().get(SDT_COMP_MAX_TIME)));
            }
        }
    }

    public void write(long time, long value) {
        if (!this.isSdtEncoding || this.sdtEncoder.encodeLong(time, value)) {
            this.pageWriter.write(this.isSdtEncoding ? this.sdtEncoder.getTime() : time, this.isSdtEncoding ? this.sdtEncoder.getLongValue() : value);
        }
        if (this.isSdtEncoding && this.isLastPoint) {
            this.pageWriter.write(time, value);
        }
        this.checkPageSizeAndMayOpenANewPage();
    }

    public void write(long time, int value) {
        if (!this.isSdtEncoding || this.sdtEncoder.encodeInt(time, value)) {
            this.pageWriter.write(this.isSdtEncoding ? this.sdtEncoder.getTime() : time, this.isSdtEncoding ? this.sdtEncoder.getIntValue() : value);
        }
        if (this.isSdtEncoding && this.isLastPoint) {
            this.pageWriter.write(time, value);
        }
        this.checkPageSizeAndMayOpenANewPage();
    }

    public void write(long time, boolean value) {
        this.pageWriter.write(time, value);
        this.checkPageSizeAndMayOpenANewPage();
    }

    public void write(long time, float value) {
        if (!this.isSdtEncoding || this.sdtEncoder.encodeFloat(time, value)) {
            this.pageWriter.write(this.isSdtEncoding ? this.sdtEncoder.getTime() : time, this.isSdtEncoding ? this.sdtEncoder.getFloatValue() : value);
        }
        if (this.isSdtEncoding && this.isLastPoint) {
            this.pageWriter.write(time, value);
        }
        this.checkPageSizeAndMayOpenANewPage();
    }

    public void write(long time, double value) {
        if (!this.isSdtEncoding || this.sdtEncoder.encodeDouble(time, value)) {
            this.pageWriter.write(this.isSdtEncoding ? this.sdtEncoder.getTime() : time, this.isSdtEncoding ? this.sdtEncoder.getDoubleValue() : value);
        }
        if (this.isSdtEncoding && this.isLastPoint) {
            this.pageWriter.write(time, value);
        }
        this.checkPageSizeAndMayOpenANewPage();
    }

    public void write(long time, Binary value) {
        this.pageWriter.write(time, value);
        this.checkPageSizeAndMayOpenANewPage();
    }

    public void write(long[] timestamps, int[] values, int batchSize) {
        if (this.isSdtEncoding) {
            batchSize = this.sdtEncoder.encode(timestamps, values, batchSize);
        }
        this.pageWriter.write(timestamps, values, batchSize);
        this.checkPageSizeAndMayOpenANewPage();
    }

    public void write(long[] timestamps, long[] values, int batchSize) {
        if (this.isSdtEncoding) {
            batchSize = this.sdtEncoder.encode(timestamps, values, batchSize);
        }
        this.pageWriter.write(timestamps, values, batchSize);
        this.checkPageSizeAndMayOpenANewPage();
    }

    public void write(long[] timestamps, boolean[] values, int batchSize) {
        this.pageWriter.write(timestamps, values, batchSize);
        this.checkPageSizeAndMayOpenANewPage();
    }

    public void write(long[] timestamps, float[] values, int batchSize) {
        if (this.isSdtEncoding) {
            batchSize = this.sdtEncoder.encode(timestamps, values, batchSize);
        }
        this.pageWriter.write(timestamps, values, batchSize);
        this.checkPageSizeAndMayOpenANewPage();
    }

    public void write(long[] timestamps, double[] values, int batchSize) {
        if (this.isSdtEncoding) {
            batchSize = this.sdtEncoder.encode(timestamps, values, batchSize);
        }
        this.pageWriter.write(timestamps, values, batchSize);
        this.checkPageSizeAndMayOpenANewPage();
    }

    public void write(long[] timestamps, Binary[] values, int batchSize) {
        this.pageWriter.write(timestamps, values, batchSize);
        this.checkPageSizeAndMayOpenANewPage();
    }

    private void checkPageSizeAndMayOpenANewPage() {
        if (this.pageWriter.getPointNumber() == (long)this.maxNumberOfPointsInPage) {
            logger.debug("current line count reaches the upper bound, write page {}", (Object)this.measurementSchema);
            this.writePageToPageBuffer();
        } else if (this.pageWriter.getPointNumber() >= (long)this.valueCountInOnePageForNextCheck) {
            long currentPageSize = this.pageWriter.estimateMaxMemSize();
            if (currentPageSize > this.pageSizeThreshold) {
                logger.debug("enough size, write page {}, pageSizeThreshold:{}, currentPateSize:{}, valueCountInOnePage:{}", this.measurementSchema.getMeasurementId(), this.pageSizeThreshold, currentPageSize, this.pageWriter.getPointNumber());
                this.writePageToPageBuffer();
                this.valueCountInOnePageForNextCheck = 1500;
            } else {
                this.valueCountInOnePageForNextCheck = (int)((float)this.pageSizeThreshold / (float)currentPageSize * (float)this.pageWriter.getPointNumber());
            }
        }
    }

    private void writePageToPageBuffer() {
        try {
            if (this.numOfPages == 0) {
                this.firstPageStatistics = this.pageWriter.getStatistics();
                this.sizeWithoutStatistic = this.pageWriter.writePageHeaderAndDataIntoBuff(this.pageBuffer, true);
            } else if (this.numOfPages == 1) {
                byte[] b = this.pageBuffer.toByteArray();
                this.pageBuffer.reset();
                this.pageBuffer.write(b, 0, this.sizeWithoutStatistic);
                this.firstPageStatistics.serialize(this.pageBuffer);
                this.pageBuffer.write(b, this.sizeWithoutStatistic, b.length - this.sizeWithoutStatistic);
                this.pageWriter.writePageHeaderAndDataIntoBuff(this.pageBuffer, false);
                this.firstPageStatistics = null;
            } else {
                this.pageWriter.writePageHeaderAndDataIntoBuff(this.pageBuffer, false);
            }
            ++this.numOfPages;
            this.statistics.mergeStatistics(this.pageWriter.getStatistics());
        }
        catch (IOException e) {
            logger.error("meet error in pageWriter.writePageHeaderAndDataIntoBuff,ignore this page:", e);
        }
        finally {
            this.pageWriter.reset(this.measurementSchema);
        }
    }

    @Override
    public void writeToFileWriter(TsFileIOWriter tsfileWriter) throws IOException {
        this.sealCurrentPage();
        this.writeAllPagesOfChunkToTsFile(tsfileWriter, this.statistics);
        this.pageBuffer.reset();
        this.numOfPages = 0;
        this.sizeWithoutStatistic = 0;
        this.firstPageStatistics = null;
        this.statistics = Statistics.getStatsByType(this.measurementSchema.getType());
    }

    @Override
    public long estimateMaxSeriesMemSize() {
        return (long)this.pageBuffer.size() + this.pageWriter.estimateMaxMemSize() + (long)PageHeader.estimateMaxPageHeaderSizeWithoutStatistics() + (long)this.pageWriter.getStatistics().getSerializedSize();
    }

    @Override
    public long getSerializedChunkSize() {
        if (this.pageBuffer.size() == 0) {
            return 0L;
        }
        return (long)ChunkHeader.getSerializedSize(this.measurementSchema.getMeasurementId(), this.pageBuffer.size()) + (long)this.pageBuffer.size();
    }

    @Override
    public void sealCurrentPage() {
        if (this.pageWriter != null && this.pageWriter.getPointNumber() > 0L) {
            this.writePageToPageBuffer();
        }
    }

    @Override
    public void clearPageWriter() {
        this.pageWriter = null;
    }

    @Override
    public boolean checkIsUnsealedPageOverThreshold(long size, long pointNum, boolean returnTrueIfPageEmpty) {
        if (returnTrueIfPageEmpty && this.pageWriter.getPointNumber() == 0L) {
            return true;
        }
        return this.pageWriter.getPointNumber() >= pointNum || this.pageWriter.estimateMaxMemSize() >= size;
    }

    @Override
    public boolean checkIsChunkSizeOverThreshold(long size, long pointNum, boolean returnTrueIfChunkEmpty) {
        if (returnTrueIfChunkEmpty && this.statistics.getCount() + this.pageWriter.getPointNumber() == 0L) {
            return true;
        }
        return this.estimateMaxSeriesMemSize() >= size || this.statistics.getCount() + this.pageWriter.getPointNumber() >= pointNum;
    }

    @Override
    public boolean isEmpty() {
        return this.statistics.getCount() + this.pageWriter.getPointNumber() == 0L;
    }

    public TSDataType getDataType() {
        return this.measurementSchema.getType();
    }

    public void writePageHeaderAndDataIntoBuff(ByteBuffer data, PageHeader header) throws PageException {
        try {
            logger.debug("start to flush a page header into buffer, buffer position {} ", (Object)this.pageBuffer.size());
            if (this.numOfPages == 0) {
                this.firstPageStatistics = header.getStatistics();
                this.sizeWithoutStatistic += ReadWriteForEncodingUtils.writeUnsignedVarInt(header.getUncompressedSize(), this.pageBuffer);
                this.sizeWithoutStatistic += ReadWriteForEncodingUtils.writeUnsignedVarInt(header.getCompressedSize(), this.pageBuffer);
            } else if (this.numOfPages == 1) {
                byte[] b = this.pageBuffer.toByteArray();
                this.pageBuffer.reset();
                this.pageBuffer.write(b, 0, this.sizeWithoutStatistic);
                this.firstPageStatistics.serialize(this.pageBuffer);
                this.pageBuffer.write(b, this.sizeWithoutStatistic, b.length - this.sizeWithoutStatistic);
                ReadWriteForEncodingUtils.writeUnsignedVarInt(header.getUncompressedSize(), this.pageBuffer);
                ReadWriteForEncodingUtils.writeUnsignedVarInt(header.getCompressedSize(), this.pageBuffer);
                header.getStatistics().serialize(this.pageBuffer);
                this.firstPageStatistics = null;
            } else {
                ReadWriteForEncodingUtils.writeUnsignedVarInt(header.getUncompressedSize(), this.pageBuffer);
                ReadWriteForEncodingUtils.writeUnsignedVarInt(header.getCompressedSize(), this.pageBuffer);
                header.getStatistics().serialize(this.pageBuffer);
            }
            logger.debug("finish to flush a page header {} of {} into buffer, buffer position {} ", header, this.measurementSchema.getMeasurementId(), this.pageBuffer.size());
            this.statistics.mergeStatistics(header.getStatistics());
        }
        catch (IOException e) {
            throw new PageException("IO Exception in writeDataPageHeader,ignore this page", e);
        }
        ++this.numOfPages;
        try (WritableByteChannel channel = Channels.newChannel(this.pageBuffer);){
            channel.write(data);
        }
        catch (IOException e) {
            throw new PageException(e);
        }
    }

    private void writeAllPagesOfChunkToTsFile(TsFileIOWriter writer, Statistics<? extends Serializable> statistics) throws IOException {
        if (statistics.getCount() == 0L) {
            return;
        }
        writer.startFlushChunk(this.measurementSchema.getMeasurementId(), this.compressor.getType(), this.measurementSchema.getType(), this.measurementSchema.getEncodingType(), statistics, this.pageBuffer.size(), this.numOfPages, 0);
        long dataOffset = writer.getPos();
        writer.writeBytesToStream(this.pageBuffer);
        int dataSize = (int)(writer.getPos() - dataOffset);
        if (dataSize != this.pageBuffer.size()) {
            throw new IOException("Bytes written is inconsistent with the size of data: " + dataSize + " != " + this.pageBuffer.size());
        }
        writer.endCurrentChunk();
    }

    public void setIsMerging(boolean isMerging) {
        this.isMerging = isMerging;
    }

    public boolean isMerging() {
        return this.isMerging;
    }

    public void setLastPoint(boolean isLastPoint) {
        this.isLastPoint = isLastPoint;
    }

    public PageWriter getPageWriter() {
        return this.pageWriter;
    }
}

