/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iotdb.confignode.procedure;

import com.google.common.base.Preconditions;
import java.io.IOException;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Deque;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import org.apache.iotdb.commons.concurrent.ThreadName;
import org.apache.iotdb.confignode.procedure.CompletedProcedureContainer;
import org.apache.iotdb.confignode.procedure.CompletedProcedureRecycler;
import org.apache.iotdb.confignode.procedure.InternalProcedure;
import org.apache.iotdb.confignode.procedure.Procedure;
import org.apache.iotdb.confignode.procedure.RootProcedureStack;
import org.apache.iotdb.confignode.procedure.StoppableThread;
import org.apache.iotdb.confignode.procedure.TimeoutExecutorThread;
import org.apache.iotdb.confignode.procedure.exception.ProcedureException;
import org.apache.iotdb.confignode.procedure.exception.ProcedureSuspendedException;
import org.apache.iotdb.confignode.procedure.exception.ProcedureYieldException;
import org.apache.iotdb.confignode.procedure.scheduler.ProcedureScheduler;
import org.apache.iotdb.confignode.procedure.scheduler.SimpleProcedureScheduler;
import org.apache.iotdb.confignode.procedure.state.ProcedureLockState;
import org.apache.iotdb.confignode.procedure.state.ProcedureState;
import org.apache.iotdb.confignode.procedure.store.IProcedureStore;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ProcedureExecutor<Env> {
    private static final Logger LOG = LoggerFactory.getLogger(ProcedureExecutor.class);
    private final ConcurrentHashMap<Long, CompletedProcedureContainer<Env>> completed = new ConcurrentHashMap();
    private final ConcurrentHashMap<Long, RootProcedureStack<Env>> rollbackStack = new ConcurrentHashMap();
    private final ConcurrentHashMap<Long, Procedure> procedures = new ConcurrentHashMap();
    private ThreadGroup threadGroup;
    private CopyOnWriteArrayList<WorkerThread> workerThreads;
    private TimeoutExecutorThread<Env> timeoutExecutor;
    private TimeoutExecutorThread<Env> workerMonitorExecutor;
    private int corePoolSize;
    private int maxPoolSize;
    private final ProcedureScheduler scheduler;
    private final AtomicLong lastProcId = new AtomicLong(-1L);
    private final AtomicLong workId = new AtomicLong(0L);
    private final AtomicInteger activeExecutorCount = new AtomicInteger(0);
    private final AtomicBoolean running = new AtomicBoolean(false);
    private final Env environment;
    private final IProcedureStore store;

    public ProcedureExecutor(Env environment, IProcedureStore store, ProcedureScheduler scheduler) {
        this.environment = environment;
        this.scheduler = scheduler;
        this.store = store;
        this.lastProcId.incrementAndGet();
    }

    public ProcedureExecutor(Env environment, IProcedureStore store) {
        this(environment, store, new SimpleProcedureScheduler());
    }

    public void init(int numThreads) {
        this.corePoolSize = numThreads;
        this.maxPoolSize = 10 * numThreads;
        this.threadGroup = new ThreadGroup(ThreadName.CONFIG_NODE_PROCEDURE_WORKER.getName());
        this.timeoutExecutor = new TimeoutExecutorThread(this, this.threadGroup, ThreadName.CONFIG_NODE_TIMEOUT_EXECUTOR.getName());
        this.workerMonitorExecutor = new TimeoutExecutorThread(this, this.threadGroup, ThreadName.CONFIG_NODE_WORKER_THREAD_MONITOR.getName());
        this.workId.set(0L);
        this.workerThreads = new CopyOnWriteArrayList();
        for (int i = 0; i < this.corePoolSize; ++i) {
            this.workerThreads.add(new WorkerThread(this.threadGroup));
        }
        this.workerMonitorExecutor.add(new WorkerMonitor());
        this.scheduler.start();
        this.recover();
    }

    private void recover() {
        int runnableCount = 0;
        int failedCount = 0;
        int waitingCount = 0;
        int waitingTimeoutCount = 0;
        ArrayList<Procedure> procedureList = new ArrayList<Procedure>();
        this.store.load(procedureList);
        for (Procedure proc : procedureList) {
            if (proc.isFinished()) {
                this.completed.putIfAbsent(proc.getProcId(), new CompletedProcedureContainer(proc));
            } else if (!proc.hasParent()) {
                this.rollbackStack.put(proc.getProcId(), new RootProcedureStack());
            }
            this.procedures.putIfAbsent(proc.getProcId(), proc);
            switch (proc.getState()) {
                case RUNNABLE: {
                    ++runnableCount;
                    break;
                }
                case FAILED: {
                    ++failedCount;
                    break;
                }
                case WAITING: {
                    ++waitingCount;
                    break;
                }
                case WAITING_TIMEOUT: {
                    ++waitingTimeoutCount;
                    break;
                }
            }
        }
        ArrayList<Procedure> runnableList = new ArrayList<Procedure>(runnableCount);
        ArrayList<Procedure> failedList = new ArrayList<Procedure>(failedCount);
        ArrayList<Procedure> waitingList = new ArrayList<Procedure>(waitingCount);
        ArrayList<Procedure> waitingTimeoutList = new ArrayList<Procedure>(waitingTimeoutCount);
        for (Procedure proc : procedureList) {
            RootProcedureStack<Env> rootStack;
            Procedure parent;
            if (proc.isFinished() && !proc.hasParent()) continue;
            long rootProcedureId = this.getRootProcId(proc);
            if (proc.hasParent() && (parent = this.procedures.get(proc.getParentProcId())) != null && !proc.isFinished()) {
                parent.incChildrenLatch();
            }
            if ((rootStack = this.rollbackStack.get(rootProcedureId)) != null) {
                rootStack.loadStack(proc);
            }
            proc.setRootProcedureId(rootProcedureId);
            switch (proc.getState()) {
                case RUNNABLE: {
                    runnableList.add(proc);
                    break;
                }
                case FAILED: {
                    failedList.add(proc);
                    break;
                }
                case WAITING: {
                    waitingList.add(proc);
                    break;
                }
                case WAITING_TIMEOUT: {
                    waitingTimeoutList.add(proc);
                    break;
                }
                case ROLLEDBACK: 
                case INITIALIZING: {
                    LOG.error("Unexpected state:{} for {}", (Object)proc.getState(), (Object)proc);
                    throw new UnsupportedOperationException("Unexpected state");
                }
            }
        }
        waitingList.forEach(procedure -> {
            if (procedure.hasChildren()) {
                procedure.setState(ProcedureState.RUNNABLE);
                runnableList.add((Procedure)procedure);
            } else {
                procedure.afterRecover(this.environment);
            }
        });
        this.restoreLocks();
        waitingTimeoutList.forEach(procedure -> {
            procedure.afterRecover(this.environment);
            this.timeoutExecutor.add((Procedure<Env>)procedure);
        });
        failedList.forEach(this.scheduler::addBack);
        runnableList.forEach(procedure -> {
            procedure.afterRecover(this.environment);
            this.scheduler.addBack((Procedure)procedure);
        });
        this.scheduler.signalAll();
    }

    public long getRootProcId(Procedure proc) {
        return Procedure.getRootProcedureId(this.procedures, proc);
    }

    private void releaseLock(Procedure<Env> procedure, boolean force) {
        if (force || !procedure.holdLock(this.environment) || procedure.isFinished()) {
            procedure.doReleaseLock(this.environment, this.store);
        }
    }

    private void restoreLock(Procedure procedure, Set<Long> restored) {
        procedure.restoreLock(this.environment);
        restored.add(procedure.getProcId());
    }

    private void restoreLocks(Deque<Procedure<Env>> stack, Set<Long> restored) {
        while (!stack.isEmpty()) {
            this.restoreLock(stack.pop(), restored);
        }
    }

    private void restoreLocks() {
        HashSet restored = new HashSet();
        ArrayDeque stack = new ArrayDeque();
        this.procedures.values().forEach(procedure -> {
            while (procedure != null) {
                if (restored.contains(procedure.getProcId())) {
                    this.restoreLocks(stack, restored);
                    return;
                }
                if (!procedure.hasParent()) {
                    this.restoreLock((Procedure)procedure, restored);
                    this.restoreLocks(stack, restored);
                    return;
                }
                stack.push(procedure);
                procedure = this.procedures.get(procedure.getParentProcId());
            }
        });
    }

    public void startWorkers() {
        if (!this.running.compareAndSet(false, true)) {
            LOG.warn("Already running");
            return;
        }
        this.timeoutExecutor.start();
        this.workerMonitorExecutor.start();
        for (WorkerThread workerThread : this.workerThreads) {
            workerThread.start();
        }
    }

    public void startCompletedCleaner(long cleanTimeInterval, long cleanEvictTTL) {
        this.addInternalProcedure(new CompletedProcedureRecycler<Env>(this.store, this.completed, cleanTimeInterval, cleanEvictTTL));
    }

    private void addInternalProcedure(InternalProcedure interalProcedure) {
        if (interalProcedure == null) {
            return;
        }
        interalProcedure.setState(ProcedureState.WAITING_TIMEOUT);
        this.timeoutExecutor.add(interalProcedure);
    }

    public boolean removeInternalProcedure(InternalProcedure internalProcedure) {
        if (internalProcedure == null) {
            return true;
        }
        internalProcedure.setState(ProcedureState.SUCCESS);
        return this.timeoutExecutor.remove(internalProcedure);
    }

    private long nextProcId() {
        long procId = this.lastProcId.incrementAndGet();
        if (procId < 0L) {
            while (!this.lastProcId.compareAndSet(procId, 0L) && (procId = this.lastProcId.get()) < 0L) {
            }
            while (this.procedures.containsKey(procId)) {
                procId = this.lastProcId.incrementAndGet();
            }
        }
        return procId;
    }

    private void executeProcedure(Procedure<Env> proc) {
        if (proc.isFinished()) {
            LOG.debug("{} is already finished.", proc);
            return;
        }
        Long rootProcId = this.getRootProcedureId(proc);
        if (rootProcId == null) {
            LOG.warn("Rollback because parent is done/rolledback, proc is {}", proc);
            this.executeRollback(proc);
            return;
        }
        RootProcedureStack<Env> rootProcStack = this.rollbackStack.get(rootProcId);
        if (rootProcStack == null) {
            LOG.warn("Rollback stack is null for {}", (Object)proc.getProcId());
            return;
        }
        block14: do {
            if (!rootProcStack.acquire()) {
                if (rootProcStack.setRollback()) {
                    switch (this.executeRootStackRollback(rootProcId, rootProcStack)) {
                        case LOCK_ACQUIRED: {
                            break block14;
                        }
                        case LOCK_EVENT_WAIT: {
                            LOG.info("LOCK_EVENT_WAIT rollback " + proc);
                            rootProcStack.unsetRollback();
                            break block14;
                        }
                        case LOCK_YIELD_WAIT: {
                            rootProcStack.unsetRollback();
                            this.scheduler.yield(proc);
                            break block14;
                        }
                        default: {
                            throw new UnsupportedOperationException();
                        }
                    }
                }
                if (proc.wasExecuted()) break;
                switch (this.executeRollback(proc)) {
                    case LOCK_ACQUIRED: {
                        break block14;
                    }
                    case LOCK_EVENT_WAIT: {
                        LOG.info("LOCK_EVENT_WAIT can't rollback child running for {}", proc);
                        break block14;
                    }
                    case LOCK_YIELD_WAIT: {
                        this.scheduler.yield(proc);
                        break block14;
                    }
                    default: {
                        throw new UnsupportedOperationException();
                    }
                }
            }
            ProcedureLockState lockState = this.acquireLock(proc);
            switch (lockState) {
                case LOCK_ACQUIRED: {
                    this.executeProcedure(rootProcStack, proc);
                    break;
                }
                case LOCK_EVENT_WAIT: 
                case LOCK_YIELD_WAIT: {
                    LOG.info("{} lockstate is {}", proc, (Object)lockState);
                    break;
                }
                default: {
                    throw new UnsupportedOperationException();
                }
            }
            rootProcStack.release();
            if (!proc.isSuccess()) continue;
            LOG.info("{} finished in {}ms successfully.", proc, (Object)proc.elapsedTime());
            if (proc.getProcId() == rootProcId.longValue()) {
                this.rootProcedureCleanup(proc);
            } else {
                this.executeCompletionCleanup(proc);
            }
            return;
        } while (rootProcStack.isFailed());
    }

    private void executeProcedure(RootProcedureStack rootProcStack, Procedure<Env> proc) {
        boolean reExecute;
        Preconditions.checkArgument((proc.getState() == ProcedureState.RUNNABLE ? 1 : 0) != 0, (Object)("NOT RUNNABLE! " + proc));
        boolean suspended = false;
        Object[] subprocs = null;
        do {
            reExecute = false;
            proc.resetPersistance();
            try {
                subprocs = proc.doExecute(this.environment);
                if (subprocs != null && subprocs.length == 0) {
                    subprocs = null;
                }
            }
            catch (ProcedureSuspendedException e) {
                LOG.debug("Suspend {}", proc);
                suspended = true;
            }
            catch (ProcedureYieldException e) {
                LOG.debug("Yield {}", proc);
                this.yieldProcedure(proc);
            }
            catch (InterruptedException e) {
                LOG.warn("Interrupt during execution, suspend or retry it later.", (Throwable)e);
                this.yieldProcedure(proc);
            }
            catch (Throwable e) {
                LOG.error("CODE-BUG:{}", proc, (Object)e);
                proc.setFailure(new ProcedureException(e.getMessage(), e));
            }
            if (!proc.isFailed()) {
                if (subprocs != null) {
                    if (subprocs.length == 1 && subprocs[0] == proc) {
                        subprocs = null;
                        reExecute = true;
                    } else {
                        subprocs = this.initializeChildren(rootProcStack, proc, (Procedure<Env>[])subprocs);
                        LOG.info("Initialized sub procs:{}", (Object)Arrays.toString(subprocs));
                    }
                } else if (proc.getState() == ProcedureState.WAITING_TIMEOUT) {
                    LOG.info("Added into timeoutExecutor {}", proc);
                } else if (!suspended) {
                    proc.setState(ProcedureState.SUCCESS);
                }
            }
            rootProcStack.addRollbackStep(proc);
            if (proc.needPersistance()) {
                this.updateStoreOnExecution(rootProcStack, proc, (Procedure<Env>[])subprocs);
            }
            if (!this.store.isRunning()) {
                return;
            }
            if (!proc.isRunnable() || suspended || !proc.isYieldAfterExecution(this.environment)) continue;
            this.yieldProcedure(proc);
            return;
        } while (reExecute);
        if (subprocs != null && !proc.isFailed()) {
            this.submitChildrenProcedures((Procedure<Env>[])subprocs);
        }
        this.releaseLock(proc, false);
        if (!suspended && proc.isFinished() && proc.hasParent()) {
            this.countDownChildren(rootProcStack, proc);
        }
    }

    private void countDownChildren(RootProcedureStack rootProcStack, Procedure<Env> proc) {
        Procedure parent = this.procedures.get(proc.getParentProcId());
        if (parent == null && rootProcStack.isRollingback()) {
            return;
        }
        if (parent != null && parent.tryRunnable()) {
            this.store.update(parent);
            this.scheduler.addFront(parent);
            LOG.info("Finished subprocedure pid={}, resume processing ppid={}", (Object)proc.getProcId(), (Object)parent.getProcId());
        }
    }

    private void submitChildrenProcedures(Procedure<Env>[] subprocs) {
        for (Procedure<Env> subproc : subprocs) {
            this.procedures.put(subproc.getProcId(), subproc);
            this.scheduler.addFront(subproc);
        }
    }

    private void updateStoreOnExecution(RootProcedureStack rootProcStack, Procedure<Env> proc, Procedure<Env>[] subprocs) {
        if (subprocs != null && !proc.isFailed()) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Stored {}, children {}", proc, (Object)Arrays.toString(subprocs));
            }
            this.store.update(subprocs);
        } else {
            LOG.debug("Store update {}", proc);
            if (proc.isFinished() && !proc.hasParent()) {
                long[] childProcIds = rootProcStack.getSubprocedureIds();
                if (childProcIds != null) {
                    this.store.delete(childProcIds);
                    for (long childProcId : childProcIds) {
                        this.procedures.remove(childProcId);
                    }
                } else {
                    this.store.update(proc);
                }
            } else {
                this.store.update(proc);
            }
        }
    }

    private Procedure<Env>[] initializeChildren(RootProcedureStack rootProcStack, Procedure<Env> proc, Procedure<Env>[] subprocs) {
        long rootProcedureId = this.getRootProcedureId(proc);
        for (int i = 0; i < subprocs.length; ++i) {
            Procedure<Env> subproc = subprocs[i];
            if (subproc == null) {
                String errMsg = "subproc[" + i + "] is null, aborting procedure";
                proc.setFailure(new ProcedureException(errMsg, new IllegalArgumentException(errMsg)));
                return null;
            }
            subproc.setParentProcId(proc.getProcId());
            subproc.setRootProcId(rootProcedureId);
            subproc.setProcId(this.nextProcId());
            subproc.setProcRunnable();
            rootProcStack.addSubProcedure(subproc);
        }
        if (!proc.isFailed()) {
            proc.setChildrenLatch(subprocs.length);
            switch (proc.getState()) {
                case RUNNABLE: {
                    proc.setState(ProcedureState.WAITING);
                    break;
                }
                case WAITING_TIMEOUT: {
                    this.timeoutExecutor.add(proc);
                    break;
                }
            }
        }
        return subprocs;
    }

    private void yieldProcedure(Procedure<Env> proc) {
        this.releaseLock(proc, false);
        this.scheduler.yield(proc);
    }

    private ProcedureLockState executeRootStackRollback(Long rootProcId, RootProcedureStack procedureStack) {
        Procedure rootProcedure = this.procedures.get(rootProcId);
        ProcedureException exception = rootProcedure.getException();
        if (exception == null) {
            exception = procedureStack.getException();
            rootProcedure.setFailure(exception);
            this.store.update(rootProcedure);
        }
        List subprocStack = procedureStack.getSubproceduresStack();
        int stackTail = subprocStack.size();
        while (stackTail-- > 0) {
            Procedure<Env> procedure = subprocStack.get(stackTail);
            if (procedure.isSuccess()) {
                subprocStack.remove(stackTail);
                this.cleanupAfterRollback(procedure);
                continue;
            }
            ProcedureLockState lockState = this.acquireLock(procedure);
            if (lockState != ProcedureLockState.LOCK_ACQUIRED) {
                return lockState;
            }
            lockState = this.executeRollback(procedure);
            this.releaseLock(procedure, false);
            boolean abortRollback = lockState != ProcedureLockState.LOCK_ACQUIRED;
            if (abortRollback |= !this.isRunning() || !this.store.isRunning()) {
                return lockState;
            }
            if (!procedure.isFinished() && procedure.isYieldAfterExecution(this.environment)) {
                return ProcedureLockState.LOCK_YIELD_WAIT;
            }
            if (procedure == rootProcedure) continue;
            this.executeCompletionCleanup(procedure);
        }
        LOG.info("Rolled back {}, time duration is {}", (Object)rootProcedure, (Object)rootProcedure.elapsedTime());
        this.rootProcedureCleanup(rootProcedure);
        return ProcedureLockState.LOCK_ACQUIRED;
    }

    private ProcedureLockState acquireLock(Procedure<Env> proc) {
        if (proc.hasLock()) {
            return ProcedureLockState.LOCK_ACQUIRED;
        }
        return proc.doAcquireLock(this.environment, this.store);
    }

    private ProcedureLockState executeRollback(Procedure<Env> procedure) {
        try {
            procedure.doRollback(this.environment);
        }
        catch (IOException e) {
            LOG.error("Roll back failed for {}", procedure, (Object)e);
        }
        catch (InterruptedException e) {
            LOG.warn("Interrupted exception occured for {}", procedure, (Object)e);
        }
        catch (Throwable t) {
            LOG.error("CODE-BUG: runtime exception for {}", procedure, (Object)t);
        }
        this.cleanupAfterRollback(procedure);
        return ProcedureLockState.LOCK_ACQUIRED;
    }

    private void cleanupAfterRollback(Procedure<Env> procedure) {
        if (procedure.removeStackIndex()) {
            if (!procedure.isSuccess()) {
                procedure.setState(ProcedureState.ROLLEDBACK);
            }
            if (procedure.hasParent()) {
                this.store.delete(procedure.getProcId());
                this.procedures.remove(procedure.getProcId());
            } else {
                long[] childProcIds = this.rollbackStack.get(procedure.getProcId()).getSubprocedureIds();
                if (childProcIds != null) {
                    this.store.delete(childProcIds);
                } else {
                    this.store.update(procedure);
                }
            }
        } else {
            this.store.update(procedure);
        }
    }

    private void executeCompletionCleanup(Procedure<Env> proc) {
        if (proc.hasLock()) {
            this.releaseLock(proc, true);
        }
        try {
            proc.completionCleanup(this.environment);
        }
        catch (Throwable e) {
            LOG.error("CODE-BUG:Uncaught runtime exception for procedure {}", proc, (Object)e);
        }
    }

    private void rootProcedureCleanup(Procedure<Env> proc) {
        this.executeCompletionCleanup(proc);
        CompletedProcedureContainer<Env> retainer = new CompletedProcedureContainer<Env>(proc);
        this.completed.put(proc.getProcId(), retainer);
        this.rollbackStack.remove(proc.getProcId());
        this.procedures.remove(proc.getProcId());
    }

    private Long getRootProcedureId(Procedure<Env> proc) {
        return Procedure.getRootProcedureId(this.procedures, proc);
    }

    private long pushProcedure(Procedure<Env> procedure) {
        long currentProcId = procedure.getProcId();
        RootProcedureStack stack = new RootProcedureStack();
        this.rollbackStack.put(currentProcId, stack);
        this.procedures.put(currentProcId, procedure);
        this.scheduler.addBack(procedure);
        return procedure.getProcId();
    }

    public int getWorkerThreadCount() {
        return this.workerThreads.size();
    }

    public boolean isRunning() {
        return this.running.get();
    }

    public void stop() {
        if (!this.running.getAndSet(false)) {
            return;
        }
        LOG.info("Stopping");
        this.scheduler.stop();
        this.timeoutExecutor.sendStopSignal();
    }

    public void join() {
        this.timeoutExecutor.awaitTermination();
        this.workerMonitorExecutor.awaitTermination();
        for (WorkerThread workerThread : this.workerThreads) {
            workerThread.awaitTermination();
        }
        try {
            this.threadGroup.destroy();
        }
        catch (IllegalThreadStateException e) {
            LOG.warn("ProcedureExecutor threadGroup {} contains running threads which are used by non-procedure module.", (Object)this.threadGroup);
            this.threadGroup.list();
        }
    }

    public boolean isStarted(long procId) {
        Procedure procedure = this.procedures.get(procId);
        if (procedure == null) {
            return this.completed.get(procId) != null;
        }
        return procedure.wasExecuted();
    }

    public boolean isFinished(long procId) {
        return !this.procedures.containsKey(procId);
    }

    public ConcurrentHashMap<Long, Procedure> getProcedures() {
        return this.procedures;
    }

    public long submitProcedure(Procedure<Env> procedure) {
        Preconditions.checkArgument((this.lastProcId.get() >= 0L ? 1 : 0) != 0);
        Preconditions.checkArgument((procedure.getState() == ProcedureState.INITIALIZING ? 1 : 0) != 0);
        Preconditions.checkArgument((!procedure.hasParent() ? 1 : 0) != 0, (String)"Unexpected parent", procedure);
        long currentProcId = this.nextProcId();
        procedure.setProcId(currentProcId);
        procedure.setProcRunnable();
        this.store.update(procedure);
        LOG.debug("{} is stored.", procedure);
        return this.pushProcedure(procedure);
    }

    public boolean abort(long procId, boolean force) {
        Procedure procedure = this.procedures.get(procId);
        if (procedure != null) {
            if (!force && procedure.wasExecuted()) {
                return false;
            }
            return procedure.abort(this.environment);
        }
        return false;
    }

    public boolean abort(long procId) {
        return this.abort(procId, true);
    }

    public Procedure<Env> getResult(long procId) {
        CompletedProcedureContainer<Env> retainer = this.completed.get(procId);
        if (retainer == null) {
            return null;
        }
        return retainer.getProcedure();
    }

    public Procedure<Env> getResultOrProcedure(long procId) {
        CompletedProcedureContainer<Env> retainer = this.completed.get(procId);
        if (retainer == null) {
            return this.procedures.get(procId);
        }
        return retainer.getProcedure();
    }

    public ProcedureScheduler getScheduler() {
        return this.scheduler;
    }

    public Env getEnvironment() {
        return this.environment;
    }

    public IProcedureStore getStore() {
        return this.store;
    }

    public RootProcedureStack<Env> getRollbackStack(long rootProcId) {
        return this.rollbackStack.get(rootProcId);
    }

    private final class WorkerMonitor
    extends InternalProcedure<Env> {
        private static final int DEFAULT_WORKER_MONITOR_INTERVAL = 5000;
        private static final int DEFAULT_WORKER_STUCK_THRESHOLD = 10000;
        private static final float DEFAULT_WORKER_ADD_STUCK_PERCENTAGE = 0.5f;

        public WorkerMonitor() {
            super(5000L);
            this.updateTimestamp();
        }

        private int checkForStuckWorkers() {
            int stuckCount = 0;
            for (WorkerThread worker : ProcedureExecutor.this.workerThreads) {
                if (worker.activeProcedure.get() == null || worker.getCurrentRunTime() < 10000L) continue;
                ++stuckCount;
                LOG.warn("Worker stuck {}, run time {} ms", (Object)worker, (Object)worker.getCurrentRunTime());
            }
            return stuckCount;
        }

        private void checkThreadCount(int stuckCount) {
            if (stuckCount < 1 || !ProcedureExecutor.this.scheduler.hasRunnables()) {
                return;
            }
            float stuckPerc = (float)stuckCount / (float)ProcedureExecutor.this.workerThreads.size();
            if (stuckPerc >= 0.5f && ProcedureExecutor.this.workerThreads.size() < ProcedureExecutor.this.maxPoolSize) {
                KeepAliveWorkerThread worker = new KeepAliveWorkerThread(ProcedureExecutor.this.threadGroup);
                ProcedureExecutor.this.workerThreads.add(worker);
                worker.start();
                LOG.debug("Added new worker thread {}", (Object)worker);
            }
        }

        @Override
        protected void periodicExecute(Env env) {
            int stuckCount = this.checkForStuckWorkers();
            this.checkThreadCount(stuckCount);
            this.updateTimestamp();
        }
    }

    private final class KeepAliveWorkerThread
    extends WorkerThread {
        public KeepAliveWorkerThread(ThreadGroup group) {
            super(group, "KAProcExecWorker-");
            this.keepAliveTime = TimeUnit.SECONDS.toMillis(10L);
        }

        @Override
        protected boolean keepAlive(long lastUpdate) {
            return System.currentTimeMillis() - lastUpdate < this.keepAliveTime;
        }
    }

    private class WorkerThread
    extends StoppableThread {
        private final AtomicLong startTime;
        private final AtomicReference<Procedure<Env>> activeProcedure;
        protected long keepAliveTime;

        public WorkerThread(ThreadGroup threadGroup) {
            this(threadGroup, "ProcExecWorker-");
        }

        public WorkerThread(ThreadGroup threadGroup, String prefix) {
            super(threadGroup, prefix + ProcedureExecutor.this.workId.incrementAndGet());
            this.startTime = new AtomicLong(Long.MAX_VALUE);
            this.activeProcedure = new AtomicReference();
            this.keepAliveTime = -1L;
            this.setDaemon(true);
        }

        @Override
        public void sendStopSignal() {
            ProcedureExecutor.this.scheduler.signalAll();
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void run() {
            long lastUpdated = System.currentTimeMillis();
            try {
                while (ProcedureExecutor.this.isRunning() && this.keepAlive(lastUpdated)) {
                    Procedure procedure = ProcedureExecutor.this.scheduler.poll(this.keepAliveTime, TimeUnit.MILLISECONDS);
                    if (procedure == null) continue;
                    this.activeProcedure.set(procedure);
                    int activeCount = ProcedureExecutor.this.activeExecutorCount.incrementAndGet();
                    this.startTime.set(System.currentTimeMillis());
                    ProcedureExecutor.this.executeProcedure(procedure);
                    activeCount = ProcedureExecutor.this.activeExecutorCount.decrementAndGet();
                    LOG.trace("Halt pid={}, activeCount={}", (Object)procedure.getProcId(), (Object)activeCount);
                    this.activeProcedure.set(null);
                    lastUpdated = System.currentTimeMillis();
                    this.startTime.set(lastUpdated);
                }
            }
            catch (Throwable throwable) {
                if (this.activeProcedure.get() != null) {
                    LOG.warn("Worker terminated {}", this.activeProcedure.get(), (Object)throwable);
                }
            }
            finally {
                LOG.debug("Worker terminated.");
            }
            ProcedureExecutor.this.workerThreads.remove(this);
        }

        protected boolean keepAlive(long lastUpdated) {
            return true;
        }

        @Override
        public String toString() {
            Procedure p = this.activeProcedure.get();
            return this.getName() + "(pid=" + (p == null ? Long.valueOf(-1L) : p.getProcId() + ")");
        }

        public long getCurrentRunTime() {
            return System.currentTimeMillis() - this.startTime.get();
        }
    }
}

