/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iotdb.tsfile.compress;

import com.github.luben.zstd.Zstd;
import java.io.IOException;
import java.nio.ByteBuffer;
import net.jpountz.lz4.LZ4Compressor;
import net.jpountz.lz4.LZ4Factory;
import net.jpountz.lz4.LZ4SafeDecompressor;
import org.apache.iotdb.tsfile.compress.ICompressor;
import org.apache.iotdb.tsfile.exception.compress.CompressionTypeNotSupportedException;
import org.apache.iotdb.tsfile.file.metadata.enums.CompressionType;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.xerial.snappy.Snappy;

public interface IUnCompressor {
    public static IUnCompressor getUnCompressor(CompressionType name) {
        if (name == null) {
            throw new CompressionTypeNotSupportedException("NULL");
        }
        switch (name) {
            case UNCOMPRESSED: {
                return new NoUnCompressor();
            }
            case SNAPPY: {
                return new SnappyUnCompressor();
            }
            case LZ4: {
                return new LZ4UnCompressor();
            }
            case GZIP: {
                return new GZIPUnCompressor();
            }
            case ZSTD: {
                return new ZstdUnCompressor();
            }
            case LZMA2: {
                return new LZMA2UnCompressor();
            }
        }
        throw new CompressionTypeNotSupportedException(name.toString());
    }

    public int getUncompressedLength(byte[] var1, int var2, int var3) throws IOException;

    public int getUncompressedLength(ByteBuffer var1) throws IOException;

    public byte[] uncompress(byte[] var1) throws IOException;

    public int uncompress(byte[] var1, int var2, int var3, byte[] var4, int var5) throws IOException;

    public int uncompress(ByteBuffer var1, ByteBuffer var2) throws IOException;

    public CompressionType getCodecName();

    public static class LZMA2UnCompressor
    implements IUnCompressor {
        @Override
        public int getUncompressedLength(byte[] array, int offset, int length) {
            throw new UnsupportedOperationException("unsupported get uncompress length");
        }

        @Override
        public int getUncompressedLength(ByteBuffer buffer) {
            throw new UnsupportedOperationException("unsupported get uncompress length");
        }

        @Override
        public byte[] uncompress(byte[] byteArray) throws IOException {
            if (null == byteArray) {
                return new byte[0];
            }
            return ICompressor.LZMA2Compress.uncompress(byteArray);
        }

        @Override
        public int uncompress(byte[] byteArray, int offset, int length, byte[] output, int outOffset) throws IOException {
            byte[] dataBefore = new byte[length];
            System.arraycopy(byteArray, offset, dataBefore, 0, length);
            byte[] res = ICompressor.LZMA2Compress.uncompress(dataBefore);
            System.arraycopy(res, 0, output, outOffset, res.length);
            return res.length;
        }

        @Override
        public int uncompress(ByteBuffer compressed, ByteBuffer uncompressed) throws IOException {
            int length = compressed.remaining();
            byte[] dataBefore = new byte[length];
            compressed.get(dataBefore, 0, length);
            byte[] res = ICompressor.LZMA2Compress.uncompress(dataBefore);
            uncompressed.put(res);
            return res.length;
        }

        @Override
        public CompressionType getCodecName() {
            return CompressionType.LZMA2;
        }
    }

    public static class ZstdUnCompressor
    implements IUnCompressor {
        @Override
        public int getUncompressedLength(byte[] array, int offset, int length) throws IOException {
            return (int)Zstd.decompressedSize((byte[])array, (int)offset, (int)length);
        }

        @Override
        public int getUncompressedLength(ByteBuffer buffer) throws IOException {
            return (int)Zstd.decompressedSize((ByteBuffer)buffer);
        }

        @Override
        public byte[] uncompress(byte[] byteArray) throws IOException {
            return Zstd.decompress((byte[])byteArray, (int)this.getUncompressedLength(byteArray, 0, byteArray.length));
        }

        @Override
        public int uncompress(byte[] byteArray, int offset, int length, byte[] output, int outOffset) throws IOException {
            return (int)Zstd.decompressByteArray((byte[])output, (int)outOffset, (int)output.length, (byte[])byteArray, (int)offset, (int)byteArray.length);
        }

        @Override
        public int uncompress(ByteBuffer compressed, ByteBuffer uncompressed) throws IOException {
            return Zstd.decompress((ByteBuffer)uncompressed, (ByteBuffer)compressed);
        }

        @Override
        public CompressionType getCodecName() {
            return CompressionType.ZSTD;
        }
    }

    public static class GZIPUnCompressor
    implements IUnCompressor {
        @Override
        public int getUncompressedLength(byte[] array, int offset, int length) {
            throw new UnsupportedOperationException("unsupported get uncompress length");
        }

        @Override
        public int getUncompressedLength(ByteBuffer buffer) {
            throw new UnsupportedOperationException("unsupported get uncompress length");
        }

        @Override
        public byte[] uncompress(byte[] byteArray) throws IOException {
            if (null == byteArray) {
                return new byte[0];
            }
            return ICompressor.GZIPCompress.uncompress(byteArray);
        }

        @Override
        public int uncompress(byte[] byteArray, int offset, int length, byte[] output, int outOffset) throws IOException {
            byte[] dataBefore = new byte[length];
            System.arraycopy(byteArray, offset, dataBefore, 0, length);
            byte[] res = ICompressor.GZIPCompress.uncompress(dataBefore);
            System.arraycopy(res, 0, output, outOffset, res.length);
            return res.length;
        }

        @Override
        public int uncompress(ByteBuffer compressed, ByteBuffer uncompressed) throws IOException {
            int length = compressed.remaining();
            byte[] dataBefore = new byte[length];
            compressed.get(dataBefore, 0, length);
            byte[] res = ICompressor.GZIPCompress.uncompress(dataBefore);
            uncompressed.put(res);
            return res.length;
        }

        @Override
        public CompressionType getCodecName() {
            return CompressionType.GZIP;
        }
    }

    public static class LZ4UnCompressor
    implements IUnCompressor {
        private static final Logger logger = LoggerFactory.getLogger(LZ4Compressor.class);
        private static final String UNCOMPRESS_INPUT_ERROR = "tsfile-compression LZ4UnCompressor: errors occurs when uncompress input byte";
        private static final int MAX_COMPRESS_RATIO = 255;
        private LZ4SafeDecompressor decompressor;

        public LZ4UnCompressor() {
            LZ4Factory factory = LZ4Factory.fastestInstance();
            this.decompressor = factory.safeDecompressor();
        }

        @Override
        public int getUncompressedLength(byte[] array, int offset, int length) {
            throw new UnsupportedOperationException("unsupported get uncompress length");
        }

        @Override
        public int getUncompressedLength(ByteBuffer buffer) {
            throw new UnsupportedOperationException("unsupported get uncompress length");
        }

        @Override
        public byte[] uncompress(byte[] bytes) throws IOException {
            if (bytes == null) {
                return new byte[0];
            }
            try {
                return this.decompressor.decompress(bytes, 255 * bytes.length);
            }
            catch (RuntimeException e) {
                logger.error(UNCOMPRESS_INPUT_ERROR, (Throwable)e);
                throw new IOException(e);
            }
        }

        @Override
        public int uncompress(byte[] byteArray, int offset, int length, byte[] output, int outOffset) throws IOException {
            try {
                return this.decompressor.decompress(byteArray, offset, length, output, offset);
            }
            catch (RuntimeException e) {
                logger.error(UNCOMPRESS_INPUT_ERROR, (Throwable)e);
                throw new IOException(e);
            }
        }

        @Override
        public int uncompress(ByteBuffer compressed, ByteBuffer uncompressed) throws IOException {
            if (compressed == null || !compressed.hasRemaining()) {
                return 0;
            }
            try {
                this.decompressor.decompress(compressed, uncompressed);
                return compressed.limit();
            }
            catch (RuntimeException e) {
                logger.error(UNCOMPRESS_INPUT_ERROR, (Throwable)e);
                throw new IOException(e);
            }
        }

        @Override
        public CompressionType getCodecName() {
            return CompressionType.LZ4;
        }
    }

    public static class SnappyUnCompressor
    implements IUnCompressor {
        private static final Logger logger = LoggerFactory.getLogger(SnappyUnCompressor.class);

        @Override
        public int getUncompressedLength(byte[] array, int offset, int length) throws IOException {
            return Snappy.uncompressedLength((byte[])array, (int)offset, (int)length);
        }

        @Override
        public int getUncompressedLength(ByteBuffer buffer) throws IOException {
            return Snappy.uncompressedLength((ByteBuffer)buffer);
        }

        @Override
        public byte[] uncompress(byte[] bytes) {
            if (bytes == null) {
                return new byte[0];
            }
            try {
                return Snappy.uncompress((byte[])bytes);
            }
            catch (IOException e) {
                logger.error("tsfile-compression SnappyUnCompressor: errors occurs when uncompress input byte", (Throwable)e);
                return new byte[0];
            }
        }

        @Override
        public int uncompress(byte[] byteArray, int offset, int length, byte[] output, int outOffset) throws IOException {
            return Snappy.uncompress((byte[])byteArray, (int)offset, (int)length, (byte[])output, (int)outOffset);
        }

        @Override
        public int uncompress(ByteBuffer compressed, ByteBuffer uncompressed) {
            if (compressed == null || !compressed.hasRemaining()) {
                return 0;
            }
            try {
                return Snappy.uncompress((ByteBuffer)compressed, (ByteBuffer)uncompressed);
            }
            catch (IOException e) {
                logger.error("tsfile-compression SnappyUnCompressor: errors occurs when uncompress input byte", (Throwable)e);
                return 0;
            }
        }

        @Override
        public CompressionType getCodecName() {
            return CompressionType.SNAPPY;
        }
    }

    public static class NoUnCompressor
    implements IUnCompressor {
        @Override
        public int getUncompressedLength(byte[] array, int offset, int length) {
            return length;
        }

        @Override
        public int getUncompressedLength(ByteBuffer buffer) {
            return buffer.remaining();
        }

        @Override
        public byte[] uncompress(byte[] byteArray) {
            return byteArray;
        }

        @Override
        public int uncompress(byte[] byteArray, int offset, int length, byte[] output, int outOffset) {
            System.arraycopy(byteArray, offset, output, outOffset, length);
            return length;
        }

        @Override
        public int uncompress(ByteBuffer compressed, ByteBuffer uncompressed) throws IOException {
            throw new IOException("NoUnCompressor does not support this method.");
        }

        @Override
        public CompressionType getCodecName() {
            return CompressionType.UNCOMPRESSED;
        }
    }
}

