/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iotdb.tsfile.write.record;

import java.io.DataOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import org.apache.iotdb.tsfile.exception.write.UnSupportedDataTypeException;
import org.apache.iotdb.tsfile.file.metadata.enums.TSDataType;
import org.apache.iotdb.tsfile.utils.Binary;
import org.apache.iotdb.tsfile.utils.BitMap;
import org.apache.iotdb.tsfile.utils.BytesUtils;
import org.apache.iotdb.tsfile.utils.PublicBAOS;
import org.apache.iotdb.tsfile.utils.ReadWriteIOUtils;
import org.apache.iotdb.tsfile.write.schema.MeasurementSchema;

public class Tablet {
    private static final int DEFAULT_SIZE = 1024;
    private static final String NOT_SUPPORT_DATATYPE = "Data type %s is not supported.";
    public String deviceId;
    private List<MeasurementSchema> schemas;
    private final Map<String, Integer> measurementIndex;
    public long[] timestamps;
    public Object[] values;
    public BitMap[] bitMaps;
    public int rowSize;
    private final int maxRowNumber;

    public Tablet(String deviceId, List<MeasurementSchema> schemas) {
        this(deviceId, schemas, 1024);
    }

    public Tablet(String deviceId, List<MeasurementSchema> schemas, int maxRowNumber) {
        this.deviceId = deviceId;
        this.schemas = new ArrayList<MeasurementSchema>(schemas);
        this.maxRowNumber = maxRowNumber;
        this.measurementIndex = new HashMap<String, Integer>();
        this.constructMeasurementIndexMap();
        this.createColumns();
        this.reset();
    }

    public Tablet(String deviceId, List<MeasurementSchema> schemas, long[] timestamps, Object[] values, BitMap[] bitMaps, int maxRowNumber) {
        this.deviceId = deviceId;
        this.schemas = schemas;
        this.timestamps = timestamps;
        this.values = values;
        this.bitMaps = bitMaps;
        this.maxRowNumber = maxRowNumber;
        this.rowSize = maxRowNumber;
        this.measurementIndex = new HashMap<String, Integer>();
        this.constructMeasurementIndexMap();
    }

    private void constructMeasurementIndexMap() {
        int indexInSchema = 0;
        for (MeasurementSchema schema : this.schemas) {
            this.measurementIndex.put(schema.getMeasurementId(), indexInSchema);
            ++indexInSchema;
        }
    }

    public void setDeviceId(String deviceId) {
        this.deviceId = deviceId;
    }

    public void setSchemas(List<MeasurementSchema> schemas) {
        this.schemas = schemas;
    }

    public void initBitMaps() {
        this.bitMaps = new BitMap[this.schemas.size()];
        for (int column = 0; column < this.schemas.size(); ++column) {
            this.bitMaps[column] = new BitMap(this.getMaxRowNumber());
        }
    }

    public void addTimestamp(int rowIndex, long timestamp) {
        this.timestamps[rowIndex] = timestamp;
    }

    public void addValue(String measurementId, int rowIndex, Object value) {
        int indexOfSchema = this.measurementIndex.get(measurementId);
        MeasurementSchema measurementSchema = this.schemas.get(indexOfSchema);
        this.addValueOfDataType(measurementSchema.getType(), rowIndex, indexOfSchema, value);
    }

    private void addValueOfDataType(TSDataType dataType, int rowIndex, int indexOfSchema, Object value) {
        if (value == null) {
            if (this.bitMaps == null) {
                this.bitMaps = new BitMap[this.values.length];
            }
            if (this.bitMaps[indexOfSchema] == null) {
                this.bitMaps[indexOfSchema] = new BitMap(this.maxRowNumber);
            }
            this.bitMaps[indexOfSchema].mark(rowIndex);
        }
        switch (dataType) {
            case TEXT: {
                Binary[] sensor = (Binary[])this.values[indexOfSchema];
                if (value instanceof Binary) {
                    sensor[rowIndex] = (Binary)value;
                    break;
                }
                sensor[rowIndex] = value != null ? new Binary((String)value) : Binary.EMPTY_VALUE;
                break;
            }
            case FLOAT: {
                float[] sensor = (float[])this.values[indexOfSchema];
                sensor[rowIndex] = value != null ? ((Float)value).floatValue() : Float.MIN_VALUE;
                break;
            }
            case INT32: {
                int[] sensor = (int[])this.values[indexOfSchema];
                sensor[rowIndex] = value != null ? (Integer)value : Integer.MIN_VALUE;
                break;
            }
            case INT64: {
                long[] sensor = (long[])this.values[indexOfSchema];
                sensor[rowIndex] = value != null ? (Long)value : Long.MIN_VALUE;
                break;
            }
            case DOUBLE: {
                double[] sensor = (double[])this.values[indexOfSchema];
                sensor[rowIndex] = value != null ? (Double)value : Double.MIN_VALUE;
                break;
            }
            case BOOLEAN: {
                boolean[] sensor = (boolean[])this.values[indexOfSchema];
                sensor[rowIndex] = value != null && (Boolean)value != false;
                break;
            }
            default: {
                throw new UnSupportedDataTypeException(String.format(NOT_SUPPORT_DATATYPE, new Object[]{dataType}));
            }
        }
    }

    public List<MeasurementSchema> getSchemas() {
        return this.schemas;
    }

    public int getMaxRowNumber() {
        return this.maxRowNumber;
    }

    public void reset() {
        this.rowSize = 0;
        if (this.bitMaps != null) {
            for (BitMap bitMap : this.bitMaps) {
                if (bitMap == null) continue;
                bitMap.reset();
            }
        }
    }

    private void createColumns() {
        this.timestamps = new long[this.maxRowNumber];
        int valueColumnsSize = this.schemas.size();
        this.values = new Object[valueColumnsSize];
        int columnIndex = 0;
        for (MeasurementSchema schema : this.schemas) {
            TSDataType dataType = schema.getType();
            this.values[columnIndex] = this.createValueColumnOfDataType(dataType);
            ++columnIndex;
        }
    }

    private Object createValueColumnOfDataType(TSDataType dataType) {
        Object[] valueColumn;
        switch (dataType) {
            case INT32: {
                valueColumn = new int[this.maxRowNumber];
                break;
            }
            case INT64: {
                valueColumn = new long[this.maxRowNumber];
                break;
            }
            case FLOAT: {
                valueColumn = new float[this.maxRowNumber];
                break;
            }
            case DOUBLE: {
                valueColumn = new double[this.maxRowNumber];
                break;
            }
            case BOOLEAN: {
                valueColumn = new boolean[this.maxRowNumber];
                break;
            }
            case TEXT: {
                valueColumn = new Binary[this.maxRowNumber];
                break;
            }
            default: {
                throw new UnSupportedDataTypeException(String.format(NOT_SUPPORT_DATATYPE, new Object[]{dataType}));
            }
        }
        return valueColumn;
    }

    public int getTimeBytesSize() {
        return this.rowSize * 8;
    }

    public int getTotalValueOccupation() {
        int valueOccupation = 0;
        int columnIndex = 0;
        for (MeasurementSchema schema : this.schemas) {
            valueOccupation += this.calOccupationOfOneColumn(schema.getType(), columnIndex);
            ++columnIndex;
        }
        if (this.bitMaps != null) {
            for (BitMap bitMap : this.bitMaps) {
                ++valueOccupation;
                if (bitMap == null || bitMap.isAllUnmarked()) continue;
                valueOccupation += this.rowSize / 8 + 1;
            }
        }
        return valueOccupation;
    }

    private int calOccupationOfOneColumn(TSDataType dataType, int columnIndex) {
        int valueOccupation = 0;
        switch (dataType) {
            case BOOLEAN: {
                valueOccupation += this.rowSize;
                break;
            }
            case FLOAT: 
            case INT32: {
                valueOccupation += this.rowSize * 4;
                break;
            }
            case INT64: 
            case DOUBLE: {
                valueOccupation += this.rowSize * 8;
                break;
            }
            case TEXT: {
                valueOccupation += this.rowSize * 4;
                Binary[] binaries = (Binary[])this.values[columnIndex];
                for (int rowIndex = 0; rowIndex < this.rowSize; ++rowIndex) {
                    valueOccupation += binaries[rowIndex].getLength();
                }
                break;
            }
            default: {
                throw new UnSupportedDataTypeException(String.format(NOT_SUPPORT_DATATYPE, new Object[]{dataType}));
            }
        }
        return valueOccupation;
    }

    public ByteBuffer serialize() throws IOException {
        PublicBAOS byteArrayOutputStream = new PublicBAOS();
        DataOutputStream outputStream = new DataOutputStream(byteArrayOutputStream);
        this.serialize(outputStream);
        return ByteBuffer.wrap(byteArrayOutputStream.getBuf(), 0, byteArrayOutputStream.size());
    }

    public void serialize(DataOutputStream stream) throws IOException {
        ReadWriteIOUtils.write(this.deviceId, (OutputStream)stream);
        ReadWriteIOUtils.write(this.rowSize, (OutputStream)stream);
        this.writeMeasurementSchemas(stream);
        this.writeTimes(stream);
        this.writeBitMaps(stream);
        this.writeValues(stream);
    }

    private void writeMeasurementSchemas(DataOutputStream stream) throws IOException {
        ReadWriteIOUtils.write(this.schemas.size(), (OutputStream)stream);
        for (MeasurementSchema schema : this.schemas) {
            if (schema == null) {
                ReadWriteIOUtils.write(BytesUtils.boolToByte(false), (OutputStream)stream);
                continue;
            }
            ReadWriteIOUtils.write(BytesUtils.boolToByte(true), (OutputStream)stream);
            schema.serializeTo(stream);
        }
    }

    private void writeTimes(DataOutputStream stream) throws IOException {
        for (long time : this.timestamps) {
            ReadWriteIOUtils.write(time, (OutputStream)stream);
        }
    }

    private void writeBitMaps(DataOutputStream stream) throws IOException {
        ReadWriteIOUtils.write(BytesUtils.boolToByte(this.bitMaps != null), (OutputStream)stream);
        if (this.bitMaps != null) {
            for (BitMap bitMap : this.bitMaps) {
                if (bitMap == null) {
                    ReadWriteIOUtils.write(BytesUtils.boolToByte(false), (OutputStream)stream);
                    continue;
                }
                ReadWriteIOUtils.write(BytesUtils.boolToByte(true), (OutputStream)stream);
                stream.write(bitMap.getByteArray());
            }
        }
    }

    private void writeValues(DataOutputStream stream) throws IOException {
        for (int i = 0; i < this.values.length; ++i) {
            this.serializeColumn(this.schemas.get(i).getType(), this.values[i], stream);
        }
    }

    private void serializeColumn(TSDataType dataType, Object column, DataOutputStream stream) throws IOException {
        switch (dataType) {
            case INT32: {
                int[] intValues = (int[])column;
                for (int j = 0; j < this.rowSize; ++j) {
                    ReadWriteIOUtils.write(intValues[j], (OutputStream)stream);
                }
                break;
            }
            case INT64: {
                long[] longValues = (long[])column;
                for (int j = 0; j < this.rowSize; ++j) {
                    ReadWriteIOUtils.write(longValues[j], (OutputStream)stream);
                }
                break;
            }
            case FLOAT: {
                float[] floatValues = (float[])column;
                for (int j = 0; j < this.rowSize; ++j) {
                    ReadWriteIOUtils.write(floatValues[j], (OutputStream)stream);
                }
                break;
            }
            case DOUBLE: {
                double[] doubleValues = (double[])column;
                for (int j = 0; j < this.rowSize; ++j) {
                    ReadWriteIOUtils.write(doubleValues[j], (OutputStream)stream);
                }
                break;
            }
            case BOOLEAN: {
                boolean[] boolValues = (boolean[])column;
                for (int j = 0; j < this.rowSize; ++j) {
                    ReadWriteIOUtils.write(BytesUtils.boolToByte(boolValues[j]), (OutputStream)stream);
                }
                break;
            }
            case TEXT: {
                Binary[] binaryValues = (Binary[])column;
                for (int j = 0; j < this.rowSize; ++j) {
                    ReadWriteIOUtils.write(binaryValues[j], (OutputStream)stream);
                }
                break;
            }
            default: {
                throw new UnSupportedDataTypeException(String.format(NOT_SUPPORT_DATATYPE, new Object[]{dataType}));
            }
        }
    }

    public static Tablet deserialize(ByteBuffer byteBuffer) {
        String deviceId = ReadWriteIOUtils.readString(byteBuffer);
        int rowSize = ReadWriteIOUtils.readInt(byteBuffer);
        int schemaSize = ReadWriteIOUtils.readInt(byteBuffer);
        ArrayList<MeasurementSchema> schemas = new ArrayList<MeasurementSchema>();
        for (int i = 0; i < schemaSize; ++i) {
            boolean hasSchema = BytesUtils.byteToBool(byteBuffer.get());
            if (!hasSchema) continue;
            schemas.add(MeasurementSchema.deserializeFrom(byteBuffer));
        }
        long[] times = new long[rowSize];
        for (int i = 0; i < rowSize; ++i) {
            times[i] = ReadWriteIOUtils.readLong(byteBuffer);
        }
        boolean hasBitMaps = BytesUtils.byteToBool(byteBuffer.get());
        BitMap[] bitMaps = null;
        if (hasBitMaps) {
            bitMaps = Tablet.readBitMapsFromBuffer(byteBuffer, schemaSize, rowSize);
        }
        TSDataType[] dataTypes = (TSDataType[])schemas.stream().map(MeasurementSchema::getType).toArray(TSDataType[]::new);
        Object[] values = Tablet.readTabletValuesFromBuffer(byteBuffer, dataTypes, schemaSize, rowSize);
        Tablet tablet = new Tablet(deviceId, schemas, times, values, bitMaps, rowSize);
        tablet.constructMeasurementIndexMap();
        return tablet;
    }

    public static BitMap[] readBitMapsFromBuffer(ByteBuffer buffer, int columns, int size) {
        if (!buffer.hasRemaining()) {
            return null;
        }
        BitMap[] bitMaps = new BitMap[columns];
        for (int i = 0; i < columns; ++i) {
            boolean hasBitMap = BytesUtils.byteToBool(buffer.get());
            if (!hasBitMap) continue;
            byte[] bytes = new byte[size / 8 + 1];
            for (int j = 0; j < bytes.length; ++j) {
                bytes[j] = buffer.get();
            }
            bitMaps[i] = new BitMap(size, bytes);
        }
        return bitMaps;
    }

    public static Object[] readTabletValuesFromBuffer(ByteBuffer buffer, TSDataType[] types, int columns, int size) {
        Object[] values = new Object[columns];
        block8: for (int i = 0; i < columns; ++i) {
            switch (types[i]) {
                case BOOLEAN: {
                    boolean[] boolValues = new boolean[size];
                    for (int index = 0; index < size; ++index) {
                        boolValues[index] = BytesUtils.byteToBool(buffer.get());
                    }
                    values[i] = boolValues;
                    continue block8;
                }
                case INT32: {
                    int[] intValues = new int[size];
                    for (int index = 0; index < size; ++index) {
                        intValues[index] = buffer.getInt();
                    }
                    values[i] = intValues;
                    continue block8;
                }
                case INT64: {
                    long[] longValues = new long[size];
                    for (int index = 0; index < size; ++index) {
                        longValues[index] = buffer.getLong();
                    }
                    values[i] = longValues;
                    continue block8;
                }
                case FLOAT: {
                    float[] floatValues = new float[size];
                    for (int index = 0; index < size; ++index) {
                        floatValues[index] = buffer.getFloat();
                    }
                    values[i] = floatValues;
                    continue block8;
                }
                case DOUBLE: {
                    double[] doubleValues = new double[size];
                    for (int index = 0; index < size; ++index) {
                        doubleValues[index] = buffer.getDouble();
                    }
                    values[i] = doubleValues;
                    continue block8;
                }
                case TEXT: {
                    Binary[] binaryValues = new Binary[size];
                    for (int index = 0; index < size; ++index) {
                        int binarySize = buffer.getInt();
                        byte[] binaryValue = new byte[binarySize];
                        buffer.get(binaryValue);
                        binaryValues[index] = new Binary(binaryValue);
                    }
                    values[i] = binaryValues;
                    continue block8;
                }
                default: {
                    throw new UnSupportedDataTypeException(String.format("data type %s is not supported when convert data at client", new Object[]{types[i]}));
                }
            }
        }
        return values;
    }

    public boolean equals(Object o) {
        boolean flag;
        if (this == o) {
            return true;
        }
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        Tablet that = (Tablet)o;
        boolean bl = flag = that.rowSize == this.rowSize && Arrays.equals(that.timestamps, this.timestamps) && Arrays.equals(that.bitMaps, this.bitMaps) && Objects.equals(that.schemas, this.schemas) && Objects.equals(that.measurementIndex, this.measurementIndex) && Objects.equals(that.deviceId, this.deviceId);
        if (!flag) {
            return false;
        }
        Object[] thatValues = that.values;
        if (thatValues.length != this.values.length) {
            return false;
        }
        int n = this.values.length;
        block8: for (int i = 0; i < n; ++i) {
            switch (this.schemas.get(i).getType()) {
                case INT32: {
                    if (Arrays.equals((int[])thatValues[i], (int[])this.values[i])) continue block8;
                    return false;
                }
                case INT64: {
                    if (Arrays.equals((long[])thatValues[i], (long[])this.values[i])) continue block8;
                    return false;
                }
                case FLOAT: {
                    if (Arrays.equals((float[])thatValues[i], (float[])this.values[i])) continue block8;
                    return false;
                }
                case DOUBLE: {
                    if (Arrays.equals((double[])thatValues[i], (double[])this.values[i])) continue block8;
                    return false;
                }
                case BOOLEAN: {
                    if (Arrays.equals((boolean[])thatValues[i], (boolean[])this.values[i])) continue block8;
                    return false;
                }
                case TEXT: {
                    if (Arrays.equals((Binary[])thatValues[i], (Binary[])this.values[i])) continue block8;
                    return false;
                }
                default: {
                    throw new UnSupportedDataTypeException(String.format(NOT_SUPPORT_DATATYPE, new Object[]{this.schemas.get(i).getType()}));
                }
            }
        }
        return true;
    }
}

