/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iotdb.db.engine.storagegroup.timeindex;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import org.apache.iotdb.commons.exception.IllegalPathException;
import org.apache.iotdb.commons.path.PartialPath;
import org.apache.iotdb.commons.utils.SerializeUtils;
import org.apache.iotdb.db.engine.StorageEngine;
import org.apache.iotdb.db.engine.storagegroup.TsFileResource;
import org.apache.iotdb.db.engine.storagegroup.timeindex.FileTimeIndex;
import org.apache.iotdb.db.engine.storagegroup.timeindex.ITimeIndex;
import org.apache.iotdb.db.exception.PartitionViolationException;
import org.apache.iotdb.tsfile.utils.FilePathUtils;
import org.apache.iotdb.tsfile.utils.Pair;
import org.apache.iotdb.tsfile.utils.RamUsageEstimator;
import org.apache.iotdb.tsfile.utils.ReadWriteIOUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DeviceTimeIndex
implements ITimeIndex {
    private static final Logger logger = LoggerFactory.getLogger(DeviceTimeIndex.class);
    public static final int INIT_ARRAY_SIZE = 64;
    protected long[] startTimes;
    protected long[] endTimes;
    private long minStartTime = Long.MAX_VALUE;
    private long maxEndTime = Long.MIN_VALUE;
    protected Map<String, Integer> deviceToIndex;

    public DeviceTimeIndex() {
        this.deviceToIndex = new ConcurrentHashMap<String, Integer>();
        this.startTimes = new long[64];
        this.endTimes = new long[64];
        this.initTimes(this.startTimes, Long.MAX_VALUE);
        this.initTimes(this.endTimes, Long.MIN_VALUE);
    }

    public DeviceTimeIndex(Map<String, Integer> deviceToIndex, long[] startTimes, long[] endTimes) {
        this.startTimes = startTimes;
        this.endTimes = endTimes;
        this.deviceToIndex = deviceToIndex;
    }

    @Override
    public void serialize(OutputStream outputStream) throws IOException {
        ReadWriteIOUtils.write((byte)this.getTimeIndexType(), (OutputStream)outputStream);
        int deviceNum = this.deviceToIndex.size();
        ReadWriteIOUtils.write((int)deviceNum, (OutputStream)outputStream);
        for (int i = 0; i < deviceNum; ++i) {
            ReadWriteIOUtils.write((long)this.startTimes[i], (OutputStream)outputStream);
            ReadWriteIOUtils.write((long)this.endTimes[i], (OutputStream)outputStream);
        }
        for (Map.Entry<String, Integer> stringIntegerEntry : this.deviceToIndex.entrySet()) {
            String deviceName = stringIntegerEntry.getKey();
            int index = stringIntegerEntry.getValue();
            ReadWriteIOUtils.write((String)deviceName, (OutputStream)outputStream);
            ReadWriteIOUtils.write((int)index, (OutputStream)outputStream);
        }
    }

    @Override
    public DeviceTimeIndex deserialize(InputStream inputStream) throws IOException {
        int i;
        int deviceNum = ReadWriteIOUtils.readInt((InputStream)inputStream);
        this.startTimes = new long[deviceNum];
        this.endTimes = new long[deviceNum];
        for (i = 0; i < deviceNum; ++i) {
            this.startTimes[i] = ReadWriteIOUtils.readLong((InputStream)inputStream);
            this.endTimes[i] = ReadWriteIOUtils.readLong((InputStream)inputStream);
            this.minStartTime = Math.min(this.minStartTime, this.startTimes[i]);
            this.maxEndTime = Math.max(this.maxEndTime, this.endTimes[i]);
        }
        for (i = 0; i < deviceNum; ++i) {
            String path = ReadWriteIOUtils.readString((InputStream)inputStream).intern();
            int index = ReadWriteIOUtils.readInt((InputStream)inputStream);
            this.deviceToIndex.put(path, index);
        }
        return this;
    }

    @Override
    public DeviceTimeIndex deserialize(ByteBuffer buffer) {
        int i;
        int deviceNum = buffer.getInt();
        this.startTimes = new long[deviceNum];
        this.endTimes = new long[deviceNum];
        for (i = 0; i < deviceNum; ++i) {
            this.startTimes[i] = buffer.getLong();
            this.endTimes[i] = buffer.getLong();
            this.minStartTime = Math.min(this.minStartTime, this.startTimes[i]);
            this.maxEndTime = Math.max(this.maxEndTime, this.endTimes[i]);
        }
        for (i = 0; i < deviceNum; ++i) {
            String path = SerializeUtils.deserializeString((ByteBuffer)buffer).intern();
            int index = buffer.getInt();
            this.deviceToIndex.put(path, index);
        }
        return this;
    }

    @Override
    public void close() {
        this.startTimes = Arrays.copyOfRange(this.startTimes, 0, this.deviceToIndex.size());
        this.endTimes = Arrays.copyOfRange(this.endTimes, 0, this.deviceToIndex.size());
    }

    public Set<String> getDevices() {
        return this.deviceToIndex.keySet();
    }

    @Override
    public Set<String> getDevices(String tsFilePath, TsFileResource tsFileResource) {
        return this.deviceToIndex.keySet();
    }

    public static Set<String> getDevices(InputStream inputStream) throws IOException {
        int deviceNum = ReadWriteIOUtils.readInt((InputStream)inputStream);
        ReadWriteIOUtils.skip((InputStream)inputStream, (long)(2L * (long)deviceNum * 8L));
        HashSet<String> devices = new HashSet<String>();
        for (int i = 0; i < deviceNum; ++i) {
            String path = ReadWriteIOUtils.readString((InputStream)inputStream).intern();
            ReadWriteIOUtils.skip((InputStream)inputStream, (long)4L);
            devices.add(path);
        }
        return devices;
    }

    @Override
    public boolean endTimeEmpty() {
        for (long endTime : this.endTimes) {
            if (endTime == Long.MIN_VALUE) continue;
            return false;
        }
        return true;
    }

    @Override
    public boolean stillLives(long ttlLowerBound) {
        if (ttlLowerBound == Long.MAX_VALUE) {
            return true;
        }
        for (long endTime : this.endTimes) {
            if (endTime < ttlLowerBound) continue;
            return true;
        }
        return false;
    }

    @Override
    public long calculateRamSize() {
        return RamUsageEstimator.sizeOf(this.deviceToIndex) + RamUsageEstimator.sizeOf((long[])this.startTimes) + RamUsageEstimator.sizeOf((long[])this.endTimes);
    }

    private int getDeviceIndex(String deviceId) {
        int index;
        if (this.deviceToIndex.containsKey(deviceId)) {
            index = this.deviceToIndex.get(deviceId);
        } else {
            index = this.deviceToIndex.size();
            this.deviceToIndex.put(deviceId.intern(), index);
            if (this.startTimes.length <= index) {
                this.startTimes = this.enLargeArray(this.startTimes, Long.MAX_VALUE);
                this.endTimes = this.enLargeArray(this.endTimes, Long.MIN_VALUE);
            }
        }
        return index;
    }

    private void initTimes(long[] times, long defaultTime) {
        Arrays.fill(times, defaultTime);
    }

    private long[] enLargeArray(long[] array, long defaultValue) {
        long[] tmp = new long[array.length * 2];
        this.initTimes(tmp, defaultValue);
        System.arraycopy(array, 0, tmp, 0, array.length);
        return tmp;
    }

    @Override
    public long getTimePartition(String tsFilePath) {
        try {
            if (this.deviceToIndex != null && !this.deviceToIndex.isEmpty()) {
                return StorageEngine.getTimePartition(this.startTimes[this.deviceToIndex.values().iterator().next()]);
            }
            String[] filePathSplits = FilePathUtils.splitTsFilePath((String)tsFilePath);
            return Long.parseLong(filePathSplits[filePathSplits.length - 2]);
        }
        catch (NumberFormatException e) {
            return 0L;
        }
    }

    private long getTimePartitionWithCheck() {
        long partitionId = -1L;
        for (int index : this.deviceToIndex.values()) {
            long p = StorageEngine.getTimePartition(this.startTimes[index]);
            if (partitionId == -1L) {
                partitionId = p;
            } else if (partitionId != p) {
                return -1L;
            }
            if (partitionId == (p = StorageEngine.getTimePartition(this.endTimes[index]))) continue;
            return -1L;
        }
        return partitionId;
    }

    @Override
    public long getTimePartitionWithCheck(String tsFilePath) throws PartitionViolationException {
        long partitionId = this.getTimePartitionWithCheck();
        if (partitionId == -1L) {
            throw new PartitionViolationException(tsFilePath);
        }
        return partitionId;
    }

    @Override
    public boolean isSpanMultiTimePartitions() {
        long partitionId = this.getTimePartitionWithCheck();
        return partitionId == -1L;
    }

    @Override
    public void updateStartTime(String deviceId, long time) {
        long startTime = this.getStartTime(deviceId);
        if (time < startTime) {
            int index = this.getDeviceIndex(deviceId);
            this.startTimes[index] = time;
        }
        this.minStartTime = Math.min(this.minStartTime, time);
    }

    @Override
    public void updateEndTime(String deviceId, long time) {
        long endTime = this.getEndTime(deviceId);
        if (time > endTime) {
            int index = this.getDeviceIndex(deviceId);
            this.endTimes[index] = time;
        }
        this.maxEndTime = Math.max(this.maxEndTime, time);
    }

    @Override
    public void putStartTime(String deviceId, long time) {
        int index = this.getDeviceIndex(deviceId);
        this.startTimes[index] = time;
        this.minStartTime = Math.min(this.minStartTime, time);
    }

    @Override
    public void putEndTime(String deviceId, long time) {
        int index = this.getDeviceIndex(deviceId);
        this.endTimes[index] = time;
        this.maxEndTime = Math.max(this.maxEndTime, time);
    }

    @Override
    public long getStartTime(String deviceId) {
        if (!this.deviceToIndex.containsKey(deviceId)) {
            return Long.MAX_VALUE;
        }
        return this.startTimes[this.deviceToIndex.get(deviceId)];
    }

    @Override
    public long getEndTime(String deviceId) {
        if (!this.deviceToIndex.containsKey(deviceId)) {
            return Long.MIN_VALUE;
        }
        return this.endTimes[this.deviceToIndex.get(deviceId)];
    }

    @Override
    public boolean checkDeviceIdExist(String deviceId) {
        return this.deviceToIndex.containsKey(deviceId);
    }

    @Override
    public long getMinStartTime() {
        return this.minStartTime;
    }

    @Override
    public long getMaxEndTime() {
        return this.maxEndTime;
    }

    @Override
    public int compareDegradePriority(ITimeIndex timeIndex) {
        if (timeIndex instanceof DeviceTimeIndex) {
            return Long.compare(this.getMinStartTime(), timeIndex.getMinStartTime());
        }
        if (timeIndex instanceof FileTimeIndex) {
            return -1;
        }
        logger.error("Wrong timeIndex type {}", (Object)timeIndex.getClass().getName());
        throw new RuntimeException("Wrong timeIndex type " + timeIndex.getClass().getName());
    }

    @Override
    public boolean mayContainsDevice(String device) {
        return this.deviceToIndex.containsKey(device);
    }

    @Override
    public long[] getStartAndEndTime(String deviceId) {
        Integer index = this.deviceToIndex.get(deviceId);
        if (index == null) {
            return null;
        }
        int i = index;
        return new long[]{this.startTimes[i], this.endTimes[i]};
    }

    @Override
    public Pair<Long, Long> getPossibleStartTimeAndEndTime(PartialPath devicePattern) {
        boolean hasMatchedDevice = false;
        long startTime = Long.MAX_VALUE;
        long endTime = Long.MIN_VALUE;
        for (Map.Entry<String, Integer> entry : this.deviceToIndex.entrySet()) {
            try {
                if (!devicePattern.matchFullPath(new PartialPath(entry.getKey()))) continue;
                hasMatchedDevice = true;
                if (this.startTimes[entry.getValue()] < startTime) {
                    startTime = this.startTimes[entry.getValue()];
                }
                if (this.endTimes[entry.getValue()] <= endTime) continue;
                endTime = this.endTimes[entry.getValue()];
            }
            catch (IllegalPathException illegalPathException) {}
        }
        return hasMatchedDevice ? new Pair((Object)startTime, (Object)endTime) : null;
    }

    @Override
    public byte getTimeIndexType() {
        return 1;
    }
}

