/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iotdb.consensus.ratis;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.StandardCopyOption;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import org.apache.iotdb.common.rpc.thrift.TSStatus;
import org.apache.iotdb.consensus.IStateMachine;
import org.apache.iotdb.consensus.common.DataSet;
import org.apache.iotdb.consensus.common.request.ByteBufferConsensusRequest;
import org.apache.iotdb.consensus.common.request.IConsensusRequest;
import org.apache.iotdb.consensus.ratis.RequestMessage;
import org.apache.iotdb.consensus.ratis.ResponseMessage;
import org.apache.iotdb.consensus.ratis.SnapshotStorage;
import org.apache.iotdb.consensus.ratis.Utils;
import org.apache.ratis.proto.RaftProtos;
import org.apache.ratis.protocol.Message;
import org.apache.ratis.protocol.RaftGroupId;
import org.apache.ratis.protocol.RaftGroupMemberId;
import org.apache.ratis.protocol.RaftPeerId;
import org.apache.ratis.server.RaftServer;
import org.apache.ratis.server.protocol.TermIndex;
import org.apache.ratis.server.storage.RaftStorage;
import org.apache.ratis.statemachine.StateMachineStorage;
import org.apache.ratis.statemachine.TransactionContext;
import org.apache.ratis.statemachine.impl.BaseStateMachine;
import org.apache.ratis.util.FileUtils;
import org.apache.ratis.util.LifeCycle;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ApplicationStateMachineProxy
extends BaseStateMachine {
    private final Logger logger = LoggerFactory.getLogger(ApplicationStateMachineProxy.class);
    private final IStateMachine applicationStateMachine;
    private final IStateMachine.RetryPolicy retryPolicy;
    private File statemachineDir;
    private final SnapshotStorage snapshotStorage;
    private final RaftGroupId groupId;

    public ApplicationStateMachineProxy(IStateMachine stateMachine, RaftGroupId id) {
        this.applicationStateMachine = stateMachine;
        this.groupId = id;
        this.retryPolicy = this.applicationStateMachine instanceof IStateMachine.RetryPolicy ? (IStateMachine.RetryPolicy)((Object)this.applicationStateMachine) : new IStateMachine.RetryPolicy(){};
        this.snapshotStorage = new SnapshotStorage(this.applicationStateMachine, this.groupId);
        this.applicationStateMachine.start();
    }

    public void initialize(RaftServer raftServer, RaftGroupId raftGroupId, RaftStorage storage) throws IOException {
        this.getLifeCycle().startAndTransition(() -> {
            this.snapshotStorage.init(storage);
            this.statemachineDir = this.snapshotStorage.getStateMachineDir();
            this.loadSnapshot(this.snapshotStorage.findLatestSnapshotDir());
        }, new Class[0]);
    }

    public void reinitialize() {
        this.setLastAppliedTermIndex(null);
        this.loadSnapshot(this.snapshotStorage.findLatestSnapshotDir());
        if (this.getLifeCycleState() == LifeCycle.State.PAUSED) {
            this.getLifeCycle().transition(LifeCycle.State.STARTING);
            this.getLifeCycle().transition(LifeCycle.State.RUNNING);
        }
    }

    public void pause() {
        if (this.getLifeCycleState() == LifeCycle.State.RUNNING) {
            this.getLifeCycle().transition(LifeCycle.State.PAUSING);
            this.getLifeCycle().transition(LifeCycle.State.PAUSED);
        }
    }

    public void close() throws IOException {
        this.getLifeCycle().checkStateAndClose(this.applicationStateMachine::stop);
    }

    public CompletableFuture<Message> applyTransaction(TransactionContext trx) {
        RaftProtos.LogEntryProto log = trx.getLogEntry();
        this.updateLastAppliedTermIndex(log.getTerm(), log.getIndex());
        IConsensusRequest applicationRequest = null;
        if (trx.getClientRequest() != null && trx.getClientRequest().getMessage() instanceof RequestMessage) {
            RequestMessage requestMessage = (RequestMessage)trx.getClientRequest().getMessage();
            applicationRequest = requestMessage.getActualRequest();
        } else {
            applicationRequest = new ByteBufferConsensusRequest(log.getStateMachineLogEntry().getLogData().asReadOnlyByteBuffer());
        }
        Message ret = null;
        this.waitUntilSystemNotReadOnly();
        TSStatus finalStatus = null;
        boolean shouldRetry = false;
        boolean firstTry = true;
        do {
            try {
                if (!firstTry) {
                    Thread.sleep(this.retryPolicy.getSleepTime());
                }
                TSStatus result = this.applicationStateMachine.write(applicationRequest);
                if (firstTry) {
                    finalStatus = result;
                    firstTry = false;
                } else {
                    finalStatus = this.retryPolicy.updateResult(finalStatus, result);
                }
                shouldRetry = this.retryPolicy.shouldRetry(finalStatus);
                if (shouldRetry) continue;
                ret = new ResponseMessage(finalStatus);
                break;
            }
            catch (InterruptedException i) {
                this.logger.warn("{} interrupted when retry sleep", (Object)this);
                Thread.currentThread().interrupt();
            }
            catch (Throwable rte) {
                this.logger.error("application statemachine throws a runtime exception: ", rte);
                ret = Message.valueOf((String)("internal error. statemachine throws a runtime exception: " + rte));
                if (!this.applicationStateMachine.isReadOnly()) break;
                this.waitUntilSystemNotReadOnly();
                shouldRetry = true;
            }
        } while (shouldRetry);
        return CompletableFuture.completedFuture(ret);
    }

    private void waitUntilSystemNotReadOnly() {
        while (this.applicationStateMachine.isReadOnly()) {
            try {
                TimeUnit.SECONDS.sleep(60L);
            }
            catch (InterruptedException e) {
                throw new RuntimeException(e);
            }
        }
    }

    public CompletableFuture<Message> query(Message request) {
        if (!(request instanceof RequestMessage)) {
            this.logger.error("An RequestMessage is required but got {}", (Object)request);
            return CompletableFuture.completedFuture(new ResponseMessage(null));
        }
        RequestMessage requestMessage = (RequestMessage)request;
        DataSet result = this.applicationStateMachine.read(requestMessage.getActualRequest());
        return CompletableFuture.completedFuture(new ResponseMessage(result));
    }

    public long takeSnapshot() throws IOException {
        TermIndex lastApplied = this.getLastAppliedTermIndex();
        if (lastApplied.getTerm() <= 0L || lastApplied.getIndex() <= 0L) {
            return -1L;
        }
        String metadata = Utils.getMetadataFromTermIndex(lastApplied);
        File snapshotTmpDir = this.snapshotStorage.getSnapshotTmpDir(metadata);
        FileUtils.deleteFully((File)snapshotTmpDir);
        snapshotTmpDir.mkdirs();
        if (!snapshotTmpDir.isDirectory()) {
            this.logger.error("Unable to create temp snapshotDir at {}", (Object)snapshotTmpDir);
            return -1L;
        }
        boolean applicationTakeSnapshotSuccess = this.applicationStateMachine.takeSnapshot(snapshotTmpDir, this.snapshotStorage.getSnapshotTmpId(metadata), metadata);
        if (!applicationTakeSnapshotSuccess) {
            this.deleteIncompleteSnapshot(snapshotTmpDir);
            return -1L;
        }
        File snapshotDir = this.snapshotStorage.getSnapshotDir(metadata);
        FileUtils.deleteFully((File)snapshotDir);
        try {
            Files.move(snapshotTmpDir.toPath(), snapshotDir.toPath(), StandardCopyOption.ATOMIC_MOVE);
        }
        catch (IOException e) {
            this.logger.error("{} atomic rename {} to {} failed with exception {}", new Object[]{this, snapshotTmpDir, snapshotDir, e});
            this.deleteIncompleteSnapshot(snapshotTmpDir);
            return -1L;
        }
        this.snapshotStorage.updateSnapshotCache();
        return lastApplied.getIndex();
    }

    private void deleteIncompleteSnapshot(File snapshotDir) throws IOException {
        boolean isEmpty = snapshotDir.delete();
        if (!isEmpty) {
            this.logger.info("Snapshot directory is incomplete, deleting " + snapshotDir.getAbsolutePath());
            FileUtils.deleteFully((File)snapshotDir);
        }
    }

    private void loadSnapshot(File latestSnapshotDir) {
        this.snapshotStorage.updateSnapshotCache();
        if (latestSnapshotDir == null) {
            return;
        }
        this.applicationStateMachine.loadSnapshot(latestSnapshotDir);
        TermIndex snapshotTermIndex = Utils.getTermIndexFromDir(latestSnapshotDir);
        this.updateLastAppliedTermIndex(snapshotTermIndex.getTerm(), snapshotTermIndex.getIndex());
    }

    public StateMachineStorage getStateMachineStorage() {
        return this.snapshotStorage;
    }

    public void notifyLeaderChanged(RaftGroupMemberId groupMemberId, RaftPeerId newLeaderId) {
        this.applicationStateMachine.event().notifyLeaderChanged(Utils.fromRaftGroupIdToConsensusGroupId(groupMemberId.getGroupId()), Utils.fromRaftPeerIdToNodeId(newLeaderId));
    }

    public void notifyConfigurationChanged(long term, long index, RaftProtos.RaftConfigurationProto newRaftConfiguration) {
        this.applicationStateMachine.event().notifyConfigurationChanged(term, index, Utils.fromRaftProtoListAndRaftGroupIdToPeers(newRaftConfiguration.getPeersList(), this.groupId));
    }
}

