/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iotdb.consensus.iot;

import java.io.DataOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.OutputStream;
import java.nio.ByteBuffer;
import java.nio.file.CopyOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.apache.commons.io.FileUtils;
import org.apache.iotdb.common.rpc.thrift.TEndPoint;
import org.apache.iotdb.common.rpc.thrift.TSStatus;
import org.apache.iotdb.commons.client.IClientManager;
import org.apache.iotdb.commons.service.metric.MetricService;
import org.apache.iotdb.commons.service.metric.enums.Metric;
import org.apache.iotdb.commons.service.metric.enums.Tag;
import org.apache.iotdb.consensus.IStateMachine;
import org.apache.iotdb.consensus.common.DataSet;
import org.apache.iotdb.consensus.common.Peer;
import org.apache.iotdb.consensus.common.request.IConsensusRequest;
import org.apache.iotdb.consensus.common.request.IndexedConsensusRequest;
import org.apache.iotdb.consensus.config.IoTConsensusConfig;
import org.apache.iotdb.consensus.exception.ConsensusGroupModifyPeerException;
import org.apache.iotdb.consensus.iot.IoTConsensusServerMetrics;
import org.apache.iotdb.consensus.iot.client.AsyncIoTConsensusServiceClient;
import org.apache.iotdb.consensus.iot.client.SyncIoTConsensusServiceClient;
import org.apache.iotdb.consensus.iot.logdispatcher.LogDispatcher;
import org.apache.iotdb.consensus.iot.snapshot.SnapshotFragmentReader;
import org.apache.iotdb.consensus.iot.thrift.TActivatePeerReq;
import org.apache.iotdb.consensus.iot.thrift.TActivatePeerRes;
import org.apache.iotdb.consensus.iot.thrift.TBuildSyncLogChannelReq;
import org.apache.iotdb.consensus.iot.thrift.TBuildSyncLogChannelRes;
import org.apache.iotdb.consensus.iot.thrift.TCleanupTransferredSnapshotReq;
import org.apache.iotdb.consensus.iot.thrift.TCleanupTransferredSnapshotRes;
import org.apache.iotdb.consensus.iot.thrift.TInactivatePeerReq;
import org.apache.iotdb.consensus.iot.thrift.TInactivatePeerRes;
import org.apache.iotdb.consensus.iot.thrift.TRemoveSyncLogChannelReq;
import org.apache.iotdb.consensus.iot.thrift.TRemoveSyncLogChannelRes;
import org.apache.iotdb.consensus.iot.thrift.TSendSnapshotFragmentReq;
import org.apache.iotdb.consensus.iot.thrift.TSendSnapshotFragmentRes;
import org.apache.iotdb.consensus.iot.thrift.TTriggerSnapshotLoadReq;
import org.apache.iotdb.consensus.iot.thrift.TTriggerSnapshotLoadRes;
import org.apache.iotdb.consensus.iot.thrift.TWaitSyncLogCompleteReq;
import org.apache.iotdb.consensus.iot.thrift.TWaitSyncLogCompleteRes;
import org.apache.iotdb.consensus.iot.wal.ConsensusReqReader;
import org.apache.iotdb.consensus.iot.wal.GetConsensusReqReaderPlan;
import org.apache.iotdb.metrics.metricsets.IMetricSet;
import org.apache.iotdb.metrics.utils.MetricLevel;
import org.apache.iotdb.rpc.RpcUtils;
import org.apache.iotdb.rpc.TSStatusCode;
import org.apache.iotdb.tsfile.utils.PublicBAOS;
import org.apache.thrift.TException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class IoTConsensusServerImpl {
    private static final String CONFIGURATION_FILE_NAME = "configuration.dat";
    private static final String CONFIGURATION_TMP_FILE_NAME = "configuration.dat.tmp";
    private static final String SNAPSHOT_DIR_NAME = "snapshot";
    private final Logger logger = LoggerFactory.getLogger(IoTConsensusServerImpl.class);
    private final Peer thisNode;
    private final IStateMachine stateMachine;
    private final Lock stateMachineLock = new ReentrantLock();
    private final Condition stateMachineCondition = this.stateMachineLock.newCondition();
    private final String storageDir;
    private final List<Peer> configuration;
    private final AtomicLong index;
    private final LogDispatcher logDispatcher;
    private final IoTConsensusConfig config;
    private final ConsensusReqReader reader;
    private volatile boolean active = true;
    private String latestSnapshotId;
    private final IClientManager<TEndPoint, SyncIoTConsensusServiceClient> syncClientManager;
    private final IoTConsensusServerMetrics metrics;
    private final String consensusGroupId;

    public IoTConsensusServerImpl(String storageDir, Peer thisNode, List<Peer> configuration, IStateMachine stateMachine, IClientManager<TEndPoint, AsyncIoTConsensusServiceClient> clientManager, IClientManager<TEndPoint, SyncIoTConsensusServiceClient> syncClientManager, IoTConsensusConfig config) {
        this.storageDir = storageDir;
        this.thisNode = thisNode;
        this.stateMachine = stateMachine;
        this.syncClientManager = syncClientManager;
        this.configuration = configuration;
        if (configuration.isEmpty()) {
            this.recoverConfiguration();
        } else {
            this.persistConfiguration();
        }
        this.config = config;
        this.logDispatcher = new LogDispatcher(this, clientManager);
        this.reader = (ConsensusReqReader)((Object)stateMachine.read(new GetConsensusReqReaderPlan()));
        long currentSearchIndex = this.reader.getCurrentSearchIndex();
        if (1 == configuration.size()) {
            this.reader.setSafelyDeletedSearchIndex(Long.MAX_VALUE);
        }
        this.index = new AtomicLong(currentSearchIndex);
        this.consensusGroupId = thisNode.getGroupId().toString();
        this.metrics = new IoTConsensusServerMetrics(this);
    }

    public IStateMachine getStateMachine() {
        return this.stateMachine;
    }

    public void start() {
        MetricService.getInstance().addMetricSet((IMetricSet)this.metrics);
        this.stateMachine.start();
        this.logDispatcher.start();
    }

    public void stop() {
        this.logDispatcher.stop();
        this.stateMachine.stop();
        MetricService.getInstance().removeMetricSet((IMetricSet)this.metrics);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public TSStatus write(IConsensusRequest request) {
        long consensusWriteStartTime = System.currentTimeMillis();
        this.stateMachineLock.lock();
        try {
            AtomicLong atomicLong;
            long getStateMachineLockTime = System.currentTimeMillis();
            MetricService.getInstance().getOrCreateHistogram(Metric.STAGE.toString(), MetricLevel.IMPORTANT, new String[]{Tag.NAME.toString(), Metric.IOT_CONSENSUS.toString(), Tag.TYPE.toString(), "getStateMachineLock", Tag.REGION.toString(), this.consensusGroupId}).update(getStateMachineLockTime - consensusWriteStartTime);
            if (this.needBlockWrite()) {
                this.logger.info("[Throttle Down] index:{}, safeIndex:{}", (Object)this.getIndex(), (Object)this.getCurrentSafelyDeletedSearchIndex());
                try {
                    boolean timeout;
                    boolean bl = timeout = !this.stateMachineCondition.await(this.config.getReplication().getThrottleTimeOutMs(), TimeUnit.MILLISECONDS);
                    if (timeout) {
                        TSStatus tSStatus = RpcUtils.getStatus((TSStatusCode)TSStatusCode.WRITE_PROCESS_REJECT, (String)"Reject write because there are too many requests need to process");
                        return tSStatus;
                    }
                }
                catch (InterruptedException e) {
                    this.logger.error("Failed to throttle down because ", (Throwable)e);
                    Thread.currentThread().interrupt();
                }
            }
            long writeToStateMachineStartTime = System.currentTimeMillis();
            IndexedConsensusRequest indexedConsensusRequest = this.buildIndexedConsensusRequestForLocalRequest(request);
            MetricService.getInstance().getOrCreateHistogram(Metric.STAGE.toString(), MetricLevel.IMPORTANT, new String[]{Tag.NAME.toString(), Metric.IOT_CONSENSUS.toString(), Tag.TYPE.toString(), "checkingBeforeWrite", Tag.REGION.toString(), this.consensusGroupId}).update(writeToStateMachineStartTime - getStateMachineLockTime);
            if (indexedConsensusRequest.getSearchIndex() % 1000L == 0L) {
                this.logger.info("DataRegion[{}]: index after build: safeIndex:{}, searchIndex: {}", new Object[]{this.thisNode.getGroupId(), this.getCurrentSafelyDeletedSearchIndex(), indexedConsensusRequest.getSearchIndex()});
            }
            TSStatus result = this.stateMachine.write(indexedConsensusRequest);
            long writeToStateMachineEndTime = System.currentTimeMillis();
            MetricService.getInstance().getOrCreateHistogram(Metric.STAGE.toString(), MetricLevel.IMPORTANT, new String[]{Tag.NAME.toString(), Metric.IOT_CONSENSUS.toString(), Tag.TYPE.toString(), "writeStateMachine", Tag.REGION.toString(), this.consensusGroupId}).update(writeToStateMachineEndTime - writeToStateMachineStartTime);
            if (result.getCode() == TSStatusCode.SUCCESS_STATUS.getStatusCode()) {
                atomicLong = this.index;
                synchronized (atomicLong) {
                    this.logDispatcher.offer(indexedConsensusRequest);
                    this.index.incrementAndGet();
                }
                MetricService.getInstance().getOrCreateHistogram(Metric.STAGE.toString(), MetricLevel.IMPORTANT, new String[]{Tag.NAME.toString(), Metric.IOT_CONSENSUS.toString(), Tag.TYPE.toString(), "offerRequestToQueue", Tag.REGION.toString(), this.consensusGroupId}).update(System.currentTimeMillis() - writeToStateMachineEndTime);
            } else {
                this.logger.debug("{}: write operation failed. searchIndex: {}. Code: {}", new Object[]{this.thisNode.getGroupId(), indexedConsensusRequest.getSearchIndex(), result.getCode()});
            }
            MetricService.getInstance().getOrCreateHistogram(Metric.STAGE.toString(), MetricLevel.IMPORTANT, new String[]{Tag.NAME.toString(), Metric.IOT_CONSENSUS.toString(), Tag.TYPE.toString(), "consensusWrite", Tag.REGION.toString(), this.consensusGroupId}).update(System.currentTimeMillis() - consensusWriteStartTime);
            atomicLong = result;
            return atomicLong;
        }
        finally {
            this.stateMachineLock.unlock();
        }
    }

    public DataSet read(IConsensusRequest request) {
        return this.stateMachine.read(request);
    }

    public void takeSnapshot() throws ConsensusGroupModifyPeerException {
        try {
            this.latestSnapshotId = String.format("%s_%s_%d", SNAPSHOT_DIR_NAME, this.thisNode.getGroupId().getId(), System.currentTimeMillis());
            File snapshotDir = new File(this.storageDir, this.latestSnapshotId);
            if (snapshotDir.exists()) {
                FileUtils.deleteDirectory((File)snapshotDir);
            }
            if (!snapshotDir.mkdirs()) {
                throw new ConsensusGroupModifyPeerException(String.format("%s: cannot mkdir for snapshot", this.thisNode.getGroupId()));
            }
            if (!this.stateMachine.takeSnapshot(snapshotDir)) {
                throw new ConsensusGroupModifyPeerException("unknown error when taking snapshot");
            }
        }
        catch (IOException e) {
            throw new ConsensusGroupModifyPeerException("error when taking snapshot", e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void transitSnapshot(Peer targetPeer) throws ConsensusGroupModifyPeerException {
        File snapshotDir = new File(this.storageDir, this.latestSnapshotId);
        List<Path> snapshotPaths = this.stateMachine.getSnapshotFiles(snapshotDir);
        this.logger.info("transit snapshots: {}", snapshotPaths);
        try (SyncIoTConsensusServiceClient client = (SyncIoTConsensusServiceClient)this.syncClientManager.borrowClient((Object)targetPeer.getEndpoint());){
            for (Path path : snapshotPaths) {
                try (SnapshotFragmentReader reader = new SnapshotFragmentReader(this.latestSnapshotId, path);){
                    while (reader.hasNext()) {
                        TSendSnapshotFragmentReq req = reader.next().toTSendSnapshotFragmentReq();
                        req.setConsensusGroupId(targetPeer.getGroupId().convertToTConsensusGroupId());
                        TSendSnapshotFragmentRes res = client.sendSnapshotFragment(req);
                        if (this.isSuccess(res.getStatus())) continue;
                        throw new ConsensusGroupModifyPeerException(String.format("error when sending snapshot fragment to %s", targetPeer));
                    }
                }
            }
        }
        catch (IOException | TException e) {
            throw new ConsensusGroupModifyPeerException(String.format("error when send snapshot file to %s", targetPeer), e);
        }
    }

    public void receiveSnapshotFragment(String snapshotId, String originalFilePath, ByteBuffer fileChunk) throws ConsensusGroupModifyPeerException {
        try {
            String targetFilePath = this.calculateSnapshotPath(snapshotId, originalFilePath);
            File targetFile = new File(this.storageDir, targetFilePath);
            Path parentDir = Paths.get(targetFile.getParent(), new String[0]);
            if (!Files.exists(parentDir, new LinkOption[0])) {
                Files.createDirectories(parentDir, new FileAttribute[0]);
            }
            Files.write(Paths.get(targetFile.getAbsolutePath(), new String[0]), fileChunk.array(), StandardOpenOption.CREATE, StandardOpenOption.APPEND);
        }
        catch (IOException e) {
            throw new ConsensusGroupModifyPeerException(String.format("error when receiving snapshot %s", snapshotId), e);
        }
    }

    private String calculateSnapshotPath(String snapshotId, String originalFilePath) throws ConsensusGroupModifyPeerException {
        if (!originalFilePath.contains(snapshotId)) {
            throw new ConsensusGroupModifyPeerException(String.format("invalid snapshot file. snapshotId: %s, filePath: %s", snapshotId, originalFilePath));
        }
        return originalFilePath.substring(originalFilePath.indexOf(snapshotId));
    }

    public void loadSnapshot(String snapshotId) {
        this.stateMachine.loadSnapshot(new File(this.storageDir, snapshotId));
    }

    public void inactivePeer(Peer peer) throws ConsensusGroupModifyPeerException {
        try (SyncIoTConsensusServiceClient client = (SyncIoTConsensusServiceClient)this.syncClientManager.borrowClient((Object)peer.getEndpoint());){
            TInactivatePeerRes res = client.inactivatePeer(new TInactivatePeerReq(peer.getGroupId().convertToTConsensusGroupId()));
            if (!this.isSuccess(res.status)) {
                throw new ConsensusGroupModifyPeerException(String.format("error when inactivating %s. %s", peer, res.getStatus()));
            }
        }
        catch (IOException | TException e) {
            throw new ConsensusGroupModifyPeerException(String.format("error when inactivating %s", peer), e);
        }
    }

    public void triggerSnapshotLoad(Peer peer) throws ConsensusGroupModifyPeerException {
        try (SyncIoTConsensusServiceClient client = (SyncIoTConsensusServiceClient)this.syncClientManager.borrowClient((Object)peer.getEndpoint());){
            TTriggerSnapshotLoadRes res = client.triggerSnapshotLoad(new TTriggerSnapshotLoadReq(this.thisNode.getGroupId().convertToTConsensusGroupId(), this.latestSnapshotId));
            if (!this.isSuccess(res.status)) {
                throw new ConsensusGroupModifyPeerException(String.format("error when triggering snapshot load %s. %s", peer, res.getStatus()));
            }
        }
        catch (IOException | TException e) {
            throw new ConsensusGroupModifyPeerException(String.format("error when activating %s", peer), e);
        }
    }

    public void activePeer(Peer peer) throws ConsensusGroupModifyPeerException {
        try (SyncIoTConsensusServiceClient client = (SyncIoTConsensusServiceClient)this.syncClientManager.borrowClient((Object)peer.getEndpoint());){
            TActivatePeerRes res = client.activatePeer(new TActivatePeerReq(peer.getGroupId().convertToTConsensusGroupId()));
            if (!this.isSuccess(res.status)) {
                throw new ConsensusGroupModifyPeerException(String.format("error when activating %s. %s", peer, res.getStatus()));
            }
        }
        catch (IOException | TException e) {
            throw new ConsensusGroupModifyPeerException(String.format("error when activating %s", peer), e);
        }
    }

    public void notifyPeersToBuildSyncLogChannel(Peer targetPeer) throws ConsensusGroupModifyPeerException {
        ArrayList<Peer> currentMembers = new ArrayList<Peer>(this.configuration);
        this.logger.info("[IoTConsensus] notify current peers to build sync log. group member: {}, target: {}", currentMembers, (Object)targetPeer);
        for (Peer peer : currentMembers) {
            this.logger.info("[IoTConsensus] build sync log channel from {}", (Object)peer);
            if (peer.equals(this.thisNode)) {
                this.buildSyncLogChannel(targetPeer, this.index.get());
                continue;
            }
            try {
                SyncIoTConsensusServiceClient client = (SyncIoTConsensusServiceClient)this.syncClientManager.borrowClient((Object)peer.getEndpoint());
                try {
                    TBuildSyncLogChannelRes res = client.buildSyncLogChannel(new TBuildSyncLogChannelReq(targetPeer.getGroupId().convertToTConsensusGroupId(), targetPeer.getEndpoint(), targetPeer.getNodeId()));
                    if (this.isSuccess(res.status)) continue;
                    throw new ConsensusGroupModifyPeerException(String.format("build sync log channel failed from %s to %s", peer, targetPeer));
                }
                finally {
                    if (client == null) continue;
                    client.close();
                }
            }
            catch (IOException | TException e) {
                this.logger.error("cannot notify {} to build sync log channel. Please check the status of this node manually", (Object)peer, (Object)e);
            }
        }
    }

    public void notifyPeersToRemoveSyncLogChannel(Peer targetPeer) throws ConsensusGroupModifyPeerException {
        ArrayList<Peer> currentMembers = new ArrayList<Peer>(this.configuration);
        for (Peer peer : currentMembers) {
            if (peer.equals(targetPeer)) continue;
            if (peer.equals(this.thisNode)) {
                this.removeSyncLogChannel(targetPeer);
                continue;
            }
            try {
                SyncIoTConsensusServiceClient client = (SyncIoTConsensusServiceClient)this.syncClientManager.borrowClient((Object)peer.getEndpoint());
                try {
                    TRemoveSyncLogChannelRes res = client.removeSyncLogChannel(new TRemoveSyncLogChannelReq(targetPeer.getGroupId().convertToTConsensusGroupId(), targetPeer.getEndpoint(), targetPeer.getNodeId()));
                    if (this.isSuccess(res.status)) continue;
                    throw new ConsensusGroupModifyPeerException(String.format("remove sync log channel failed from %s to %s", peer, targetPeer));
                }
                finally {
                    if (client == null) continue;
                    client.close();
                }
            }
            catch (IOException | TException e) {
                throw new ConsensusGroupModifyPeerException(String.format("error when removing sync log channel to %s", peer), e);
            }
        }
    }

    public void waitTargetPeerUntilSyncLogCompleted(Peer targetPeer) throws ConsensusGroupModifyPeerException {
        long checkIntervalInMs = 10000L;
        try (SyncIoTConsensusServiceClient client = (SyncIoTConsensusServiceClient)this.syncClientManager.borrowClient((Object)targetPeer.getEndpoint());){
            while (true) {
                TWaitSyncLogCompleteRes res = client.waitSyncLogComplete(new TWaitSyncLogCompleteReq(targetPeer.getGroupId().convertToTConsensusGroupId()));
                if (res.complete) {
                    this.logger.info("{} SyncLog is completed. TargetIndex: {}, CurrentSyncIndex: {}", new Object[]{targetPeer, res.searchIndex, res.safeIndex});
                    return;
                }
                this.logger.info("{} SyncLog is still in progress. TargetIndex: {}, CurrentSyncIndex: {}", new Object[]{targetPeer, res.searchIndex, res.safeIndex});
                Thread.sleep(checkIntervalInMs);
            }
        }
        catch (IOException | TException e) {
            throw new ConsensusGroupModifyPeerException(String.format("error when waiting %s to complete SyncLog. %s", targetPeer, e.getMessage()), e);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new ConsensusGroupModifyPeerException(String.format("thread interrupted when waiting %s to complete SyncLog. %s", targetPeer, e.getMessage()), e);
        }
    }

    private boolean isSuccess(TSStatus status) {
        return status.getCode() == TSStatusCode.SUCCESS_STATUS.getStatusCode();
    }

    public void buildSyncLogChannel(Peer targetPeer) throws ConsensusGroupModifyPeerException {
        this.buildSyncLogChannel(targetPeer, this.getCurrentSafelyDeletedSearchIndex());
    }

    public void buildSyncLogChannel(Peer targetPeer, long initialSyncIndex) throws ConsensusGroupModifyPeerException {
        this.logger.info("[IoTConsensus] build sync log channel to {} with initialSyncIndex {}", (Object)targetPeer, (Object)initialSyncIndex);
        this.logDispatcher.addLogDispatcherThread(targetPeer, initialSyncIndex);
        this.configuration.add(targetPeer);
        this.logger.info("[IoTConsensus] persist new configuration: {}", this.configuration);
        this.persistConfigurationUpdate();
    }

    public void removeSyncLogChannel(Peer targetPeer) throws ConsensusGroupModifyPeerException {
        try {
            this.logDispatcher.removeLogDispatcherThread(targetPeer);
            this.logger.info("[IoTConsensus] log dispatcher to {} removed and cleanup", (Object)targetPeer);
            this.configuration.remove(targetPeer);
            this.persistConfigurationUpdate();
            this.logger.info("[IoTConsensus] configuration updated to {}", this.configuration);
        }
        catch (IOException e) {
            throw new ConsensusGroupModifyPeerException("error when remove LogDispatcherThread", e);
        }
    }

    public void persistConfiguration() {
        try (PublicBAOS publicBAOS = new PublicBAOS();
             DataOutputStream outputStream = new DataOutputStream((OutputStream)publicBAOS);){
            this.serializeConfigurationTo(outputStream);
            Files.write(Paths.get(new File(this.storageDir, CONFIGURATION_FILE_NAME).getAbsolutePath(), new String[0]), publicBAOS.getBuf(), new OpenOption[0]);
        }
        catch (IOException e) {
            this.logger.error("Unexpected error occurs when persisting configuration", (Throwable)e);
        }
    }

    public void persistConfigurationUpdate() throws ConsensusGroupModifyPeerException {
        try (PublicBAOS publicBAOS = new PublicBAOS();
             DataOutputStream outputStream = new DataOutputStream((OutputStream)publicBAOS);){
            this.serializeConfigurationTo(outputStream);
            Path tmpConfigurationPath = Paths.get(new File(this.storageDir, CONFIGURATION_TMP_FILE_NAME).getAbsolutePath(), new String[0]);
            Path configurationPath = Paths.get(new File(this.storageDir, CONFIGURATION_FILE_NAME).getAbsolutePath(), new String[0]);
            Files.write(tmpConfigurationPath, publicBAOS.getBuf(), new OpenOption[0]);
            Files.delete(configurationPath);
            Files.move(tmpConfigurationPath, configurationPath, new CopyOption[0]);
        }
        catch (IOException e) {
            throw new ConsensusGroupModifyPeerException("Unexpected error occurs when update configuration", e);
        }
    }

    private void serializeConfigurationTo(DataOutputStream outputStream) throws IOException {
        outputStream.writeInt(this.configuration.size());
        for (Peer peer : this.configuration) {
            peer.serialize(outputStream);
        }
    }

    public void recoverConfiguration() {
        try {
            Path tmpConfigurationPath = Paths.get(new File(this.storageDir, CONFIGURATION_TMP_FILE_NAME).getAbsolutePath(), new String[0]);
            Path configurationPath = Paths.get(new File(this.storageDir, CONFIGURATION_FILE_NAME).getAbsolutePath(), new String[0]);
            if (Files.exists(tmpConfigurationPath, new LinkOption[0])) {
                if (Files.exists(configurationPath, new LinkOption[0])) {
                    Files.delete(configurationPath);
                }
                Files.move(tmpConfigurationPath, configurationPath, new CopyOption[0]);
            }
            ByteBuffer buffer = ByteBuffer.wrap(Files.readAllBytes(configurationPath));
            int size = buffer.getInt();
            for (int i = 0; i < size; ++i) {
                this.configuration.add(Peer.deserialize(buffer));
            }
            this.logger.info("Recover IoTConsensus server Impl, configuration: {}", this.configuration);
        }
        catch (IOException e) {
            this.logger.error("Unexpected error occurs when recovering configuration", (Throwable)e);
        }
    }

    public IndexedConsensusRequest buildIndexedConsensusRequestForLocalRequest(IConsensusRequest request) {
        return new IndexedConsensusRequest(this.index.get() + 1L, Collections.singletonList(request));
    }

    public IndexedConsensusRequest buildIndexedConsensusRequestForRemoteRequest(long syncIndex, List<IConsensusRequest> requests) {
        return new IndexedConsensusRequest(-1L, syncIndex, requests);
    }

    public long getCurrentSafelyDeletedSearchIndex() {
        return this.logDispatcher.getMinSyncIndex().orElseGet(this.index::get);
    }

    public String getStorageDir() {
        return this.storageDir;
    }

    public Peer getThisNode() {
        return this.thisNode;
    }

    public List<Peer> getConfiguration() {
        return this.configuration;
    }

    public long getIndex() {
        return this.index.get();
    }

    public IoTConsensusConfig getConfig() {
        return this.config;
    }

    public boolean needBlockWrite() {
        return this.reader.getTotalSize() > this.config.getReplication().getWalThrottleThreshold();
    }

    public boolean unblockWrite() {
        return this.reader.getTotalSize() < this.config.getReplication().getWalThrottleThreshold();
    }

    public void signal() {
        this.stateMachineLock.lock();
        try {
            this.stateMachineCondition.signalAll();
        }
        finally {
            this.stateMachineLock.unlock();
        }
    }

    public AtomicLong getIndexObject() {
        return this.index;
    }

    public boolean isReadOnly() {
        return this.stateMachine.isReadOnly();
    }

    public boolean isActive() {
        return this.active;
    }

    public void setActive(boolean active) {
        this.logger.info("set {} active status to {}", (Object)this.thisNode, (Object)active);
        this.active = active;
    }

    public void cleanupRemoteSnapshot(Peer targetPeer) throws ConsensusGroupModifyPeerException {
        try (SyncIoTConsensusServiceClient client = (SyncIoTConsensusServiceClient)this.syncClientManager.borrowClient((Object)targetPeer.getEndpoint());){
            TCleanupTransferredSnapshotReq req = new TCleanupTransferredSnapshotReq(targetPeer.getGroupId().convertToTConsensusGroupId(), this.latestSnapshotId);
            TCleanupTransferredSnapshotRes res = client.cleanupTransferredSnapshot(req);
            if (!this.isSuccess(res.getStatus())) {
                throw new ConsensusGroupModifyPeerException(String.format("cleanup remote snapshot failed of %s ,status is %s", targetPeer, res.getStatus()));
            }
        }
        catch (IOException | TException e) {
            throw new ConsensusGroupModifyPeerException(String.format("cleanup remote snapshot failed of %s", targetPeer), e);
        }
    }

    public void cleanupTransferredSnapshot(String snapshotId) throws ConsensusGroupModifyPeerException {
        File snapshotDir = new File(this.storageDir, snapshotId);
        if (snapshotDir.exists()) {
            try {
                FileUtils.deleteDirectory((File)snapshotDir);
            }
            catch (IOException e) {
                throw new ConsensusGroupModifyPeerException(e);
            }
        }
    }
}

