/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iotdb.db.engine;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.ConcurrentModificationException;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import org.apache.commons.io.FileUtils;
import org.apache.iotdb.db.concurrent.IoTDBThreadPoolFactory;
import org.apache.iotdb.db.conf.IoTDBConfig;
import org.apache.iotdb.db.conf.IoTDBDescriptor;
import org.apache.iotdb.db.engine.fileSystem.SystemFileFactory;
import org.apache.iotdb.db.engine.querycontext.QueryDataSource;
import org.apache.iotdb.db.engine.storagegroup.StorageGroupProcessor;
import org.apache.iotdb.db.engine.storagegroup.TsFileResource;
import org.apache.iotdb.db.exception.StorageEngineException;
import org.apache.iotdb.db.exception.TsFileProcessorException;
import org.apache.iotdb.db.exception.path.PathException;
import org.apache.iotdb.db.exception.query.QueryProcessException;
import org.apache.iotdb.db.exception.runtime.StorageEngineFailureException;
import org.apache.iotdb.db.exception.storageGroup.StorageGroupException;
import org.apache.iotdb.db.exception.storageGroup.StorageGroupProcessorException;
import org.apache.iotdb.db.metadata.MManager;
import org.apache.iotdb.db.metadata.MNode;
import org.apache.iotdb.db.qp.physical.crud.BatchInsertPlan;
import org.apache.iotdb.db.qp.physical.crud.InsertPlan;
import org.apache.iotdb.db.query.context.QueryContext;
import org.apache.iotdb.db.query.control.JobFileManager;
import org.apache.iotdb.db.service.IService;
import org.apache.iotdb.db.service.ServiceType;
import org.apache.iotdb.db.utils.FilePathUtils;
import org.apache.iotdb.db.utils.UpgradeUtils;
import org.apache.iotdb.tsfile.file.metadata.enums.CompressionType;
import org.apache.iotdb.tsfile.file.metadata.enums.TSDataType;
import org.apache.iotdb.tsfile.file.metadata.enums.TSEncoding;
import org.apache.iotdb.tsfile.read.common.Path;
import org.apache.iotdb.tsfile.read.expression.impl.SingleSeriesExpression;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class StorageEngine
implements IService {
    private final Logger logger;
    private static final IoTDBConfig config = IoTDBDescriptor.getInstance().getConfig();
    private static final long TTL_CHECK_INTERVAL = 60000L;
    private final String systemDir;
    private final ConcurrentHashMap<String, StorageGroupProcessor> processorMap = new ConcurrentHashMap();
    private static final ExecutorService recoveryThreadPool = IoTDBThreadPoolFactory.newFixedThreadPool(Runtime.getRuntime().availableProcessors(), "Recovery-Thread-Pool");
    private static final StorageEngine INSTANCE = new StorageEngine();
    private ScheduledExecutorService ttlCheckThread;

    public static StorageEngine getInstance() {
        return INSTANCE;
    }

    private StorageEngine() {
        this.logger = LoggerFactory.getLogger(StorageEngine.class);
        this.systemDir = FilePathUtils.regularizePath(config.getSystemDir()) + "storage_groups";
        try {
            FileUtils.forceMkdir((File)SystemFileFactory.INSTANCE.getFile(this.systemDir));
        }
        catch (IOException e) {
            throw new StorageEngineFailureException(e);
        }
        UpgradeUtils.recoverUpgrade();
        List<MNode> sgNodes = MManager.getInstance().getAllStorageGroups();
        ArrayList<Future<Void>> futures = new ArrayList<Future<Void>>();
        for (MNode mNode : sgNodes) {
            futures.add(recoveryThreadPool.submit(() -> {
                StorageGroupProcessor processor = new StorageGroupProcessor(this.systemDir, storageGroup.getFullPath());
                processor.setDataTTL(storageGroup.getDataTTL());
                this.processorMap.put(storageGroup.getFullPath(), processor);
                this.logger.info("Storage Group Processor {} is recovered successfully", (Object)storageGroup.getFullPath());
                return null;
            }));
        }
        for (Future future : futures) {
            try {
                future.get();
            }
            catch (InterruptedException | ExecutionException e) {
                throw new StorageEngineFailureException("StorageEngine failed to recover.", e);
            }
        }
    }

    @Override
    public void start() {
        this.ttlCheckThread = Executors.newSingleThreadScheduledExecutor();
        this.ttlCheckThread.scheduleAtFixedRate(this::checkTTL, 60000L, 60000L, TimeUnit.MILLISECONDS);
    }

    private void checkTTL() {
        try {
            for (StorageGroupProcessor processor : this.processorMap.values()) {
                processor.checkFilesTTL();
            }
        }
        catch (ConcurrentModificationException concurrentModificationException) {
        }
        catch (Exception e) {
            this.logger.error("An error occurred when checking TTL", (Throwable)e);
        }
    }

    @Override
    public void stop() {
        this.syncCloseAllProcessor();
        this.ttlCheckThread.shutdownNow();
        recoveryThreadPool.shutdownNow();
        try {
            this.ttlCheckThread.awaitTermination(30L, TimeUnit.SECONDS);
        }
        catch (InterruptedException e) {
            this.logger.warn("TTL check thread still doesn't exit after 30s");
        }
    }

    @Override
    public ServiceType getID() {
        return ServiceType.STORAGE_ENGINE_SERVICE;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public StorageGroupProcessor getProcessor(String path) throws StorageEngineException {
        String storageGroupName = "";
        try {
            storageGroupName = MManager.getInstance().getStorageGroupNameByPath(path);
            StorageGroupProcessor processor = this.processorMap.get(storageGroupName);
            if (processor == null) {
                String string = storageGroupName = storageGroupName.intern();
                synchronized (string) {
                    processor = this.processorMap.get(storageGroupName);
                    if (processor == null) {
                        this.logger.info("construct a processor instance, the storage group is {}, Thread is {}", (Object)storageGroupName, (Object)Thread.currentThread().getId());
                        processor = new StorageGroupProcessor(this.systemDir, storageGroupName);
                        processor.setDataTTL(MManager.getInstance().getNodeByPathWithCheck(storageGroupName).getDataTTL());
                        this.processorMap.put(storageGroupName, processor);
                    }
                }
            }
            return processor;
        }
        catch (PathException | StorageGroupException | StorageGroupProcessorException e) {
            this.logger.error("Fail to get StorageGroupProcessor {}", (Object)storageGroupName, (Object)e);
            throw new StorageEngineException(e);
        }
    }

    public synchronized void reset() {
        this.processorMap.clear();
    }

    public void insert(InsertPlan insertPlan) throws StorageEngineException, QueryProcessException {
        StorageGroupProcessor storageGroupProcessor;
        try {
            storageGroupProcessor = this.getProcessor(insertPlan.getDeviceId());
        }
        catch (StorageEngineException e) {
            this.logger.warn("get StorageGroupProcessor of device {} failed, because {}", new Object[]{insertPlan.getDeviceId(), e.getMessage(), e});
            throw new StorageEngineException(e);
        }
        try {
            storageGroupProcessor.insert(insertPlan);
        }
        catch (QueryProcessException e) {
            throw new QueryProcessException(e);
        }
    }

    public Integer[] insertBatch(BatchInsertPlan batchInsertPlan) throws StorageEngineException {
        StorageGroupProcessor storageGroupProcessor;
        try {
            storageGroupProcessor = this.getProcessor(batchInsertPlan.getDeviceId());
        }
        catch (StorageEngineException e) {
            this.logger.warn("get StorageGroupProcessor of device {} failed, because {}", new Object[]{batchInsertPlan.getDeviceId(), e.getMessage(), e});
            throw new StorageEngineException(e);
        }
        try {
            return storageGroupProcessor.insertBatch(batchInsertPlan);
        }
        catch (QueryProcessException e) {
            throw new StorageEngineException(e);
        }
    }

    public void syncCloseAllProcessor() {
        this.logger.info("Start closing all storage group processor");
        for (StorageGroupProcessor processor : this.processorMap.values()) {
            processor.waitForAllCurrentTsFileProcessorsClosed();
        }
    }

    public void update(String deviceId, String measurementId, long startTime, long endTime, TSDataType type, String v) {
    }

    public void delete(String deviceId, String measurementId, long timestamp) throws StorageEngineException {
        StorageGroupProcessor storageGroupProcessor = this.getProcessor(deviceId);
        try {
            storageGroupProcessor.delete(deviceId, measurementId, timestamp);
        }
        catch (IOException e) {
            throw new StorageEngineException(e.getMessage());
        }
    }

    public QueryDataSource query(SingleSeriesExpression seriesExpression, QueryContext context, JobFileManager filePathsManager) throws StorageEngineException {
        String deviceId = seriesExpression.getSeriesPath().getDevice();
        String measurementId = seriesExpression.getSeriesPath().getMeasurement();
        StorageGroupProcessor storageGroupProcessor = this.getProcessor(deviceId);
        return storageGroupProcessor.query(deviceId, measurementId, context, filePathsManager);
    }

    public Set calTopKMeasurement(String deviceId, String sensorId, double k) throws StorageEngineException {
        StorageGroupProcessor storageGroupProcessor = this.getProcessor(deviceId);
        return storageGroupProcessor.calTopKMeasurement(sensorId, k);
    }

    public boolean appendFileToStorageGroupProcessor(String storageGroupName, TsFileResource appendFile, String appendFilePath) throws StorageEngineException {
        return true;
    }

    public List<String> getOverlapFiles(String storageGroupName, TsFileResource appendFile, String uuid) throws StorageEngineException {
        return Collections.emptyList();
    }

    public int countUpgradeFiles() {
        int totalUpgradeFileNum = 0;
        for (StorageGroupProcessor storageGroupProcessor : this.processorMap.values()) {
            totalUpgradeFileNum += storageGroupProcessor.countUpgradeFiles();
        }
        return totalUpgradeFileNum;
    }

    public void upgradeAll() throws StorageEngineException {
        if (IoTDBDescriptor.getInstance().getConfig().isReadOnly()) {
            throw new StorageEngineException("Current system mode is read only, does not support file upgrade");
        }
        for (StorageGroupProcessor storageGroupProcessor : this.processorMap.values()) {
            storageGroupProcessor.upgrade();
        }
    }

    public void mergeAll(boolean fullMerge) throws StorageEngineException {
        if (IoTDBDescriptor.getInstance().getConfig().isReadOnly()) {
            throw new StorageEngineException("Current system mode is read only, does not support merge");
        }
        for (StorageGroupProcessor storageGroupProcessor : this.processorMap.values()) {
            storageGroupProcessor.merge(fullMerge);
        }
    }

    public void deleteAllDataFilesInOneStorageGroup(String storageGroupName) {
        if (this.processorMap.containsKey(storageGroupName)) {
            this.syncDeleteDataFiles(storageGroupName);
        }
    }

    private void syncDeleteDataFiles(String storageGroupName) {
        this.logger.info("Force to delete the data in storage group processor {}", (Object)storageGroupName);
        StorageGroupProcessor processor = this.processorMap.get(storageGroupName);
        processor.syncDeleteDataFiles();
    }

    public void addTimeSeries(Path path, TSDataType dataType, TSEncoding encoding, CompressionType compressor, Map<String, String> props) throws StorageEngineException {
        StorageGroupProcessor storageGroupProcessor = this.getProcessor(path.getDevice());
        storageGroupProcessor.addMeasurement(path.getMeasurement(), dataType, encoding, compressor, props);
    }

    public synchronized boolean deleteAll() {
        this.logger.info("Start deleting all storage groups' timeseries");
        for (String storageGroup : MManager.getInstance().getAllStorageGroupNames()) {
            this.deleteAllDataFilesInOneStorageGroup(storageGroup);
        }
        return true;
    }

    public void setTTL(String storageGroup, long dataTTL) throws StorageEngineException {
        StorageGroupProcessor storageGroupProcessor = this.getProcessor(storageGroup);
        storageGroupProcessor.setDataTTL(dataTTL);
    }

    public void deleteStorageGroup(String storageGroupName) {
        this.deleteAllDataFilesInOneStorageGroup(storageGroupName);
        StorageGroupProcessor processor = this.processorMap.remove(storageGroupName);
        if (processor != null) {
            processor.deleteFolder(this.systemDir);
        }
    }

    public void loadNewTsFile(TsFileResource newTsFileResource) throws TsFileProcessorException, StorageEngineException {
        this.getProcessor(newTsFileResource.getFile().getParentFile().getName()).loadNewTsFile(newTsFileResource);
    }

    public void deleteTsfile(File deletedTsfile) throws StorageEngineException {
        this.getProcessor(deletedTsfile.getParentFile().getName()).deleteTsfile(deletedTsfile);
    }
}

