/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iotdb.db.engine.cq;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.ReentrantLock;
import org.apache.iotdb.db.concurrent.IoTDBThreadPoolFactory;
import org.apache.iotdb.db.conf.IoTDBDescriptor;
import org.apache.iotdb.db.engine.cq.CQLogReader;
import org.apache.iotdb.db.engine.cq.CQLogWriter;
import org.apache.iotdb.db.engine.cq.ContinuousQuerySchemaCheckTask;
import org.apache.iotdb.db.engine.cq.ContinuousQueryTask;
import org.apache.iotdb.db.engine.cq.ContinuousQueryTaskPoolManager;
import org.apache.iotdb.db.engine.fileSystem.SystemFileFactory;
import org.apache.iotdb.db.exception.ContinuousQueryException;
import org.apache.iotdb.db.exception.StartupException;
import org.apache.iotdb.db.qp.physical.PhysicalPlan;
import org.apache.iotdb.db.qp.physical.sys.CreateContinuousQueryPlan;
import org.apache.iotdb.db.qp.physical.sys.DropContinuousQueryPlan;
import org.apache.iotdb.db.qp.utils.DateTimeUtils;
import org.apache.iotdb.db.query.dataset.ShowContinuousQueriesResult;
import org.apache.iotdb.db.service.IService;
import org.apache.iotdb.db.service.ServiceType;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ContinuousQueryService
implements IService {
    private static final Logger LOGGER = LoggerFactory.getLogger(ContinuousQueryService.class);
    private static final long SYSTEM_STARTUP_TIME = DateTimeUtils.currentTime();
    private static final ContinuousQueryTaskPoolManager TASK_POOL_MANAGER = ContinuousQueryTaskPoolManager.getInstance();
    private static final long TASK_SUBMIT_CHECK_INTERVAL = IoTDBDescriptor.getInstance().getConfig().getContinuousQueryMinimumEveryInterval() / 2L;
    private static final String LOG_FILE_DIR = IoTDBDescriptor.getInstance().getConfig().getSystemDir() + File.separator + "cq" + File.separator;
    private static final String LOG_FILE_NAME = LOG_FILE_DIR + "cqlog.bin";
    private ScheduledExecutorService continuousQueryTaskSubmitThread;
    private final ConcurrentHashMap<String, CreateContinuousQueryPlan> continuousQueryPlans = new ConcurrentHashMap();
    private final ConcurrentHashMap<String, Long> nextExecutionTimestamps = new ConcurrentHashMap();
    private CQLogWriter logWriter;
    private final ReentrantLock registrationLock = new ReentrantLock();
    private static final ContinuousQueryService INSTANCE = new ContinuousQueryService();

    public void doRecovery() throws StartupException {
        try {
            File logDir = SystemFileFactory.INSTANCE.getFile(LOG_FILE_DIR);
            if (!logDir.exists()) {
                logDir.mkdir();
                return;
            }
            File logFile = SystemFileFactory.INSTANCE.getFile(LOG_FILE_NAME);
            if (!logFile.exists()) {
                return;
            }
            try (CQLogReader logReader = new CQLogReader(logFile);){
                block15: while (logReader.hasNext()) {
                    PhysicalPlan plan = logReader.next();
                    switch (plan.getOperatorType()) {
                        case CREATE_CONTINUOUS_QUERY: {
                            CreateContinuousQueryPlan createContinuousQueryPlan = (CreateContinuousQueryPlan)plan;
                            this.register(createContinuousQueryPlan, false);
                            continue block15;
                        }
                        case DROP_CONTINUOUS_QUERY: {
                            DropContinuousQueryPlan dropContinuousQueryPlan = (DropContinuousQueryPlan)plan;
                            this.deregister(dropContinuousQueryPlan, false);
                            continue block15;
                        }
                    }
                    LOGGER.error("Unrecognizable command {}", (Object)plan.getOperatorType());
                }
            }
        }
        catch (IOException | ContinuousQueryException e) {
            LOGGER.error("Error occurred during restart CQService");
            throw new StartupException(e);
        }
    }

    @Override
    public ServiceType getID() {
        return ServiceType.CONTINUOUS_QUERY_SERVICE;
    }

    @Override
    public void start() throws StartupException {
        try {
            this.doRecovery();
            this.logWriter = new CQLogWriter(LOG_FILE_NAME);
            for (CreateContinuousQueryPlan plan : this.continuousQueryPlans.values()) {
                this.nextExecutionTimestamps.put(plan.getContinuousQueryName(), this.calculateNextExecutionTimestamp(plan, SYSTEM_STARTUP_TIME));
            }
            this.continuousQueryTaskSubmitThread = IoTDBThreadPoolFactory.newSingleThreadScheduledExecutor("CQ-Task-Submit-Thread");
            this.continuousQueryTaskSubmitThread.scheduleAtFixedRate(this::checkAndSubmitTasks, 0L, TASK_SUBMIT_CHECK_INTERVAL, DateTimeUtils.timestampPrecisionStringToTimeUnit(IoTDBDescriptor.getInstance().getConfig().getTimestampPrecision()));
            LOGGER.info("Continuous query service started.");
        }
        catch (IOException e) {
            throw new StartupException(e);
        }
    }

    private long calculateNextExecutionTimestamp(CreateContinuousQueryPlan plan, long currentTime) {
        long expectedFirstExecutionTime = plan.getFirstExecutionTimeBoundary() + plan.getForInterval();
        if (currentTime <= expectedFirstExecutionTime) {
            return expectedFirstExecutionTime;
        }
        long durationFromExpectedFirstExecutionTime = currentTime - expectedFirstExecutionTime;
        long everyInterval = plan.getEveryInterval();
        return expectedFirstExecutionTime + everyInterval * (durationFromExpectedFirstExecutionTime / everyInterval + (long)(durationFromExpectedFirstExecutionTime % everyInterval == 0L ? 0 : 1));
    }

    private void checkAndSubmitTasks() {
        long currentTimestamp = DateTimeUtils.currentTime();
        for (CreateContinuousQueryPlan plan : this.continuousQueryPlans.values()) {
            long nextExecutionTimestamp;
            for (nextExecutionTimestamp = this.nextExecutionTimestamps.get(plan.getContinuousQueryName()).longValue(); currentTimestamp >= nextExecutionTimestamp; nextExecutionTimestamp += plan.getEveryInterval()) {
                TASK_POOL_MANAGER.submit(new ContinuousQueryTask(plan, nextExecutionTimestamp));
            }
            this.nextExecutionTimestamps.replace(plan.getContinuousQueryName(), nextExecutionTimestamp);
        }
    }

    @Override
    public void stop() {
        try {
            if (this.continuousQueryTaskSubmitThread != null) {
                this.continuousQueryTaskSubmitThread.shutdown();
                try {
                    this.continuousQueryTaskSubmitThread.awaitTermination(600L, TimeUnit.MILLISECONDS);
                }
                catch (InterruptedException e) {
                    LOGGER.warn("Check thread still doesn't exit after 60s");
                    this.continuousQueryTaskSubmitThread.shutdownNow();
                    Thread.currentThread().interrupt();
                }
            }
            this.continuousQueryPlans.clear();
            if (this.logWriter != null) {
                this.logWriter.close();
                this.logWriter = null;
            }
        }
        catch (IOException e) {
            LOGGER.warn("Something wrong occurred While stopping CQService: {}", (Object)e.getMessage());
        }
    }

    public void acquireRegistrationLock() {
        this.registrationLock.lock();
    }

    public void releaseRegistrationLock() {
        this.registrationLock.unlock();
    }

    public boolean register(CreateContinuousQueryPlan plan, boolean shouldWriteLog) throws ContinuousQueryException {
        if (this.continuousQueryPlans.containsKey(plan.getContinuousQueryName())) {
            throw new ContinuousQueryException(String.format("Continuous Query [%s] already exists", plan.getContinuousQueryName()));
        }
        if (shouldWriteLog) {
            this.checkSchemaBeforeRegistration(plan);
        }
        this.acquireRegistrationLock();
        try {
            if (shouldWriteLog) {
                this.logWriter.createContinuousQuery(plan);
            }
            this.doRegister(plan);
            boolean bl = true;
            return bl;
        }
        catch (Exception e) {
            throw new ContinuousQueryException(e.getMessage());
        }
        finally {
            this.releaseRegistrationLock();
        }
    }

    private void checkSchemaBeforeRegistration(CreateContinuousQueryPlan plan) throws ContinuousQueryException {
        try {
            new ContinuousQuerySchemaCheckTask(plan, plan.getFirstExecutionTimeBoundary()).run();
        }
        catch (Exception e) {
            throw new ContinuousQueryException("Failed to create continuous query task.", e);
        }
    }

    private void doRegister(CreateContinuousQueryPlan plan) {
        this.continuousQueryPlans.put(plan.getContinuousQueryName(), plan);
        this.nextExecutionTimestamps.put(plan.getContinuousQueryName(), this.calculateNextExecutionTimestamp(plan, DateTimeUtils.currentTime()));
    }

    public void deregisterAll() throws ContinuousQueryException {
        for (String cqName : this.continuousQueryPlans.keySet()) {
            this.deregister(new DropContinuousQueryPlan(cqName), false);
        }
    }

    public boolean deregister(DropContinuousQueryPlan plan, boolean shouldWriteLog) throws ContinuousQueryException {
        if (!this.continuousQueryPlans.containsKey(plan.getContinuousQueryName())) {
            throw new ContinuousQueryException(String.format("Continuous Query [%s] does not exist", plan.getContinuousQueryName()));
        }
        this.acquireRegistrationLock();
        try {
            if (shouldWriteLog) {
                this.logWriter.dropContinuousQuery(plan);
            }
            this.doDeregister(plan);
            boolean bl = true;
            return bl;
        }
        catch (Exception e) {
            throw new ContinuousQueryException(e.getMessage());
        }
        finally {
            this.releaseRegistrationLock();
        }
    }

    private void doDeregister(DropContinuousQueryPlan plan) {
        this.continuousQueryPlans.remove(plan.getContinuousQueryName());
        this.nextExecutionTimestamps.remove(plan.getContinuousQueryName());
    }

    public List<ShowContinuousQueriesResult> getShowContinuousQueriesResultList() {
        ArrayList<ShowContinuousQueriesResult> results = new ArrayList<ShowContinuousQueriesResult>(this.continuousQueryPlans.size());
        for (CreateContinuousQueryPlan plan : this.continuousQueryPlans.values()) {
            results.add(new ShowContinuousQueriesResult(plan.getQuerySql(), plan.getContinuousQueryName(), plan.getTargetPath(), plan.getEveryInterval(), plan.getForInterval(), plan.getFirstExecutionTimeBoundary()));
        }
        return results;
    }

    private ContinuousQueryService() {
    }

    public static ContinuousQueryService getInstance() {
        return INSTANCE;
    }
}

