/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iotdb.db.engine.merge.manage;

import com.google.common.util.concurrent.RateLimiter;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentSkipListSet;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import org.apache.iotdb.db.concurrent.ThreadName;
import org.apache.iotdb.db.conf.IoTDBDescriptor;
import org.apache.iotdb.db.engine.StorageEngine;
import org.apache.iotdb.db.engine.merge.manage.MergeFuture;
import org.apache.iotdb.db.engine.merge.manage.MergeManagerMBean;
import org.apache.iotdb.db.engine.merge.manage.MergeThreadPool;
import org.apache.iotdb.db.engine.merge.task.MergeMultiChunkTask;
import org.apache.iotdb.db.engine.merge.task.MergeTask;
import org.apache.iotdb.db.exception.StorageEngineException;
import org.apache.iotdb.db.service.IService;
import org.apache.iotdb.db.service.JMXService;
import org.apache.iotdb.db.service.ServiceType;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MergeManager
implements IService,
MergeManagerMBean {
    private static final Logger logger = LoggerFactory.getLogger(MergeManager.class);
    private static final MergeManager INSTANCE = new MergeManager();
    private final String mbeanName = String.format("%s:%s=%s", "org.apache.iotdb.service", "type", this.getID().getJmxName());
    private final RateLimiter mergeWriteRateLimiter = RateLimiter.create((double)Double.MAX_VALUE);
    private AtomicInteger threadCnt = new AtomicInteger();
    private ThreadPoolExecutor mergeTaskPool;
    private ThreadPoolExecutor mergeChunkSubTaskPool;
    private ScheduledExecutorService timedMergeThreadPool;
    private ScheduledExecutorService taskCleanerThreadPool;
    private Map<String, Set<MergeFuture>> storageGroupMainTasks = new ConcurrentHashMap<String, Set<MergeFuture>>();
    private Map<String, Set<MergeFuture>> storageGroupSubTasks = new ConcurrentHashMap<String, Set<MergeFuture>>();

    private MergeManager() {
    }

    public RateLimiter getMergeWriteRateLimiter() {
        this.setWriteMergeRate(IoTDBDescriptor.getInstance().getConfig().getMergeWriteThroughputMbPerSec());
        return this.mergeWriteRateLimiter;
    }

    public static void mergeRateLimiterAcquire(RateLimiter limiter, long bytesLength) {
        while (bytesLength >= Integer.MAX_VALUE) {
            limiter.acquire(Integer.MAX_VALUE);
            bytesLength -= Integer.MAX_VALUE;
        }
        if (bytesLength > 0L) {
            limiter.acquire((int)bytesLength);
        }
    }

    private void setWriteMergeRate(double throughoutMbPerSec) {
        double throughout = throughoutMbPerSec * 1024.0 * 1024.0;
        if (throughout == 0.0) {
            throughout = Double.MAX_VALUE;
        }
        if (this.mergeWriteRateLimiter.getRate() != throughout) {
            this.mergeWriteRateLimiter.setRate(throughout);
        }
    }

    public static MergeManager getINSTANCE() {
        return INSTANCE;
    }

    public void submitMainTask(MergeTask mergeTask) {
        MergeFuture future = (MergeFuture)this.mergeTaskPool.submit(mergeTask);
        this.storageGroupMainTasks.computeIfAbsent(mergeTask.getStorageGroupName(), k -> new ConcurrentSkipListSet()).add(future);
    }

    public Future<Void> submitChunkSubTask(MergeMultiChunkTask.MergeChunkHeapTask task) {
        MergeFuture future = (MergeFuture)this.mergeChunkSubTaskPool.submit(task);
        this.storageGroupSubTasks.computeIfAbsent(task.getStorageGroupName(), k -> new ConcurrentSkipListSet()).add(future);
        return future;
    }

    @Override
    public void start() {
        JMXService.registerMBean(this, this.mbeanName);
        if (this.mergeTaskPool == null) {
            int chunkSubThreadNum;
            int threadNum = IoTDBDescriptor.getInstance().getConfig().getMergeThreadNum();
            if (threadNum <= 0) {
                threadNum = 1;
            }
            if ((chunkSubThreadNum = IoTDBDescriptor.getInstance().getConfig().getMergeChunkSubThreadNum()) <= 0) {
                chunkSubThreadNum = 1;
            }
            this.mergeTaskPool = new MergeThreadPool(threadNum, r -> new Thread(r, "MergeThread-" + this.threadCnt.getAndIncrement()));
            this.mergeChunkSubTaskPool = new MergeThreadPool(threadNum * chunkSubThreadNum, r -> new Thread(r, "MergeChunkSubThread-" + this.threadCnt.getAndIncrement()));
            long mergeInterval = IoTDBDescriptor.getInstance().getConfig().getMergeIntervalSec();
            if (mergeInterval > 0L) {
                this.timedMergeThreadPool = Executors.newSingleThreadScheduledExecutor(r -> new Thread(r, "TimedMergeThread"));
                this.timedMergeThreadPool.scheduleAtFixedRate(this::mergeAll, mergeInterval, mergeInterval, TimeUnit.SECONDS);
            }
            this.taskCleanerThreadPool = Executors.newSingleThreadScheduledExecutor(r -> new Thread(r, "MergeTaskCleaner"));
            this.taskCleanerThreadPool.scheduleAtFixedRate(this::cleanFinishedTask, 30L, 30L, TimeUnit.MINUTES);
            logger.info("MergeManager started");
        }
    }

    @Override
    public void stop() {
        if (this.mergeTaskPool != null) {
            if (this.timedMergeThreadPool != null) {
                this.timedMergeThreadPool.shutdownNow();
                this.timedMergeThreadPool = null;
            }
            this.taskCleanerThreadPool.shutdownNow();
            this.taskCleanerThreadPool = null;
            this.mergeTaskPool.shutdownNow();
            this.mergeChunkSubTaskPool.shutdownNow();
            logger.info("Waiting for task pool to shut down");
            long startTime = System.currentTimeMillis();
            while (!this.mergeTaskPool.isTerminated() || !this.mergeChunkSubTaskPool.isTerminated()) {
                int timeMillis = 0;
                try {
                    Thread.sleep(200L);
                }
                catch (InterruptedException e) {
                    logger.error("CompactionMergeTaskPoolManager {} shutdown", (Object)ThreadName.COMPACTION_SERVICE.getName(), (Object)e);
                    Thread.currentThread().interrupt();
                }
                long time = System.currentTimeMillis() - startTime;
                if ((timeMillis += 200) % 60000 != 0) continue;
                logger.warn("MergeManager has wait for {} seconds to stop", (Object)(time / 1000L));
            }
            this.mergeTaskPool = null;
            this.storageGroupMainTasks.clear();
            this.storageGroupSubTasks.clear();
            logger.info("MergeManager stopped");
        }
        JMXService.deregisterMBean(this.mbeanName);
    }

    @Override
    public void waitAndStop(long milliseconds) {
        if (this.mergeTaskPool != null) {
            if (this.timedMergeThreadPool != null) {
                this.awaitTermination(this.timedMergeThreadPool, milliseconds);
                this.timedMergeThreadPool = null;
            }
            this.awaitTermination(this.taskCleanerThreadPool, milliseconds);
            this.taskCleanerThreadPool = null;
            this.awaitTermination(this.mergeTaskPool, milliseconds);
            this.awaitTermination(this.mergeChunkSubTaskPool, milliseconds);
            logger.info("Waiting for task pool to shut down");
            long startTime = System.currentTimeMillis();
            while (!this.mergeTaskPool.isTerminated() || !this.mergeChunkSubTaskPool.isTerminated()) {
                int timeMillis = 0;
                try {
                    Thread.sleep(200L);
                }
                catch (InterruptedException e) {
                    logger.error("CompactionMergeTaskPoolManager {} shutdown", (Object)ThreadName.COMPACTION_SERVICE.getName(), (Object)e);
                    Thread.currentThread().interrupt();
                }
                long time = System.currentTimeMillis() - startTime;
                if ((timeMillis += 200) % 60000 != 0) continue;
                logger.warn("MergeManager has wait for {} seconds to stop", (Object)(time / 1000L));
            }
            this.mergeTaskPool = null;
            this.storageGroupMainTasks.clear();
            this.storageGroupSubTasks.clear();
            logger.info("MergeManager stopped");
        }
    }

    private void awaitTermination(ExecutorService service, long milliseconds) {
        try {
            service.shutdown();
            service.awaitTermination(milliseconds, TimeUnit.MILLISECONDS);
        }
        catch (InterruptedException e) {
            logger.warn("MergeThreadPool can not be closed in {} ms", (Object)milliseconds);
            Thread.currentThread().interrupt();
        }
        service.shutdownNow();
    }

    @Override
    public ServiceType getID() {
        return ServiceType.MERGE_SERVICE;
    }

    private void mergeAll() {
        try {
            StorageEngine.getInstance().mergeAll(IoTDBDescriptor.getInstance().getConfig().isForceFullMerge());
        }
        catch (StorageEngineException e) {
            logger.error("Cannot perform a global merge because", (Throwable)e);
        }
    }

    @Override
    public void abortMerge(String storageGroup) {
        Set subTasks = this.storageGroupSubTasks.getOrDefault(storageGroup, Collections.emptySet());
        Iterator subIterator = subTasks.iterator();
        while (subIterator.hasNext()) {
            Future next = (Future)subIterator.next();
            if (!next.isDone() && !next.isCancelled()) {
                next.cancel(true);
            }
            subIterator.remove();
        }
        Set mainTasks = this.storageGroupMainTasks.getOrDefault(storageGroup, Collections.emptySet());
        Iterator mainIterator = mainTasks.iterator();
        while (mainIterator.hasNext()) {
            Future next = (Future)mainIterator.next();
            if (!next.isDone() && !next.isCancelled()) {
                next.cancel(true);
            }
            mainIterator.remove();
        }
    }

    private void cleanFinishedTask() {
        for (Set<MergeFuture> subTasks : this.storageGroupSubTasks.values()) {
            subTasks.removeIf(next -> next.isDone() || next.isCancelled());
        }
        for (Set<MergeFuture> mainTasks : this.storageGroupMainTasks.values()) {
            mainTasks.removeIf(next -> next.isDone() || next.isCancelled());
        }
    }

    public Map<String, List<TaskStatus>>[] collectTaskStatus() {
        Set<MergeFuture> tasks;
        String storageGroup;
        Map[] result = new Map[]{new HashMap(), new HashMap()};
        for (Map.Entry<String, Set<MergeFuture>> stringSetEntry : this.storageGroupMainTasks.entrySet()) {
            storageGroup = stringSetEntry.getKey();
            tasks = stringSetEntry.getValue();
            for (MergeFuture task : tasks) {
                result[0].computeIfAbsent(storageGroup, s -> new ArrayList()).add(new TaskStatus(task));
            }
        }
        for (Map.Entry<String, Set<MergeFuture>> stringSetEntry : this.storageGroupSubTasks.entrySet()) {
            storageGroup = stringSetEntry.getKey();
            tasks = stringSetEntry.getValue();
            for (MergeFuture task : tasks) {
                result[1].computeIfAbsent(storageGroup, s -> new ArrayList()).add(new TaskStatus(task));
            }
        }
        return result;
    }

    public String genMergeTaskReport() {
        List<TaskStatus> statusList;
        String storageGroup;
        Map<String, List<TaskStatus>>[] statusMaps = this.collectTaskStatus();
        StringBuilder builder = new StringBuilder("Main tasks:").append(System.lineSeparator());
        for (Map.Entry<String, List<TaskStatus>> stringListEntry : statusMaps[0].entrySet()) {
            storageGroup = stringListEntry.getKey();
            statusList = stringListEntry.getValue();
            builder.append("\t").append("Storage group: ").append(storageGroup).append(System.lineSeparator());
            for (TaskStatus status : statusList) {
                builder.append("\t\t").append(status).append(System.lineSeparator());
            }
        }
        builder.append("Sub tasks:").append(System.lineSeparator());
        for (Map.Entry<String, List<TaskStatus>> stringListEntry : statusMaps[1].entrySet()) {
            storageGroup = stringListEntry.getKey();
            statusList = stringListEntry.getValue();
            builder.append("\t").append("Storage group: ").append(storageGroup).append(System.lineSeparator());
            for (TaskStatus status : statusList) {
                builder.append("\t\t").append(status).append(System.lineSeparator());
            }
        }
        return builder.toString();
    }

    @Override
    public void printMergeStatus() {
        if (logger.isInfoEnabled()) {
            logger.info("Running tasks:\n {}", (Object)this.genMergeTaskReport());
        }
    }

    public static class TaskStatus {
        private String taskName;
        private String createdTime;
        private String progress;
        private boolean isDone;
        private boolean isCancelled;

        public TaskStatus(MergeFuture future) {
            this.taskName = future.getTaskName();
            this.createdTime = future.getCreatedTime();
            this.progress = future.getProgress();
            this.isCancelled = future.isCancelled();
            this.isDone = future.isDone();
        }

        public String toString() {
            return String.format("%s, %s, %s, done:%s, cancelled:%s", this.taskName, this.createdTime, this.progress, this.isDone, this.isCancelled);
        }

        public String getTaskName() {
            return this.taskName;
        }

        public String getCreatedTime() {
            return this.createdTime;
        }

        public String getProgress() {
            return this.progress;
        }

        public boolean isDone() {
            return this.isDone;
        }

        public boolean isCancelled() {
            return this.isCancelled;
        }
    }
}

