/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iotdb.cluster.log.manage;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import org.apache.commons.lang3.concurrent.BasicThreadFactory;
import org.apache.iotdb.cluster.config.ClusterDescriptor;
import org.apache.iotdb.cluster.exception.EntryCompactedException;
import org.apache.iotdb.cluster.exception.EntryUnavailableException;
import org.apache.iotdb.cluster.exception.GetEntriesWrongParametersException;
import org.apache.iotdb.cluster.exception.LogExecutionException;
import org.apache.iotdb.cluster.exception.TruncateCommittedEntryException;
import org.apache.iotdb.cluster.log.HardState;
import org.apache.iotdb.cluster.log.Log;
import org.apache.iotdb.cluster.log.LogApplier;
import org.apache.iotdb.cluster.log.Snapshot;
import org.apache.iotdb.cluster.log.StableEntryManager;
import org.apache.iotdb.cluster.log.manage.CommittedEntryManager;
import org.apache.iotdb.cluster.log.manage.UnCommittedEntryManager;
import org.apache.iotdb.cluster.server.monitor.Timer;
import org.apache.iotdb.tsfile.utils.RamUsageEstimator;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class RaftLogManager {
    private static final Logger logger = LoggerFactory.getLogger(RaftLogManager.class);
    private UnCommittedEntryManager unCommittedEntryManager;
    private CommittedEntryManager committedEntryManager;
    private StableEntryManager stableEntryManager;
    private long commitIndex;
    private volatile long maxHaveAppliedCommitIndex;
    private final Object changeApplyCommitIndexCond = new Object();
    private volatile long blockAppliedCommitIndex;
    private LogApplier logApplier;
    private String name;
    private ScheduledExecutorService deleteLogExecutorService;
    private ScheduledFuture<?> deleteLogFuture;
    private ExecutorService checkLogApplierExecutorService;
    private Future<?> checkLogApplierFuture;
    private int minNumOfLogsInMem = ClusterDescriptor.getInstance().getConfig().getMinNumOfLogsInMem();
    private int maxNumOfLogsInMem = ClusterDescriptor.getInstance().getConfig().getMaxNumOfLogsInMem();
    private long maxLogMemSize = ClusterDescriptor.getInstance().getConfig().getMaxMemorySizeForRaftLog();
    private final Object[] logUpdateConditions = new Object[1024];
    private List<Log> blockedUnappliedLogList;

    protected RaftLogManager(StableEntryManager stableEntryManager, LogApplier applier, String name) {
        this.logApplier = applier;
        this.name = name;
        this.setCommittedEntryManager(new CommittedEntryManager(this.maxNumOfLogsInMem));
        this.setStableEntryManager(stableEntryManager);
        try {
            this.getCommittedEntryManager().append(stableEntryManager.getAllEntriesAfterAppliedIndex());
        }
        catch (TruncateCommittedEntryException e) {
            logger.error("{}: Unexpected error:", (Object)name, (Object)e);
        }
        long first = this.getCommittedEntryManager().getDummyIndex();
        long last = this.getCommittedEntryManager().getLastIndex();
        this.setUnCommittedEntryManager(new UnCommittedEntryManager(last + 1L));
        this.commitIndex = last;
        this.maxHaveAppliedCommitIndex = first;
        this.blockAppliedCommitIndex = -1L;
        this.blockedUnappliedLogList = new CopyOnWriteArrayList<Log>();
        this.deleteLogExecutorService = new ScheduledThreadPoolExecutor(1, (ThreadFactory)new BasicThreadFactory.Builder().namingPattern("raft-log-delete-" + name).daemon(true).build());
        this.checkLogApplierExecutorService = Executors.newSingleThreadExecutor((ThreadFactory)new BasicThreadFactory.Builder().namingPattern("check-log-applier-" + name).daemon(true).build());
        int logDeleteCheckIntervalSecond = ClusterDescriptor.getInstance().getConfig().getLogDeleteCheckIntervalSecond();
        if (logDeleteCheckIntervalSecond > 0) {
            this.deleteLogFuture = this.deleteLogExecutorService.scheduleAtFixedRate(this::checkDeleteLog, logDeleteCheckIntervalSecond, logDeleteCheckIntervalSecond, TimeUnit.SECONDS);
        }
        this.checkLogApplierFuture = this.checkLogApplierExecutorService.submit(this::checkAppliedLogIndex);
        if (ClusterDescriptor.getInstance().getConfig().isEnableRaftLogPersistence()) {
            this.applyAllCommittedLogWhenStartUp();
        }
        for (int i = 0; i < this.logUpdateConditions.length; ++i) {
            this.logUpdateConditions[i] = new Object();
        }
    }

    public Snapshot getSnapshot() {
        return this.getSnapshot(-1L);
    }

    public abstract Snapshot getSnapshot(long var1);

    public void takeSnapshot() throws IOException {
        if (this.commitIndex <= 0L) {
            return;
        }
        long startTime = System.currentTimeMillis();
        if (this.blockAppliedCommitIndex < 0L) {
            return;
        }
        logger.info("{}: before take snapshot, blockAppliedCommitIndex={}, maxHaveAppliedCommitIndex={}, commitIndex={}", new Object[]{this.name, this.blockAppliedCommitIndex, this.maxHaveAppliedCommitIndex, this.commitIndex});
        while (this.blockAppliedCommitIndex > this.maxHaveAppliedCommitIndex) {
            long waitTime = System.currentTimeMillis() - startTime;
            if (waitTime <= (long)ClusterDescriptor.getInstance().getConfig().getCatchUpTimeoutMS()) continue;
            logger.error("{}: wait all log applied time out, time cost={}, blockAppliedCommitIndex={}, maxHaveAppliedCommitIndex={},commitIndex={}", new Object[]{this.name, waitTime, this.blockAppliedCommitIndex, this.maxHaveAppliedCommitIndex, this.commitIndex});
            throw new IOException("wait all log applied time out");
        }
    }

    public void updateHardState(HardState state) {
        this.getStableEntryManager().setHardStateAndFlush(state);
    }

    public HardState getHardState() {
        return this.getStableEntryManager().getHardState();
    }

    public long getCommitLogIndex() {
        return this.commitIndex;
    }

    public long getFirstIndex() {
        return this.getCommittedEntryManager().getFirstIndex();
    }

    public long getLastLogIndex() {
        long last = this.getUnCommittedEntryManager().maybeLastIndex();
        if (last != -1L) {
            return last;
        }
        return this.getCommittedEntryManager().getLastIndex();
    }

    public long getTerm(long index) throws EntryCompactedException {
        long term;
        long dummyIndex = this.getFirstIndex() - 1L;
        if (index < dummyIndex) {
            if (ClusterDescriptor.getInstance().getConfig().isEnableRaftLogPersistence()) {
                List<Log> logsInDisk = this.getStableEntryManager().getLogs(index, index);
                if (logsInDisk.isEmpty()) {
                    return -1L;
                }
                return logsInDisk.get(0).getCurrLogTerm();
            }
            return -1L;
        }
        long lastIndex = this.getLastLogIndex();
        if (index > lastIndex) {
            return -1L;
        }
        if (index >= this.getUnCommittedEntryManager().getFirstUnCommittedIndex() && (term = this.getUnCommittedEntryManager().maybeTerm(index)) != -1L) {
            return term;
        }
        return this.getCommittedEntryManager().maybeTerm(index);
    }

    public long getLastLogTerm() {
        long term = -1L;
        try {
            term = this.getTerm(this.getLastLogIndex());
        }
        catch (Exception e) {
            logger.error("{}: unexpected error when getting the last term : {}", (Object)this.name, (Object)e.getMessage());
        }
        return term;
    }

    public long getCommitLogTerm() {
        long term = -1L;
        try {
            term = this.getTerm(this.getCommitLogIndex());
        }
        catch (Exception e) {
            logger.error("{}: unexpected error when getting the last term : {}", (Object)this.name, (Object)e.getMessage());
        }
        return term;
    }

    public long maybeAppend(long lastIndex, long lastTerm, long leaderCommit, List<Log> entries) {
        if (this.matchTerm(lastTerm, lastIndex)) {
            long newLastIndex = lastIndex + (long)entries.size();
            long ci = this.findConflict(entries);
            if (ci <= this.commitIndex) {
                if (ci != -1L) {
                    logger.error("{}: entry {} conflict with committed entry [commitIndex({})]", new Object[]{this.name, ci, this.commitIndex});
                } else if (logger.isDebugEnabled() && !entries.isEmpty()) {
                    logger.debug("{}: Appending entries [{} and other {} logs] all exist locally", new Object[]{this.name, entries.get(0), entries.size() - 1});
                }
            } else {
                long offset = lastIndex + 1L;
                this.append(entries.subList((int)(ci - offset), entries.size()));
            }
            try {
                this.commitTo(Math.min(leaderCommit, newLastIndex));
            }
            catch (LogExecutionException logExecutionException) {
                // empty catch block
            }
            return newLastIndex;
        }
        return -1L;
    }

    public long maybeAppend(long lastIndex, long lastTerm, long leaderCommit, Log entry) {
        if (this.matchTerm(lastTerm, lastIndex)) {
            long newLastIndex = lastIndex + 1L;
            if (entry.getCurrLogIndex() <= this.commitIndex) {
                logger.debug("{}: entry {} conflict with committed entry [commitIndex({})]", new Object[]{this.name, entry.getCurrLogIndex(), this.commitIndex});
            } else {
                this.append(entry);
            }
            try {
                this.commitTo(Math.min(leaderCommit, newLastIndex));
            }
            catch (LogExecutionException logExecutionException) {
                // empty catch block
            }
            return newLastIndex;
        }
        return -1L;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public long append(List<Log> entries) {
        Object logUpdateCondition;
        if (entries.isEmpty()) {
            return this.getLastLogIndex();
        }
        long after = entries.get(0).getCurrLogIndex();
        if (after <= this.commitIndex) {
            logger.error("{}: after({}) is out of range [commitIndex({})]", new Object[]{this.name, after, this.commitIndex});
            return -1L;
        }
        this.getUnCommittedEntryManager().truncateAndAppend(entries);
        Object object = logUpdateCondition = this.getLogUpdateCondition(entries.get(entries.size() - 1).getCurrLogIndex());
        synchronized (object) {
            logUpdateCondition.notifyAll();
        }
        return this.getLastLogIndex();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public long append(Log entry) {
        Object logUpdateCondition;
        long after = entry.getCurrLogIndex();
        if (after <= this.commitIndex) {
            logger.error("{}: after({}) is out of range [commitIndex({})]", new Object[]{this.name, after, this.commitIndex});
            return -1L;
        }
        this.getUnCommittedEntryManager().truncateAndAppend(entry);
        Object object = logUpdateCondition = this.getLogUpdateCondition(entry.getCurrLogIndex());
        synchronized (object) {
            logUpdateCondition.notifyAll();
        }
        return this.getLastLogIndex();
    }

    public synchronized boolean maybeCommit(long leaderCommit, long term) {
        if (leaderCommit > this.commitIndex && this.matchTerm(term, leaderCommit)) {
            try {
                this.commitTo(leaderCommit);
            }
            catch (LogExecutionException logExecutionException) {
                // empty catch block
            }
            return true;
        }
        return false;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void applySnapshot(Snapshot snapshot) {
        logger.info("{}: log module starts to restore snapshot [index: {}, term: {}]", new Object[]{this.name, snapshot.getLastLogIndex(), snapshot.getLastLogTerm()});
        try {
            this.getCommittedEntryManager().compactEntries(snapshot.getLastLogIndex());
            this.getStableEntryManager().removeCompactedEntries(snapshot.getLastLogIndex());
        }
        catch (EntryUnavailableException e) {
            this.getCommittedEntryManager().applyingSnapshot(snapshot);
            this.getUnCommittedEntryManager().applyingSnapshot(snapshot);
        }
        if (this.commitIndex < snapshot.getLastLogIndex()) {
            this.commitIndex = snapshot.getLastLogIndex();
        }
        this.getStableEntryManager().clearAllLogs(this.commitIndex);
        Object object = this.changeApplyCommitIndexCond;
        synchronized (object) {
            if (this.maxHaveAppliedCommitIndex < snapshot.getLastLogIndex()) {
                this.maxHaveAppliedCommitIndex = snapshot.getLastLogIndex();
            }
        }
    }

    public boolean isLogUpToDate(long lastTerm, long lastIndex) {
        return lastTerm > this.getLastLogTerm() || lastTerm == this.getLastLogTerm() && lastIndex >= this.getLastLogIndex();
    }

    public List<Log> getEntries(long low, long high) {
        if (low >= high) {
            return Collections.emptyList();
        }
        ArrayList<Log> entries = new ArrayList<Log>();
        long offset = this.getUnCommittedEntryManager().getFirstUnCommittedIndex();
        if (low < offset) {
            entries.addAll(this.getCommittedEntryManager().getEntries(low, Math.min(high, offset)));
        }
        if (high > offset) {
            entries.addAll(this.getUnCommittedEntryManager().getEntries(Math.max(low, offset), high));
        }
        return entries;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void commitTo(long newCommitIndex) throws LogExecutionException {
        long firstLogIndex;
        if (this.commitIndex >= newCommitIndex) {
            return;
        }
        long startTime = Timer.Statistic.RAFT_SENDER_COMMIT_GET_LOGS.getOperationStartTime();
        long lo = this.getUnCommittedEntryManager().getFirstUnCommittedIndex();
        long hi = newCommitIndex + 1L;
        ArrayList<Log> entries = new ArrayList<Log>(this.getUnCommittedEntryManager().getEntries(lo, hi));
        Timer.Statistic.RAFT_SENDER_COMMIT_GET_LOGS.calOperationCostTimeFromStart(startTime);
        if (entries.isEmpty()) {
            return;
        }
        long commitLogIndex = this.getCommitLogIndex();
        if (commitLogIndex >= (firstLogIndex = ((Log)entries.get(0)).getCurrLogIndex())) {
            logger.warn("Committing logs that has already been committed: {} >= {}", (Object)commitLogIndex, (Object)firstLogIndex);
            entries.subList(0, (int)(this.getCommitLogIndex() - ((Log)entries.get(0)).getCurrLogIndex() + 1L)).clear();
        }
        try {
            boolean needToCompactLog = false;
            int numToReserveForNew = this.minNumOfLogsInMem;
            if (this.committedEntryManager.getTotalSize() + entries.size() > this.maxNumOfLogsInMem) {
                needToCompactLog = true;
                numToReserveForNew = this.maxNumOfLogsInMem - entries.size();
            }
            long newEntryMemSize = 0L;
            for (Log entry : entries) {
                if (entry.getByteSize() == 0L) {
                    logger.debug("{} should not go here, must be send to the follower, so the log has been serialized exclude single node mode", (Object)entry);
                    entry.setByteSize((int)RamUsageEstimator.sizeOf((Object)entry));
                }
                newEntryMemSize += entry.getByteSize();
            }
            int sizeToReserveForNew = this.minNumOfLogsInMem;
            if (newEntryMemSize + this.committedEntryManager.getEntryTotalMemSize() > this.maxLogMemSize) {
                needToCompactLog = true;
                sizeToReserveForNew = this.committedEntryManager.maxLogNumShouldReserve(this.maxLogMemSize - newEntryMemSize);
            }
            if (needToCompactLog) {
                int numForNew = Math.min(numToReserveForNew, sizeToReserveForNew);
                int sizeToReserveForConfig = this.minNumOfLogsInMem;
                startTime = Timer.Statistic.RAFT_SENDER_COMMIT_DELETE_EXCEEDING_LOGS.getOperationStartTime();
                RaftLogManager raftLogManager = this;
                synchronized (raftLogManager) {
                    this.innerDeleteLog(Math.min(sizeToReserveForConfig, numForNew));
                }
                Timer.Statistic.RAFT_SENDER_COMMIT_DELETE_EXCEEDING_LOGS.calOperationCostTimeFromStart(startTime);
            }
            startTime = Timer.Statistic.RAFT_SENDER_COMMIT_APPEND_AND_STABLE_LOGS.getOperationStartTime();
            this.getCommittedEntryManager().append(entries);
            if (ClusterDescriptor.getInstance().getConfig().isEnableRaftLogPersistence()) {
                this.getStableEntryManager().append(entries, this.maxHaveAppliedCommitIndex);
            }
            Log lastLog = (Log)entries.get(entries.size() - 1);
            this.getUnCommittedEntryManager().stableTo(lastLog.getCurrLogIndex());
            Timer.Statistic.RAFT_SENDER_COMMIT_APPEND_AND_STABLE_LOGS.calOperationCostTimeFromStart(startTime);
            this.commitIndex = lastLog.getCurrLogIndex();
            startTime = Timer.Statistic.RAFT_SENDER_COMMIT_APPLY_LOGS.getOperationStartTime();
            this.applyEntries(entries);
            Timer.Statistic.RAFT_SENDER_COMMIT_APPLY_LOGS.calOperationCostTimeFromStart(startTime);
            long unappliedLogSize = commitLogIndex - this.maxHaveAppliedCommitIndex;
            if (unappliedLogSize > (long)ClusterDescriptor.getInstance().getConfig().getMaxNumOfLogsInMem()) {
                logger.debug("There are too many unapplied logs [{}], wait for a while to avoid memory overflow", (Object)unappliedLogSize);
                Thread.sleep(unappliedLogSize - (long)ClusterDescriptor.getInstance().getConfig().getMaxNumOfLogsInMem());
            }
        }
        catch (TruncateCommittedEntryException e) {
            logger.error("{}: Unexpected error:", (Object)this.name, (Object)e);
        }
        catch (IOException e) {
            throw new LogExecutionException(e);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        }
    }

    public boolean matchTerm(long term, long index) {
        long t;
        try {
            t = this.getTerm(index);
        }
        catch (Exception e) {
            return false;
        }
        return t == term;
    }

    void applyEntries(List<Log> entries) {
        for (Log entry : entries) {
            if (this.blockAppliedCommitIndex > 0L && entry.getCurrLogIndex() > this.blockAppliedCommitIndex) {
                this.blockedUnappliedLogList.add(entry);
                continue;
            }
            try {
                this.logApplier.apply(entry);
            }
            catch (Exception e) {
                entry.setException(e);
                entry.setApplied(true);
            }
        }
    }

    void checkBound(long low, long high) throws EntryCompactedException, GetEntriesWrongParametersException {
        if (low > high) {
            logger.error("{}: invalid getEntries: parameter: {} > {}", new Object[]{this.name, low, high});
            throw new GetEntriesWrongParametersException(low, high);
        }
        long first = this.getFirstIndex();
        if (low < first) {
            logger.error("{}: CheckBound out of index: parameter: {} , lower bound: {} ", new Object[]{this.name, low, high});
            throw new EntryCompactedException(low, first);
        }
    }

    long findConflict(List<Log> entries) {
        for (Log entry : entries) {
            if (this.matchTerm(entry.getCurrLogTerm(), entry.getCurrLogIndex())) continue;
            if (entry.getCurrLogIndex() <= this.getLastLogIndex()) {
                logger.info("found conflict at index {}", (Object)entry.getCurrLogIndex());
            }
            return entry.getCurrLogIndex();
        }
        return -1L;
    }

    protected RaftLogManager(CommittedEntryManager committedEntryManager, StableEntryManager stableEntryManager, LogApplier applier) {
        this.setCommittedEntryManager(committedEntryManager);
        this.setStableEntryManager(stableEntryManager);
        this.logApplier = applier;
        long first = committedEntryManager.getFirstIndex();
        long last = committedEntryManager.getLastIndex();
        this.setUnCommittedEntryManager(new UnCommittedEntryManager(last + 1L));
        this.commitIndex = last;
        this.maxHaveAppliedCommitIndex = first;
        this.blockAppliedCommitIndex = -1L;
        this.blockedUnappliedLogList = new CopyOnWriteArrayList<Log>();
        this.checkLogApplierExecutorService = Executors.newSingleThreadExecutor((ThreadFactory)new BasicThreadFactory.Builder().namingPattern("check-log-applier-" + this.name).daemon(true).build());
        this.checkLogApplierFuture = this.checkLogApplierExecutorService.submit(this::checkAppliedLogIndex);
        for (int i = 0; i < this.logUpdateConditions.length; ++i) {
            this.logUpdateConditions[i] = new Object();
        }
    }

    void setMinNumOfLogsInMem(int minNumOfLogsInMem) {
        this.minNumOfLogsInMem = minNumOfLogsInMem;
    }

    public void setMaxHaveAppliedCommitIndex(long maxHaveAppliedCommitIndex) {
        this.checkLogApplierExecutorService.shutdownNow();
        try {
            this.checkLogApplierExecutorService.awaitTermination(5L, TimeUnit.SECONDS);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        }
        this.maxHaveAppliedCommitIndex = maxHaveAppliedCommitIndex;
    }

    public void close() {
        this.getStableEntryManager().close();
        if (this.deleteLogExecutorService != null) {
            this.deleteLogExecutorService.shutdownNow();
            if (this.deleteLogFuture != null) {
                this.deleteLogFuture.cancel(true);
            }
            try {
                this.deleteLogExecutorService.awaitTermination(20L, TimeUnit.SECONDS);
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                logger.warn("Close delete log thread interrupted");
            }
            this.deleteLogExecutorService = null;
        }
        if (this.checkLogApplierExecutorService != null) {
            this.checkLogApplierExecutorService.shutdownNow();
            this.checkLogApplierFuture.cancel(true);
            try {
                this.checkLogApplierExecutorService.awaitTermination(20L, TimeUnit.SECONDS);
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                logger.warn("Close check log applier thread interrupted");
            }
            this.checkLogApplierExecutorService = null;
        }
        if (this.logApplier != null) {
            this.logApplier.close();
        }
    }

    UnCommittedEntryManager getUnCommittedEntryManager() {
        return this.unCommittedEntryManager;
    }

    private void setUnCommittedEntryManager(UnCommittedEntryManager unCommittedEntryManager) {
        this.unCommittedEntryManager = unCommittedEntryManager;
    }

    CommittedEntryManager getCommittedEntryManager() {
        return this.committedEntryManager;
    }

    private void setCommittedEntryManager(CommittedEntryManager committedEntryManager) {
        this.committedEntryManager = committedEntryManager;
    }

    public StableEntryManager getStableEntryManager() {
        return this.stableEntryManager;
    }

    private void setStableEntryManager(StableEntryManager stableEntryManager) {
        this.stableEntryManager = stableEntryManager;
    }

    public long getMaxHaveAppliedCommitIndex() {
        return this.maxHaveAppliedCommitIndex;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void checkDeleteLog() {
        try {
            RaftLogManager raftLogManager = this;
            synchronized (raftLogManager) {
                if (this.committedEntryManager.getTotalSize() <= this.minNumOfLogsInMem) {
                    return;
                }
                this.innerDeleteLog(this.minNumOfLogsInMem);
            }
        }
        catch (Exception e) {
            logger.error("{}, error occurred when checking delete log", (Object)this.name, (Object)e);
        }
    }

    private void innerDeleteLog(int sizeToReserve) {
        long removeSize = (long)this.committedEntryManager.getTotalSize() - (long)sizeToReserve;
        if (removeSize <= 0L) {
            return;
        }
        long compactIndex = Math.min(this.committedEntryManager.getDummyIndex() + removeSize, this.maxHaveAppliedCommitIndex - 1L);
        try {
            logger.debug("{}: Before compaction index {}-{}, compactIndex {}, removeSize {}, committedLogSize {}, maxAppliedLog {}", new Object[]{this.name, this.getFirstIndex(), this.getLastLogIndex(), compactIndex, removeSize, this.committedEntryManager.getTotalSize(), this.maxHaveAppliedCommitIndex});
            this.getCommittedEntryManager().compactEntries(compactIndex);
            if (ClusterDescriptor.getInstance().getConfig().isEnableRaftLogPersistence()) {
                this.getStableEntryManager().removeCompactedEntries(compactIndex);
            }
            logger.debug("{}: After compaction index {}-{}, committedLogSize {}", new Object[]{this.name, this.getFirstIndex(), this.getLastLogIndex(), this.committedEntryManager.getTotalSize()});
        }
        catch (EntryUnavailableException e) {
            logger.error("{}: regular compact log entries failed, error={}", (Object)this.name, (Object)e.getMessage());
        }
    }

    public Object getLogUpdateCondition(long logIndex) {
        return this.logUpdateConditions[(int)(logIndex % (long)this.logUpdateConditions.length)];
    }

    void applyAllCommittedLogWhenStartUp() {
        long lo = this.maxHaveAppliedCommitIndex;
        long hi = this.getCommittedEntryManager().getLastIndex() + 1L;
        if (lo >= hi) {
            logger.info("{}: the maxHaveAppliedCommitIndex={}, lastIndex={}, no need to reapply", new Object[]{this.name, this.maxHaveAppliedCommitIndex, hi});
            return;
        }
        ArrayList<Log> entries = new ArrayList<Log>(this.getCommittedEntryManager().getEntries(lo, hi));
        this.applyEntries(entries);
    }

    public void checkAppliedLogIndex() {
        while (!Thread.currentThread().isInterrupted()) {
            try {
                this.doCheckAppliedLogIndex();
            }
            catch (Exception e) {
                logger.error("{}, an exception occurred when checking the applied log index", (Object)this.name, (Object)e);
            }
        }
        logger.info("{}, the check-log-applier thread {} is interrupted", (Object)this.name, (Object)Thread.currentThread().getName());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void doCheckAppliedLogIndex() {
        long nextToCheckIndex = this.maxHaveAppliedCommitIndex + 1L;
        try {
            if (nextToCheckIndex > this.commitIndex || nextToCheckIndex > this.getCommittedEntryManager().getLastIndex() || this.blockAppliedCommitIndex > 0L && this.blockAppliedCommitIndex < nextToCheckIndex) {
                Thread.sleep(5L);
                return;
            }
            Log log = this.getCommittedEntryManager().getEntry(nextToCheckIndex);
            if (log == null || log.getCurrLogIndex() != nextToCheckIndex) {
                logger.warn("{}, get log error when checking the applied log index, log={}, nextToCheckIndex={}", new Object[]{this.name, log, nextToCheckIndex});
                return;
            }
            Object object = log;
            synchronized (object) {
                while (!log.isApplied() && this.maxHaveAppliedCommitIndex < log.getCurrLogIndex()) {
                    log.wait(5L);
                }
            }
            object = this.changeApplyCommitIndexCond;
            synchronized (object) {
                this.maxHaveAppliedCommitIndex = Math.max(this.maxHaveAppliedCommitIndex, nextToCheckIndex);
            }
            logger.debug("{}: log={} is applied, nextToCheckIndex={}, commitIndex={}, maxHaveAppliedCommitIndex={}", new Object[]{this.name, log, nextToCheckIndex, this.commitIndex, this.maxHaveAppliedCommitIndex});
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            logger.info("{}: do check applied log index is interrupt", (Object)this.name);
        }
        catch (EntryCompactedException e) {
            Object object = this.changeApplyCommitIndexCond;
            synchronized (object) {
                this.maxHaveAppliedCommitIndex = Math.max(this.maxHaveAppliedCommitIndex, nextToCheckIndex);
            }
            logger.debug("{}: compacted log is assumed applied, nextToCheckIndex={}, commitIndex={}, maxHaveAppliedCommitIndex={}", new Object[]{this.name, nextToCheckIndex, this.commitIndex, this.maxHaveAppliedCommitIndex});
        }
    }

    public void resetBlockAppliedCommitIndex() {
        this.blockAppliedCommitIndex = -1L;
        this.reapplyBlockedLogs();
    }

    public void setBlockAppliedCommitIndex(long blockAppliedCommitIndex) {
        this.blockAppliedCommitIndex = blockAppliedCommitIndex;
    }

    private void reapplyBlockedLogs() {
        if (!this.blockedUnappliedLogList.isEmpty()) {
            this.applyEntries(this.blockedUnappliedLogList);
            logger.info("{}: reapply {} number of logs", (Object)this.name, (Object)this.blockedUnappliedLogList.size());
        }
        this.blockedUnappliedLogList.clear();
    }

    public String getName() {
        return this.name;
    }

    public long getBlockAppliedCommitIndex() {
        return this.blockAppliedCommitIndex;
    }
}

