/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iotdb.db.sync.receiver.transfer;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.IOException;
import java.math.BigInteger;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import org.apache.iotdb.db.concurrent.ThreadName;
import org.apache.iotdb.db.conf.IoTDBConfig;
import org.apache.iotdb.db.conf.IoTDBConstant;
import org.apache.iotdb.db.conf.IoTDBDescriptor;
import org.apache.iotdb.db.conf.directories.DirectoryManager;
import org.apache.iotdb.db.engine.storagegroup.TsFileResource;
import org.apache.iotdb.db.exception.DiskSpaceInsufficientException;
import org.apache.iotdb.db.exception.SyncDeviceOwnerConflictException;
import org.apache.iotdb.db.exception.metadata.MetadataException;
import org.apache.iotdb.db.metadata.MManager;
import org.apache.iotdb.db.sync.receiver.load.FileLoader;
import org.apache.iotdb.db.sync.receiver.load.FileLoaderManager;
import org.apache.iotdb.db.sync.receiver.load.IFileLoader;
import org.apache.iotdb.db.sync.receiver.recover.SyncReceiverLogAnalyzer;
import org.apache.iotdb.db.sync.receiver.recover.SyncReceiverLogger;
import org.apache.iotdb.db.utils.FilePathUtils;
import org.apache.iotdb.db.utils.SyncUtils;
import org.apache.iotdb.service.sync.thrift.ConfirmInfo;
import org.apache.iotdb.service.sync.thrift.SyncService;
import org.apache.iotdb.service.sync.thrift.SyncStatus;
import org.apache.thrift.TException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SyncServiceImpl
implements SyncService.Iface {
    private static final Logger logger = LoggerFactory.getLogger(SyncServiceImpl.class);
    private IoTDBConfig config = IoTDBDescriptor.getInstance().getConfig();
    private ThreadLocal<String> syncFolderPath = new ThreadLocal();
    private ThreadLocal<String> currentSG = new ThreadLocal();
    private ThreadLocal<SyncReceiverLogger> syncLog = new ThreadLocal();
    private ThreadLocal<String> senderName = new ThreadLocal();
    private ThreadLocal<File> currentFile = new ThreadLocal();
    private ThreadLocal<FileChannel> currentFileWriter = new ThreadLocal();
    private ThreadLocal<MessageDigest> messageDigest = new ThreadLocal();

    public SyncStatus check(ConfirmInfo info) {
        String ipAddress = info.address;
        String uuid = info.uuid;
        Thread.currentThread().setName(ThreadName.SYNC_SERVER.getName());
        if (!info.version.equals(IoTDBConstant.VERSION)) {
            return this.getErrorResult(String.format("Version mismatch: the sender <%s>, the receiver <%s>", info.version, IoTDBConstant.VERSION));
        }
        if (info.partitionInterval != IoTDBDescriptor.getInstance().getConfig().getPartitionInterval()) {
            return this.getErrorResult(String.format("Partition interval mismatch: the sender <%d>, the receiver <%d>", info.partitionInterval, IoTDBDescriptor.getInstance().getConfig().getPartitionInterval()));
        }
        if (SyncUtils.verifyIPSegment(this.config.getIpWhiteList(), ipAddress)) {
            this.senderName.set(ipAddress + "_" + uuid);
            if (this.checkRecovery()) {
                logger.info("Start to sync with sender {}", (Object)this.senderName.get());
                return this.getSuccessResult();
            }
            return this.getErrorResult("Receiver is processing data from previous sync tasks");
        }
        return this.getErrorResult("Sender IP is not in the white list of receiver IP and synchronization tasks are not allowed.");
    }

    private boolean checkRecovery() {
        try {
            if (this.currentFileWriter.get() != null && this.currentFileWriter.get().isOpen()) {
                this.currentFileWriter.get().close();
            }
            if (this.syncLog.get() != null) {
                this.syncLog.get().close();
            }
            return SyncReceiverLogAnalyzer.getInstance().recover(this.senderName.get());
        }
        catch (IOException e) {
            logger.error("Check recovery state fail", (Throwable)e);
            return false;
        }
    }

    public SyncStatus startSync() {
        try {
            this.initPath();
            this.currentSG.remove();
            FileLoader.createFileLoader(this.senderName.get(), this.syncFolderPath.get());
            this.syncLog.set(new SyncReceiverLogger(new File(this.syncFolderPath.get(), "sync.log")));
            return this.getSuccessResult();
        }
        catch (IOException | DiskSpaceInsufficientException e) {
            logger.error("Can not receiver data from sender", (Throwable)e);
            return this.getErrorResult(e.getMessage());
        }
    }

    private void initPath() throws DiskSpaceInsufficientException {
        String dataDir = new File(DirectoryManager.getInstance().getNextFolderForSequenceFile()).getParentFile().getAbsolutePath();
        this.syncFolderPath.set(FilePathUtils.regularizePath(dataDir) + "sync-receiver" + File.separatorChar + this.senderName.get());
    }

    public SyncStatus init(String storageGroup) {
        logger.info("Sync process started to receive data of storage group {}", (Object)storageGroup);
        this.currentSG.set(storageGroup);
        try {
            this.syncLog.get().startSyncDeletedFilesName();
        }
        catch (IOException e) {
            logger.error("Can not init sync process", (Throwable)e);
            return this.getErrorResult(e.getMessage());
        }
        return this.getSuccessResult();
    }

    public SyncStatus syncDeletedFileName(String fileName) throws TException {
        try {
            this.syncLog.get().finishSyncDeletedFileName(new File(this.getSyncDataPath(), this.currentSG.get() + File.separatorChar + fileName));
            FileLoaderManager.getInstance().getFileLoader(this.senderName.get()).addDeletedFileName(new File(this.getSyncDataPath(), this.currentSG.get() + File.separatorChar + fileName));
        }
        catch (IOException e) {
            logger.error("Can not sync deleted file", (Throwable)e);
            return this.getErrorResult(String.format("Can not sync deleted file %s because %s", fileName, e.getMessage()));
        }
        return this.getSuccessResult();
    }

    public SyncStatus initSyncData(String filename) throws TException {
        try {
            File file = this.currentSG.get() == null ? new File(this.getSyncDataPath(), filename) : new File(this.getSyncDataPath(), this.currentSG.get() + File.separatorChar + filename);
            file.delete();
            this.currentFile.set(file);
            if (!file.getParentFile().exists()) {
                file.getParentFile().mkdirs();
            }
            if (this.currentFileWriter.get() != null && this.currentFileWriter.get().isOpen()) {
                this.currentFileWriter.get().close();
            }
            this.currentFileWriter.set(new FileOutputStream(file).getChannel());
            this.syncLog.get().startSyncTsFiles();
            this.messageDigest.set(MessageDigest.getInstance("MD5"));
        }
        catch (IOException | NoSuchAlgorithmException e) {
            logger.error("Can not init sync resource for file {}", (Object)filename, (Object)e);
            return this.getErrorResult(String.format("Can not init sync resource for file %s because %s", filename, e.getMessage()));
        }
        return this.getSuccessResult();
    }

    public SyncStatus syncData(ByteBuffer buff) {
        try {
            this.currentFileWriter.get().write(buff);
            buff.flip();
            this.messageDigest.get().update(buff);
        }
        catch (IOException e) {
            logger.error("Can not sync data for file {}", (Object)this.currentFile.get().getAbsoluteFile(), (Object)e);
            return this.getErrorResult(String.format("Can not sync data for file %s because %s", this.currentFile.get().getName(), e.getMessage()));
        }
        return this.getSuccessResult();
    }

    public SyncStatus checkDataMD5(String md5OfSender) throws TException {
        String md5OfReceiver = new BigInteger(1, this.messageDigest.get().digest()).toString(16);
        try {
            if (this.currentFileWriter.get() != null && this.currentFileWriter.get().isOpen()) {
                this.currentFileWriter.get().close();
            }
            if (!md5OfSender.equals(md5OfReceiver)) {
                this.currentFile.get().delete();
                this.currentFileWriter.set(new FileOutputStream(this.currentFile.get()).getChannel());
                return this.getErrorResult(String.format("MD5 of the sender is differ from MD5 of the receiver of the file %s.", this.currentFile.get().getAbsolutePath()));
            }
            if (this.currentFile.get().getName().endsWith("mlog.txt")) {
                this.loadMetadata();
            } else if (!this.currentFile.get().getName().endsWith(".resource")) {
                logger.info("Receiver has received {} successfully.", (Object)this.currentFile.get());
                FileLoaderManager.getInstance().checkAndUpdateDeviceOwner(new TsFileResource(new File(this.currentFile.get() + ".resource")));
                this.syncLog.get().finishSyncTsfile(this.currentFile.get());
                FileLoaderManager.getInstance().getFileLoader(this.senderName.get()).addTsfile(this.currentFile.get());
            }
        }
        catch (IOException e) {
            logger.error("Can not check data MD5 for file {}", (Object)this.currentFile.get().getAbsoluteFile(), (Object)e);
            return this.getErrorResult(String.format("Can not check data MD5 for file %s because %s", this.currentFile.get().getName(), e.getMessage()));
        }
        catch (SyncDeviceOwnerConflictException e) {
            logger.error("Device owner has conflicts, skip all other tsfiles in the sg {}.", (Object)this.currentSG.get());
            return new SyncStatus(-2, String.format("Device owner has conflicts, skip all other tsfiles in the same sg %s because %s", this.currentSG.get(), e.getMessage()));
        }
        return new SyncStatus(1, md5OfReceiver);
    }

    private void loadMetadata() {
        logger.info("Start to load metadata in sync process.");
        if (this.currentFile.get().exists()) {
            try (BufferedReader br = new BufferedReader(new FileReader(this.currentFile.get()));){
                String metadataOperation;
                while ((metadataOperation = br.readLine()) != null) {
                    try {
                        MManager.getInstance().operation(metadataOperation);
                    }
                    catch (IOException | MetadataException e) {
                        logger.error("Can not operate metadata operation {} ", (Object)metadataOperation, (Object)e);
                    }
                }
            }
            catch (IOException e) {
                logger.error("Cannot read the file {}.", (Object)this.currentFile.get().getAbsoluteFile(), (Object)e);
            }
        }
    }

    public SyncStatus endSync() throws TException {
        try {
            IFileLoader loader;
            if (this.syncLog.get() != null) {
                this.syncLog.get().close();
            }
            if ((loader = FileLoaderManager.getInstance().getFileLoader(this.senderName.get())) == null) {
                return this.getErrorResult(String.format("File Loader of the storage group %s is null", this.currentSG.get()));
            }
            loader.endSync();
            logger.info("Sync process with sender {} finished.", (Object)this.senderName.get());
        }
        catch (IOException e) {
            logger.error("Can not end sync", (Throwable)e);
            return this.getErrorResult(String.format("Can not end sync because %s", e.getMessage()));
        }
        return this.getSuccessResult();
    }

    private String getSyncDataPath() {
        return this.syncFolderPath.get() + File.separatorChar + "data";
    }

    private SyncStatus getSuccessResult() {
        return new SyncStatus(1, "");
    }

    private SyncStatus getErrorResult(String errorMsg) {
        return new SyncStatus(-1, errorMsg);
    }

    public void handleClientExit() {
    }
}

