/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iotdb.db.engine;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.ConcurrentModificationException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import org.apache.commons.io.FileUtils;
import org.apache.iotdb.db.concurrent.IoTDBThreadPoolFactory;
import org.apache.iotdb.db.conf.IoTDBConfig;
import org.apache.iotdb.db.conf.IoTDBDescriptor;
import org.apache.iotdb.db.engine.fileSystem.SystemFileFactory;
import org.apache.iotdb.db.engine.flush.TsFileFlushPolicy;
import org.apache.iotdb.db.engine.querycontext.QueryDataSource;
import org.apache.iotdb.db.engine.storagegroup.StorageGroupProcessor;
import org.apache.iotdb.db.engine.storagegroup.TsFileProcessor;
import org.apache.iotdb.db.engine.storagegroup.TsFileResource;
import org.apache.iotdb.db.exception.LoadFileException;
import org.apache.iotdb.db.exception.StorageEngineException;
import org.apache.iotdb.db.exception.StorageGroupProcessorException;
import org.apache.iotdb.db.exception.WriteProcessException;
import org.apache.iotdb.db.exception.metadata.MetadataException;
import org.apache.iotdb.db.exception.metadata.StorageGroupNotSetException;
import org.apache.iotdb.db.exception.query.QueryProcessException;
import org.apache.iotdb.db.exception.runtime.StorageEngineFailureException;
import org.apache.iotdb.db.metadata.MManager;
import org.apache.iotdb.db.metadata.mnode.StorageGroupMNode;
import org.apache.iotdb.db.qp.physical.crud.InsertPlan;
import org.apache.iotdb.db.qp.physical.crud.InsertTabletPlan;
import org.apache.iotdb.db.query.context.QueryContext;
import org.apache.iotdb.db.query.control.QueryFileManager;
import org.apache.iotdb.db.service.IService;
import org.apache.iotdb.db.service.ServiceType;
import org.apache.iotdb.db.utils.FilePathUtils;
import org.apache.iotdb.db.utils.UpgradeUtils;
import org.apache.iotdb.service.rpc.thrift.TSStatus;
import org.apache.iotdb.tsfile.file.metadata.enums.TSDataType;
import org.apache.iotdb.tsfile.read.expression.impl.SingleSeriesExpression;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class StorageEngine
implements IService {
    private final Logger logger;
    private static final IoTDBConfig config = IoTDBDescriptor.getInstance().getConfig();
    private static final long TTL_CHECK_INTERVAL = 60000L;
    private final String systemDir;
    private final ConcurrentHashMap<String, StorageGroupProcessor> processorMap = new ConcurrentHashMap();
    private static final ExecutorService recoveryThreadPool = IoTDBThreadPoolFactory.newFixedThreadPool(Runtime.getRuntime().availableProcessors(), "Recovery-Thread-Pool");
    private ScheduledExecutorService ttlCheckThread;
    private TsFileFlushPolicy fileFlushPolicy = new TsFileFlushPolicy.DirectFlushPolicy();
    private static long timePartitionInterval;
    private static boolean enablePartition;

    public static StorageEngine getInstance() {
        return InstanceHolder.INSTANCE;
    }

    private StorageEngine() {
        this.logger = LoggerFactory.getLogger(StorageEngine.class);
        this.systemDir = FilePathUtils.regularizePath(config.getSystemDir()) + "storage_groups";
        if (!enablePartition) {
            timePartitionInterval = Long.MAX_VALUE;
        } else {
            String timePrecision;
            switch (timePrecision = IoTDBDescriptor.getInstance().getConfig().getTimestampPrecision()) {
                case "ns": {
                    timePartitionInterval = IoTDBDescriptor.getInstance().getConfig().getPartitionInterval() * 1000000000L;
                    break;
                }
                case "us": {
                    timePartitionInterval = IoTDBDescriptor.getInstance().getConfig().getPartitionInterval() * 1000000L;
                    break;
                }
                default: {
                    timePartitionInterval = IoTDBDescriptor.getInstance().getConfig().getPartitionInterval() * 1000L;
                }
            }
        }
        try {
            FileUtils.forceMkdir((File)SystemFileFactory.INSTANCE.getFile(this.systemDir));
        }
        catch (IOException e) {
            throw new StorageEngineFailureException(e);
        }
        UpgradeUtils.recoverUpgrade();
        List<StorageGroupMNode> sgNodes = MManager.getInstance().getAllStorageGroupNodes();
        ArrayList<Future<Void>> futures = new ArrayList<Future<Void>>();
        for (StorageGroupMNode storageGroupMNode : sgNodes) {
            futures.add(recoveryThreadPool.submit(() -> {
                try {
                    StorageGroupProcessor processor = new StorageGroupProcessor(this.systemDir, storageGroup.getFullPath(), this.fileFlushPolicy);
                    processor.setDataTTL(storageGroup.getDataTTL());
                    this.processorMap.put(storageGroup.getFullPath(), processor);
                    this.logger.info("Storage Group Processor {} is recovered successfully", (Object)storageGroup.getFullPath());
                }
                catch (Exception e) {
                    this.logger.error("meet error when recovering storage group: {}", (Object)storageGroup, (Object)e);
                }
                return null;
            }));
        }
        for (Future future : futures) {
            try {
                future.get();
            }
            catch (InterruptedException | ExecutionException e) {
                throw new StorageEngineFailureException("StorageEngine failed to recover.", e);
            }
        }
        recoveryThreadPool.shutdown();
    }

    @Override
    public void start() {
        this.ttlCheckThread = Executors.newSingleThreadScheduledExecutor();
        this.ttlCheckThread.scheduleAtFixedRate(this::checkTTL, 60000L, 60000L, TimeUnit.MILLISECONDS);
    }

    private void checkTTL() {
        try {
            for (StorageGroupProcessor processor : this.processorMap.values()) {
                processor.checkFilesTTL();
            }
        }
        catch (ConcurrentModificationException concurrentModificationException) {
        }
        catch (Exception e) {
            this.logger.error("An error occurred when checking TTL", (Throwable)e);
        }
    }

    @Override
    public void stop() {
        this.syncCloseAllProcessor();
        if (this.ttlCheckThread != null) {
            this.ttlCheckThread.shutdownNow();
            try {
                this.ttlCheckThread.awaitTermination(30L, TimeUnit.SECONDS);
            }
            catch (InterruptedException e) {
                this.logger.warn("TTL check thread still doesn't exit after 30s");
            }
        }
        recoveryThreadPool.shutdownNow();
        this.reset();
    }

    @Override
    public ServiceType getID() {
        return ServiceType.STORAGE_ENGINE_SERVICE;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public StorageGroupProcessor getProcessor(String path) throws StorageEngineException {
        try {
            String storageGroupName = MManager.getInstance().getStorageGroupName(path);
            StorageGroupProcessor processor = this.processorMap.get(storageGroupName);
            if (processor == null) {
                String string = storageGroupName = storageGroupName.intern();
                synchronized (string) {
                    processor = this.processorMap.get(storageGroupName);
                    if (processor == null) {
                        this.logger.info("construct a processor instance, the storage group is {}, Thread is {}", (Object)storageGroupName, (Object)Thread.currentThread().getId());
                        processor = new StorageGroupProcessor(this.systemDir, storageGroupName, this.fileFlushPolicy);
                        StorageGroupMNode storageGroup = MManager.getInstance().getStorageGroupNode(storageGroupName);
                        processor.setDataTTL(storageGroup.getDataTTL());
                        this.processorMap.put(storageGroupName, processor);
                    }
                }
            }
            return processor;
        }
        catch (StorageGroupProcessorException | MetadataException e) {
            throw new StorageEngineException(e);
        }
    }

    public synchronized void reset() {
        this.processorMap.clear();
    }

    public void insert(InsertPlan insertPlan) throws StorageEngineException {
        StorageGroupProcessor storageGroupProcessor = this.getProcessor(insertPlan.getDeviceId());
        try {
            storageGroupProcessor.insert(insertPlan);
        }
        catch (WriteProcessException e) {
            throw new StorageEngineException(e);
        }
    }

    public TSStatus[] insertTablet(InsertTabletPlan insertTabletPlan) throws StorageEngineException {
        StorageGroupProcessor storageGroupProcessor;
        try {
            storageGroupProcessor = this.getProcessor(insertTabletPlan.getDeviceId());
        }
        catch (StorageEngineException e) {
            this.logger.warn("get StorageGroupProcessor of device {} failed, because {}", new Object[]{insertTabletPlan.getDeviceId(), e.getMessage(), e});
            throw new StorageEngineException(e);
        }
        try {
            return storageGroupProcessor.insertTablet(insertTabletPlan);
        }
        catch (WriteProcessException e) {
            throw new StorageEngineException(e);
        }
    }

    public void syncCloseAllProcessor() {
        this.logger.info("Start closing all storage group processor");
        for (StorageGroupProcessor processor : this.processorMap.values()) {
            processor.syncCloseAllWorkingTsFileProcessors();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void asyncCloseProcessor(String storageGroupName, boolean isSeq) throws StorageGroupNotSetException {
        StorageGroupProcessor processor = this.processorMap.get(storageGroupName);
        if (processor != null) {
            this.logger.info("async closing sg processor is called for closing {}, seq = {}", (Object)storageGroupName, (Object)isSeq);
            processor.writeLock();
            try {
                if (isSeq) {
                    for (TsFileProcessor tsfileProcessor : new ArrayList<TsFileProcessor>(processor.getWorkSequenceTsFileProcessors())) {
                        processor.asyncCloseOneTsFileProcessor(true, tsfileProcessor);
                    }
                }
                for (TsFileProcessor tsfileProcessor : new ArrayList<TsFileProcessor>(processor.getWorkUnsequenceTsFileProcessor())) {
                    processor.asyncCloseOneTsFileProcessor(false, tsfileProcessor);
                }
            }
            finally {
                processor.writeUnlock();
            }
        } else {
            throw new StorageGroupNotSetException(storageGroupName);
        }
    }

    public void update(String deviceId, String measurementId, long startTime, long endTime, TSDataType type, String v) {
    }

    public void delete(String deviceId, String measurementId, long timestamp) throws StorageEngineException {
        StorageGroupProcessor storageGroupProcessor = this.getProcessor(deviceId);
        try {
            storageGroupProcessor.delete(deviceId, measurementId, timestamp);
        }
        catch (IOException e) {
            throw new StorageEngineException(e.getMessage());
        }
    }

    public QueryDataSource query(SingleSeriesExpression seriesExpression, QueryContext context, QueryFileManager filePathsManager) throws StorageEngineException, QueryProcessException {
        String deviceId = seriesExpression.getSeriesPath().getDevice();
        String measurementId = seriesExpression.getSeriesPath().getMeasurement();
        StorageGroupProcessor storageGroupProcessor = this.getProcessor(deviceId);
        return storageGroupProcessor.query(deviceId, measurementId, context, filePathsManager, seriesExpression.getFilter());
    }

    public int countUpgradeFiles() {
        int totalUpgradeFileNum = 0;
        for (StorageGroupProcessor storageGroupProcessor : this.processorMap.values()) {
            totalUpgradeFileNum += storageGroupProcessor.countUpgradeFiles();
        }
        return totalUpgradeFileNum;
    }

    public void upgradeAll() throws StorageEngineException {
        if (IoTDBDescriptor.getInstance().getConfig().isReadOnly()) {
            throw new StorageEngineException("Current system mode is read only, does not support file upgrade");
        }
        for (StorageGroupProcessor storageGroupProcessor : this.processorMap.values()) {
            storageGroupProcessor.upgrade();
        }
    }

    public void mergeAll(boolean fullMerge) throws StorageEngineException {
        if (IoTDBDescriptor.getInstance().getConfig().isReadOnly()) {
            throw new StorageEngineException("Current system mode is read only, does not support merge");
        }
        for (StorageGroupProcessor storageGroupProcessor : this.processorMap.values()) {
            storageGroupProcessor.merge(fullMerge);
        }
    }

    public void deleteAllDataFilesInOneStorageGroup(String storageGroupName) {
        if (this.processorMap.containsKey(storageGroupName)) {
            this.syncDeleteDataFiles(storageGroupName);
        }
    }

    private void syncDeleteDataFiles(String storageGroupName) {
        this.logger.info("Force to delete the data in storage group processor {}", (Object)storageGroupName);
        StorageGroupProcessor processor = this.processorMap.get(storageGroupName);
        processor.syncDeleteDataFiles();
    }

    public synchronized boolean deleteAll() {
        this.logger.info("Start deleting all storage groups' timeseries");
        this.syncCloseAllProcessor();
        for (String storageGroup : MManager.getInstance().getAllStorageGroupNames()) {
            this.deleteAllDataFilesInOneStorageGroup(storageGroup);
        }
        return true;
    }

    public void setTTL(String storageGroup, long dataTTL) throws StorageEngineException {
        StorageGroupProcessor storageGroupProcessor = this.getProcessor(storageGroup);
        storageGroupProcessor.setDataTTL(dataTTL);
    }

    public void deleteStorageGroup(String storageGroupName) {
        this.deleteAllDataFilesInOneStorageGroup(storageGroupName);
        StorageGroupProcessor processor = this.processorMap.remove(storageGroupName);
        if (processor != null) {
            processor.deleteFolder(this.systemDir);
        }
    }

    public void loadNewTsFileForSync(TsFileResource newTsFileResource) throws StorageEngineException, LoadFileException {
        this.getProcessor(newTsFileResource.getFile().getParentFile().getName()).loadNewTsFileForSync(newTsFileResource);
    }

    public void loadNewTsFile(TsFileResource newTsFileResource) throws LoadFileException, StorageEngineException, MetadataException {
        Map<String, Integer> deviceMap = newTsFileResource.getDeviceToIndexMap();
        if (deviceMap == null || deviceMap.isEmpty()) {
            throw new StorageEngineException("Can not get the corresponding storage group.");
        }
        String device = deviceMap.keySet().iterator().next();
        String storageGroupName = MManager.getInstance().getStorageGroupName(device);
        this.getProcessor(storageGroupName).loadNewTsFile(newTsFileResource);
    }

    public boolean deleteTsfileForSync(File deletedTsfile) throws StorageEngineException {
        return this.getProcessor(deletedTsfile.getParentFile().getName()).deleteTsfile(deletedTsfile);
    }

    public boolean deleteTsfile(File deletedTsfile) throws StorageEngineException {
        return this.getProcessor(this.getSgByEngineFile(deletedTsfile)).deleteTsfile(deletedTsfile);
    }

    public boolean moveTsfile(File tsfileToBeMoved, File targetDir) throws StorageEngineException, IOException {
        return this.getProcessor(this.getSgByEngineFile(tsfileToBeMoved)).moveTsfile(tsfileToBeMoved, targetDir);
    }

    private String getSgByEngineFile(File file) {
        return file.getParentFile().getParentFile().getName();
    }

    public Map<String, Map<Long, List<TsFileResource>>> getAllClosedStorageGroupTsFile() {
        HashMap<String, Map<Long, List<TsFileResource>>> ret = new HashMap<String, Map<Long, List<TsFileResource>>>();
        for (Map.Entry<String, StorageGroupProcessor> entry : this.processorMap.entrySet()) {
            List<TsFileResource> allResources = entry.getValue().getSequenceFileTreeSet();
            allResources.addAll(entry.getValue().getUnSequenceFileList());
            for (TsFileResource sequenceFile : allResources) {
                if (!sequenceFile.isClosed()) continue;
                long partitionNum = sequenceFile.getTimePartition();
                Map storageGroupFiles = ret.computeIfAbsent(entry.getKey(), n -> new HashMap());
                storageGroupFiles.computeIfAbsent(partitionNum, n -> new ArrayList()).add(sequenceFile);
            }
        }
        return ret;
    }

    public void setFileFlushPolicy(TsFileFlushPolicy fileFlushPolicy) {
        this.fileFlushPolicy = fileFlushPolicy;
    }

    public boolean isFileAlreadyExist(TsFileResource tsFileResource, String storageGroup, long partitionNum) {
        StorageGroupProcessor processor = this.processorMap.get(storageGroup);
        return processor != null && processor.isFileAlreadyExist(tsFileResource, partitionNum);
    }

    public static long getTimePartitionInterval() {
        return timePartitionInterval;
    }

    public static long getTimePartition(long time) {
        return enablePartition ? time / timePartitionInterval : 0L;
    }

    static {
        enablePartition = IoTDBDescriptor.getInstance().getConfig().isEnablePartition();
    }

    static class InstanceHolder {
        private static final StorageEngine INSTANCE = new StorageEngine();

        private InstanceHolder() {
        }
    }
}

