/*
 * Decompiled with CFR 0.152.
 */
package io.moquette.broker;

import io.moquette.broker.Authorizator;
import io.moquette.broker.ClientDescriptor;
import io.moquette.broker.IQueueRepository;
import io.moquette.broker.MQTTConnection;
import io.moquette.broker.Session;
import io.moquette.broker.SessionCorruptedException;
import io.moquette.broker.subscriptions.ISubscriptionsDirectory;
import io.moquette.broker.subscriptions.Subscription;
import io.moquette.broker.subscriptions.Topic;
import io.netty.buffer.ByteBuf;
import io.netty.buffer.Unpooled;
import io.netty.handler.codec.mqtt.MqttConnectMessage;
import io.netty.handler.codec.mqtt.MqttQoS;
import java.net.InetSocketAddress;
import java.util.Collection;
import java.util.Optional;
import java.util.Queue;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SessionRegistry {
    private static final Logger LOG = LoggerFactory.getLogger(SessionRegistry.class);
    private final ConcurrentMap<String, Session> pool = new ConcurrentHashMap<String, Session>();
    private final ISubscriptionsDirectory subscriptionsDirectory;
    private final IQueueRepository queueRepository;
    private final Authorizator authorizator;
    private final ConcurrentMap<String, Queue<EnqueuedMessage>> queues = new ConcurrentHashMap<String, Queue<EnqueuedMessage>>();

    SessionRegistry(ISubscriptionsDirectory subscriptionsDirectory, IQueueRepository queueRepository, Authorizator authorizator) {
        this.subscriptionsDirectory = subscriptionsDirectory;
        this.queueRepository = queueRepository;
        this.authorizator = authorizator;
    }

    void bindToSession(MQTTConnection mqttConnection, MqttConnectMessage msg, String clientId) {
        Session newSession;
        boolean isSessionAlreadyStored = false;
        PostConnectAction postConnectAction = PostConnectAction.NONE;
        if (!this.pool.containsKey(clientId)) {
            boolean success;
            newSession = this.createNewSession(mqttConnection, msg, clientId);
            Session previous = this.pool.putIfAbsent(clientId, newSession);
            boolean bl = success = previous == null;
            if (success) {
                LOG.trace("case 1, not existing session with CId {}", (Object)clientId);
            } else {
                postConnectAction = this.bindToExistingSession(mqttConnection, msg, clientId, newSession);
                isSessionAlreadyStored = true;
            }
        } else {
            newSession = this.createNewSession(mqttConnection, msg, clientId);
            postConnectAction = this.bindToExistingSession(mqttConnection, msg, clientId, newSession);
            isSessionAlreadyStored = true;
        }
        boolean msgCleanSessionFlag = msg.variableHeader().isCleanSession();
        boolean isSessionAlreadyPresent = !msgCleanSessionFlag && isSessionAlreadyStored;
        mqttConnection.sendConnAck(isSessionAlreadyPresent);
        if (postConnectAction == PostConnectAction.SEND_STORED_MESSAGES) {
            Session session = (Session)this.pool.get(clientId);
            session.sendQueuedMessagesWhileOffline();
        }
    }

    private PostConnectAction bindToExistingSession(MQTTConnection mqttConnection, MqttConnectMessage msg, String clientId, Session newSession) {
        PostConnectAction postConnectAction = PostConnectAction.NONE;
        boolean newIsClean = msg.variableHeader().isCleanSession();
        Session oldSession = (Session)this.pool.get(clientId);
        if (newIsClean && oldSession.disconnected()) {
            this.dropQueuesForClient(clientId);
            this.unsubscribe(oldSession);
            boolean result = oldSession.assignState(Session.SessionStatus.DISCONNECTED, Session.SessionStatus.CONNECTING);
            if (!result) {
                throw new SessionCorruptedException("old session was already changed state");
            }
            this.copySessionConfig(msg, oldSession);
            oldSession.bind(mqttConnection);
            result = oldSession.assignState(Session.SessionStatus.CONNECTING, Session.SessionStatus.CONNECTED);
            if (!result) {
                throw new SessionCorruptedException("old session moved in connected state by other thread");
            }
            boolean published = this.pool.replace(clientId, oldSession, oldSession);
            if (!published) {
                throw new SessionCorruptedException("old session was already removed");
            }
            LOG.trace("case 2, oldSession with same CId {} disconnected", (Object)clientId);
        } else if (!newIsClean && oldSession.disconnected()) {
            String username = mqttConnection.getUsername();
            this.reactivateSubscriptions(oldSession, username);
            boolean connecting = oldSession.assignState(Session.SessionStatus.DISCONNECTED, Session.SessionStatus.CONNECTING);
            if (!connecting) {
                throw new SessionCorruptedException("old session moved in connected state by other thread");
            }
            oldSession.bind(mqttConnection);
            boolean connected = oldSession.assignState(Session.SessionStatus.CONNECTING, Session.SessionStatus.CONNECTED);
            if (!connected) {
                throw new SessionCorruptedException("old session moved in other state state by other thread");
            }
            boolean published = this.pool.replace(clientId, oldSession, oldSession);
            if (!published) {
                throw new SessionCorruptedException("old session was already removed");
            }
            postConnectAction = PostConnectAction.SEND_STORED_MESSAGES;
            LOG.trace("case 3, oldSession with same CId {} disconnected", (Object)clientId);
        } else if (oldSession.connected()) {
            LOG.trace("case 4, oldSession with same CId {} still connected, force to close", (Object)clientId);
            oldSession.closeImmediately();
            boolean published = this.pool.replace(clientId, oldSession, newSession);
            if (!published) {
                throw new SessionCorruptedException("old session was already removed");
            }
        }
        return postConnectAction;
    }

    private void reactivateSubscriptions(Session session, String username) {
        for (Subscription existingSub : session.getSubscriptions()) {
            boolean topicReadable = this.authorizator.canRead(existingSub.getTopicFilter(), username, session.getClientID());
            if (topicReadable) continue;
            this.subscriptionsDirectory.removeSubscription(existingSub.getTopicFilter(), session.getClientID());
        }
    }

    private void unsubscribe(Session session) {
        for (Subscription existingSub : session.getSubscriptions()) {
            this.subscriptionsDirectory.removeSubscription(existingSub.getTopicFilter(), session.getClientID());
        }
    }

    private Session createNewSession(MQTTConnection mqttConnection, MqttConnectMessage msg, String clientId) {
        Session newSession;
        boolean clean = msg.variableHeader().isCleanSession();
        Queue sessionQueue = this.queues.computeIfAbsent(clientId, cli -> this.queueRepository.createQueue((String)cli, clean));
        if (msg.variableHeader().isWillFlag()) {
            Session.Will will = this.createWill(msg);
            newSession = new Session(clientId, clean, will, sessionQueue);
        } else {
            newSession = new Session(clientId, clean, sessionQueue);
        }
        newSession.markConnected();
        newSession.bind(mqttConnection);
        return newSession;
    }

    private void copySessionConfig(MqttConnectMessage msg, Session session) {
        boolean clean = msg.variableHeader().isCleanSession();
        Session.Will will = msg.variableHeader().isWillFlag() ? this.createWill(msg) : null;
        session.update(clean, will);
    }

    private Session.Will createWill(MqttConnectMessage msg) {
        ByteBuf willPayload = Unpooled.copiedBuffer((byte[])msg.payload().willMessageInBytes());
        String willTopic = msg.payload().willTopic();
        boolean retained = msg.variableHeader().isWillRetain();
        MqttQoS qos = MqttQoS.valueOf((int)msg.variableHeader().willQos());
        return new Session.Will(willTopic, willPayload, qos, retained);
    }

    Session retrieve(String clientID) {
        return (Session)this.pool.get(clientID);
    }

    public void remove(String clientID) {
        this.pool.remove(clientID);
    }

    public void disconnect(String clientID) {
        Session session = this.retrieve(clientID);
        if (session == null) {
            LOG.debug("Some other thread already removed the session CId={}", (Object)clientID);
            return;
        }
        session.disconnect();
    }

    private void dropQueuesForClient(String clientId) {
        this.queues.remove(clientId);
    }

    Collection<ClientDescriptor> listConnectedClients() {
        return this.pool.values().stream().filter(Session::connected).map(this::createClientDescriptor).filter(Optional::isPresent).map(Optional::get).collect(Collectors.toList());
    }

    private Optional<ClientDescriptor> createClientDescriptor(Session s) {
        String clientID = s.getClientID();
        Optional<InetSocketAddress> remoteAddressOpt = s.remoteAddress();
        return remoteAddressOpt.map(r -> new ClientDescriptor(clientID, r.getHostString(), r.getPort()));
    }

    private static enum PostConnectAction {
        NONE,
        SEND_STORED_MESSAGES;

    }

    static final class PubRelMarker
    extends EnqueuedMessage {
        PubRelMarker() {
        }
    }

    static class PublishedMessage
    extends EnqueuedMessage {
        final Topic topic;
        final MqttQoS publishingQos;
        final ByteBuf payload;

        PublishedMessage(Topic topic, MqttQoS publishingQos, ByteBuf payload) {
            this.topic = topic;
            this.publishingQos = publishingQos;
            this.payload = payload;
        }
    }

    public static abstract class EnqueuedMessage {
    }
}

