/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.ignite.internal.processors.hadoop.impl.examples;

import java.io.IOException;
import org.apache.hadoop.conf.Configurable;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.io.IntWritable;
import org.apache.hadoop.io.Text;
import org.apache.hadoop.mapreduce.Reducer;
import org.apache.ignite.internal.processors.hadoop.impl.HadoopErrorSimulator;

/**
 * Combiner and Reducer phase of WordCount job.
 */
public class HadoopWordCount2Reducer extends Reducer<Text, IntWritable, Text, IntWritable> implements Configurable {
    /** Writable container for writing sum of word counts. */
    private IntWritable totalWordCnt = new IntWritable();

    /** Flag is to check that mapper was configured before run. */
    private boolean wasConfigured;

    /** Flag is to check that mapper was set up before run. */
    private boolean wasSetUp;

    /** {@inheritDoc} */
    @Override public void reduce(Text key, Iterable<IntWritable> values, Context ctx) throws IOException, InterruptedException {
        assert wasConfigured : "Reducer should be configured";
        assert wasSetUp : "Reducer should be set up";

        int wordCnt = 0;

        for (IntWritable value : values)
            wordCnt += value.get();

        totalWordCnt.set(wordCnt);

        ctx.write(key, totalWordCnt);

        reduceError();
    }

    /**
     * Simulates reduce error if needed.
     */
    protected void reduceError() throws IOException, InterruptedException {
        HadoopErrorSimulator.instance().onReduce();
    }

    /** {@inheritDoc} */
    @Override protected void setup(Context context) throws IOException, InterruptedException {
        super.setup(context);

        wasSetUp = true;

        setupError();
    }

    /**
     * Simulates setup error if needed.
     */
    protected void setupError() throws IOException, InterruptedException {
        HadoopErrorSimulator.instance().onReduceSetup();
    }

    /** {@inheritDoc} */
    @Override protected void cleanup(Context context) throws IOException, InterruptedException {
        super.cleanup(context);

        cleanupError();
    }

    /**
     * Simulates cleanup error if needed.
     */
    protected void cleanupError() throws IOException, InterruptedException {
        HadoopErrorSimulator.instance().onReduceCleanup();
    }

    /** {@inheritDoc} */
    @Override public void setConf(Configuration conf) {
        wasConfigured = true;

        configError();
    }

    /**
     * Simulates configuration error if needed.
     */
    protected void configError() {
        HadoopErrorSimulator.instance().onReduceConfigure();
    }

    /** {@inheritDoc} */
    @Override public Configuration getConf() {
        return null;
    }
}
