/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.ignite.spi.discovery.tcp;

import java.io.BufferedOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.OutputStream;
import java.io.Serializable;
import java.net.InetAddress;
import java.net.Socket;
import java.util.concurrent.atomic.AtomicBoolean;
import org.apache.ignite.IgniteCheckedException;
import org.apache.ignite.internal.util.IgniteUtils;
import org.apache.ignite.internal.util.lang.GridAbsPredicate;
import org.apache.ignite.internal.util.typedef.internal.U;
import org.apache.ignite.marshaller.jdk.JdkMarshaller;
import org.apache.ignite.testframework.GridTestUtils;
import org.apache.ignite.testframework.junits.common.GridCommonAbstractTest;

import static org.apache.ignite.IgniteSystemProperties.IGNITE_MARSHALLER_BLACKLIST;
import static org.apache.ignite.IgniteSystemProperties.IGNITE_MARSHALLER_WHITELIST;

/**
 * Tests for whitelist and blacklist ot avoiding deserialization vulnerability.
 */
public class DiscoveryUnmarshalVulnerabilityTest extends GridCommonAbstractTest {
    /** Marshaller. */
    private static final JdkMarshaller MARSH = new JdkMarshaller();

    /** Shared value. */
    private static final AtomicBoolean SHARED = new AtomicBoolean();

    /** {@inheritDoc} */
    @Override protected void beforeTest() throws Exception {
        super.beforeTest();

        SHARED.set(false);

        System.clearProperty(IGNITE_MARSHALLER_WHITELIST);
        System.clearProperty(IGNITE_MARSHALLER_BLACKLIST);

        IgniteUtils.clearClassCache();
    }

    /**
     * @throws Exception If failed.
     */
    public void testNoLists() throws Exception {
        testExploit(true);
    }

    /**
     * @throws Exception If failed.
     */
    public void testWhiteListIncluded() throws Exception {
        String path = U.resolveIgnitePath("modules/core/src/test/config/class_list_exploit_included.txt").getPath();

        System.setProperty(IGNITE_MARSHALLER_WHITELIST, path);

        testExploit(true);
    }

    /**
     * @throws Exception If failed.
     */
    public void testWhiteListExcluded() throws Exception {
        String path = U.resolveIgnitePath("modules/core/src/test/config/class_list_exploit_excluded.txt").getPath();

        System.setProperty(IGNITE_MARSHALLER_WHITELIST, path);

        testExploit(false);
    }

    /**
     * @throws Exception If failed.
     */
    public void testBlackListIncluded() throws Exception {
        String path = U.resolveIgnitePath("modules/core/src/test/config/class_list_exploit_included.txt").getPath();

        System.setProperty(IGNITE_MARSHALLER_BLACKLIST, path);

        testExploit(false);
    }

    /**
     * @throws Exception If failed.
     */
    public void testBlackListExcluded() throws Exception {
        String path = U.resolveIgnitePath("modules/core/src/test/config/class_list_exploit_excluded.txt").getPath();

        System.setProperty(IGNITE_MARSHALLER_BLACKLIST, path);

        testExploit(true);
    }

    /**
     * @throws Exception If failed.
     */
    public void testBothListIncluded() throws Exception {
        String path = U.resolveIgnitePath("modules/core/src/test/config/class_list_exploit_included.txt").getPath();

        System.setProperty(IGNITE_MARSHALLER_WHITELIST, path);
        System.setProperty(IGNITE_MARSHALLER_BLACKLIST, path);

        testExploit(false);
    }

    /**
     * @param positive Positive.
     */
    private void testExploit(boolean positive) throws Exception {
        try {
            startGrid();

            attack(marshal(new Exploit()));

            boolean res = GridTestUtils.waitForCondition(new GridAbsPredicate() {
                @Override public boolean apply() {
                    return SHARED.get();
                }
            }, 3000L);

            if (positive)
                assertTrue(res);
            else
                assertFalse(res);
        }
        finally {
            stopAllGrids();
        }
    }

    /**
     * @param obj Object.
     */
    private static byte[] marshal(Object obj) throws IgniteCheckedException {
        return MARSH.marshal(obj);
    }

    /**
     * @param data Data.
     */
    private void attack(byte[] data) throws IOException {
        InetAddress addr = InetAddress.getLoopbackAddress();

        try (
            Socket sock = new Socket(addr, 47500);
            OutputStream oos = new BufferedOutputStream(sock.getOutputStream())
        ) {
            oos.write(U.IGNITE_HEADER);
            oos.write(data);
        }
    }

    /** */
    private static class Exploit implements Serializable {
        /**
         * @param is Input stream.
         */
        private void readObject(ObjectInputStream is) throws ClassNotFoundException, IOException {
            SHARED.set(true);
        }
    }
}
