/*
 * Decompiled with CFR 0.152.
 */
package org.apache.ignite.mesos;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.apache.ignite.mesos.ClusterProperties;
import org.apache.ignite.mesos.IgniteTask;
import org.apache.ignite.mesos.resource.ResourceProvider;
import org.apache.mesos.Protos;
import org.apache.mesos.Scheduler;
import org.apache.mesos.SchedulerDriver;

public class IgniteScheduler
implements Scheduler {
    public static final String CPU = "cpus";
    public static final String MEM = "mem";
    public static final String DISK = "disk";
    public static final String DEFAULT_PORT = ":47500..47510";
    public static final String DELIM = ",";
    private static final Logger log = Logger.getLogger(IgniteScheduler.class.getSimpleName());
    private AtomicInteger taskIdGenerator = new AtomicInteger();
    private Map<String, IgniteTask> tasks = new HashMap<String, IgniteTask>();
    private ClusterProperties clusterProps;
    private ResourceProvider resourceProvider;

    public IgniteScheduler(ClusterProperties clusterProps, ResourceProvider resourceProvider) {
        this.clusterProps = clusterProps;
        this.resourceProvider = resourceProvider;
    }

    @Override
    public synchronized void resourceOffers(SchedulerDriver schedulerDriver, List<Protos.Offer> offers) {
        log.log(Level.FINE, "Offers resources: {0}", offers.size());
        for (Protos.Offer offer : offers) {
            IgniteTask igniteTask = this.checkOffer(offer);
            if (igniteTask == null) {
                schedulerDriver.declineOffer(offer.getId());
                continue;
            }
            Protos.TaskID taskId = Protos.TaskID.newBuilder().setValue(Integer.toString(this.taskIdGenerator.incrementAndGet())).build();
            log.log(Level.INFO, "Launching task: {0}", igniteTask);
            Protos.TaskInfo task = this.createTask(offer, igniteTask, taskId);
            try {
                schedulerDriver.launchTasks(Collections.singletonList(offer.getId()), Collections.singletonList(task), Protos.Filters.newBuilder().setRefuseSeconds(1.0).build());
            }
            catch (RuntimeException e) {
                log.log(Level.SEVERE, "Failed launch task. Task id: {0}. Task info: {1}", new Object[]{taskId, task, e});
                throw e;
            }
            this.tasks.put(taskId.getValue(), igniteTask);
        }
    }

    private Protos.TaskInfo createTask(Protos.Offer offer, IgniteTask igniteTask, Protos.TaskID taskId) {
        String cfgUrl = this.clusterProps.igniteConfigUrl() != null ? this.clusterProps.igniteConfigUrl() : this.resourceProvider.igniteConfigUrl();
        Protos.CommandInfo.Builder builder = Protos.CommandInfo.newBuilder().setEnvironment(Protos.Environment.newBuilder().addVariables(Protos.Environment.Variable.newBuilder().setName("IGNITE_TCP_DISCOVERY_ADDRESSES").setValue(this.getAddress(offer.getHostname()))).addVariables(Protos.Environment.Variable.newBuilder().setName("JVM_OPTS").setValue(this.clusterProps.jmvOpts()))).addUris(Protos.CommandInfo.URI.newBuilder().setValue(this.clusterProps.ignitePackageUrl() != null ? this.clusterProps.ignitePackageUrl() : this.resourceProvider.igniteUrl()).setExtract(true)).addUris(Protos.CommandInfo.URI.newBuilder().setValue(cfgUrl));
        ArrayList<String> usersLibs = new ArrayList<String>();
        if (this.clusterProps.usersLibsUrl() != null && !this.clusterProps.usersLibsUrl().isEmpty()) {
            Collections.addAll(usersLibs, this.clusterProps.usersLibsUrl().split(DELIM));
        }
        if (this.resourceProvider.resourceUrl() != null && !this.resourceProvider.resourceUrl().isEmpty()) {
            usersLibs.addAll(this.resourceProvider.resourceUrl());
        }
        for (String url : usersLibs) {
            builder.addUris(Protos.CommandInfo.URI.newBuilder().setValue(url));
        }
        String cfgName = this.resourceProvider.configName();
        if (this.clusterProps.igniteConfigUrl() != null) {
            cfgName = this.fileName(this.clusterProps.igniteConfigUrl());
        }
        String licenceFile = null;
        if (this.clusterProps.licenceUrl() != null) {
            licenceFile = this.fileName(this.clusterProps.licenceUrl());
        }
        builder.setValue((licenceFile != null ? "find . -maxdepth 1 -name \"" + licenceFile + "\" -exec cp {} ./*/ \\; && " : "") + "find . -maxdepth 1 -name \"*.jar\" -exec cp {} ./*/libs/ \\; && " + "./*/bin/ignite.sh " + cfgName + " -J-Xmx" + String.valueOf((int)igniteTask.mem() + "m") + " -J-Xms" + String.valueOf((int)igniteTask.mem()) + "m");
        return Protos.TaskInfo.newBuilder().setName("Ignite node " + taskId.getValue()).setTaskId(taskId).setSlaveId(offer.getSlaveId()).setCommand(builder).addResources(Protos.Resource.newBuilder().setName(CPU).setType(Protos.Value.Type.SCALAR).setScalar(Protos.Value.Scalar.newBuilder().setValue(igniteTask.cpuCores()))).addResources(Protos.Resource.newBuilder().setName(MEM).setType(Protos.Value.Type.SCALAR).setScalar(Protos.Value.Scalar.newBuilder().setValue(igniteTask.mem()))).addResources(Protos.Resource.newBuilder().setName(DISK).setType(Protos.Value.Type.SCALAR).setScalar(Protos.Value.Scalar.newBuilder().setValue(igniteTask.disk()))).build();
    }

    private String fileName(String path) {
        String[] split = path.split("/");
        return split[split.length - 1];
    }

    private String getAddress(String address) {
        if (this.tasks.isEmpty()) {
            if (address != null && !address.isEmpty()) {
                return address + DEFAULT_PORT;
            }
            return "";
        }
        StringBuilder sb = new StringBuilder();
        for (IgniteTask task : this.tasks.values()) {
            sb.append(task.host()).append(DEFAULT_PORT).append(DELIM);
        }
        return sb.substring(0, sb.length() - 1);
    }

    private IgniteTask checkOffer(Protos.Offer offer) {
        if (this.clusterProps.instances() <= (double)this.tasks.size()) {
            return null;
        }
        double cpus = -1.0;
        double mem = -1.0;
        double disk = -1.0;
        if (this.clusterProps.hostnameConstraint() != null && this.clusterProps.hostnameConstraint().matcher(offer.getHostname()).matches()) {
            return null;
        }
        for (Protos.Resource resource : offer.getResourcesList()) {
            if (resource.getName().equals(CPU)) {
                if (resource.getType().equals(Protos.Value.Type.SCALAR)) {
                    cpus = resource.getScalar().getValue();
                    continue;
                }
                log.log(Level.FINE, "Cpus resource was not a scalar: {0}" + resource.getType());
                continue;
            }
            if (resource.getName().equals(MEM)) {
                if (resource.getType().equals(Protos.Value.Type.SCALAR)) {
                    mem = resource.getScalar().getValue();
                    continue;
                }
                log.log(Level.FINE, "Mem resource was not a scalar: {0}", resource.getType());
                continue;
            }
            if (!resource.getName().equals(DISK)) continue;
            if (resource.getType().equals(Protos.Value.Type.SCALAR)) {
                disk = resource.getScalar().getValue();
                continue;
            }
            log.log(Level.FINE, "Disk resource was not a scalar: {0}", resource.getType());
        }
        if (cpus < this.clusterProps.minCpuPerNode() || mem < this.clusterProps.minMemoryPerNode()) {
            log.log(Level.FINE, "Offer not sufficient for slave request: {0}", offer.getResourcesList());
            return null;
        }
        double totalCpus = 0.0;
        double totalMem = 0.0;
        double totalDisk = 0.0;
        for (IgniteTask task : this.tasks.values()) {
            totalCpus += task.cpuCores();
            totalMem += task.mem();
            totalDisk += task.disk();
        }
        cpus = Math.min(this.clusterProps.cpus() - totalCpus, Math.min(cpus, this.clusterProps.cpusPerNode()));
        mem = Math.min(this.clusterProps.memory() - totalMem, Math.min(mem, this.clusterProps.memoryPerNode()));
        disk = Math.min(this.clusterProps.disk() - totalDisk, Math.min(disk, this.clusterProps.diskPerNode()));
        if (this.clusterProps.cpusPerNode() != Double.MAX_VALUE && this.clusterProps.cpusPerNode() != cpus || this.clusterProps.memoryPerNode() != Double.MAX_VALUE && this.clusterProps.memoryPerNode() != mem) {
            log.log(Level.FINE, "Offer not sufficient for slave request: {0}", offer.getResourcesList());
            return null;
        }
        if (cpus > 0.0 && mem > 0.0) {
            return new IgniteTask(offer.getHostname(), cpus, mem, disk);
        }
        log.log(Level.FINE, "Offer not sufficient for slave request: {0}", offer.getResourcesList());
        return null;
    }

    @Override
    public synchronized void statusUpdate(SchedulerDriver schedulerDriver, Protos.TaskStatus taskStatus) {
        IgniteTask failedTask;
        String taskId = taskStatus.getTaskId().getValue();
        log.log(Level.INFO, "Received update event task: {0} is in state: {1}", new Object[]{taskId, taskStatus.getState()});
        if ((taskStatus.getState().equals(Protos.TaskState.TASK_FAILED) || taskStatus.getState().equals(Protos.TaskState.TASK_ERROR) || taskStatus.getState().equals(Protos.TaskState.TASK_FINISHED) || taskStatus.getState().equals(Protos.TaskState.TASK_KILLED) || taskStatus.getState().equals(Protos.TaskState.TASK_LOST)) && (failedTask = this.tasks.remove(taskId)) != null) {
            ArrayList<Protos.Request> requests = new ArrayList<Protos.Request>();
            Protos.Request request = Protos.Request.newBuilder().addResources(Protos.Resource.newBuilder().setType(Protos.Value.Type.SCALAR).setName(MEM).setScalar(Protos.Value.Scalar.newBuilder().setValue(failedTask.mem()))).addResources(Protos.Resource.newBuilder().setType(Protos.Value.Type.SCALAR).setName(CPU).setScalar(Protos.Value.Scalar.newBuilder().setValue(failedTask.cpuCores()))).build();
            requests.add(request);
            schedulerDriver.requestResources(requests);
        }
    }

    public void setClusterProps(ClusterProperties clusterProps) {
        this.clusterProps = clusterProps;
    }

    @Override
    public void registered(SchedulerDriver schedulerDriver, Protos.FrameworkID frameworkID, Protos.MasterInfo masterInfo) {
        log.log(Level.INFO, "Scheduler registered. Master: {0}:{1}, framework={2}", new Object[]{masterInfo.getIp(), masterInfo.getPort(), frameworkID});
    }

    @Override
    public void disconnected(SchedulerDriver schedulerDriver) {
        log.info("Scheduler disconnected.");
    }

    @Override
    public void error(SchedulerDriver schedulerDriver, String s) {
        log.log(Level.SEVERE, "Failed. Error message: {0}", s);
    }

    @Override
    public void frameworkMessage(SchedulerDriver schedulerDriver, Protos.ExecutorID executorID, Protos.SlaveID slaveID, byte[] bytes) {
    }

    @Override
    public void slaveLost(SchedulerDriver schedulerDriver, Protos.SlaveID slaveID) {
    }

    @Override
    public void executorLost(SchedulerDriver schedulerDriver, Protos.ExecutorID executorID, Protos.SlaveID slaveID, int i) {
    }

    @Override
    public void offerRescinded(SchedulerDriver schedulerDriver, Protos.OfferID offerID) {
    }

    @Override
    public void reregistered(SchedulerDriver schedulerDriver, Protos.MasterInfo masterInfo) {
    }
}

