/*
 * Decompiled with CFR 0.152.
 */
package org.apache.ignite.internal.processors.cache.persistence.filename;

import java.io.File;
import java.io.FileFilter;
import java.io.IOException;
import java.io.Serializable;
import java.time.Instant;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.UUID;
import java.util.function.Function;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.ignite.IgniteCheckedException;
import org.apache.ignite.IgniteLogger;
import org.apache.ignite.IgniteSystemProperties;
import org.apache.ignite.configuration.DataStorageConfiguration;
import org.apache.ignite.configuration.IgniteConfiguration;
import org.apache.ignite.internal.processors.cache.persistence.FileLockHolder;
import org.apache.ignite.internal.processors.cache.persistence.filename.PdsFolderSettings;
import org.apache.ignite.internal.util.typedef.internal.A;
import org.apache.ignite.internal.util.typedef.internal.CU;
import org.apache.ignite.internal.util.typedef.internal.SB;
import org.apache.ignite.internal.util.typedef.internal.U;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

public class PdsFolderResolver<L extends FileLockHolder> {
    private static final String DB_FOLDER_PREFIX = "node";
    private static final String NODEIDX_UID_SEPARATOR = "-";
    private static final String NODE_PATTERN = "node[0-9]*-";
    private static final String UUID_STR_PATTERN = "[a-fA-F0-9]{8}-[a-fA-F0-9]{4}-[a-fA-F0-9]{4}-[a-fA-F0-9]{4}-[a-fA-F0-9]{12}";
    private static final String SUBDIR_PATTERN = "node[0-9]*-[a-fA-F0-9]{8}-[a-fA-F0-9]{4}-[a-fA-F0-9]{4}-[a-fA-F0-9]{4}-[a-fA-F0-9]{12}";
    public static final FileFilter DB_SUBFOLDERS_NEW_STYLE_FILTER = new FileFilter(){

        @Override
        public boolean accept(File pathname) {
            return pathname.isDirectory() && pathname.getName().matches(PdsFolderResolver.SUBDIR_PATTERN);
        }
    };
    private static final FileFilter DB_SUBFOLDERS_OLD_STYLE_FILTER = new FileFilter(){

        @Override
        public boolean accept(File pathname) {
            String path = pathname.toString();
            return pathname.isDirectory() && !"wal".equals(pathname.getName()) && !path.contains("db/binary_meta") && !path.contains("db/marshaller") && !pathname.getName().matches(PdsFolderResolver.SUBDIR_PATTERN);
        }
    };
    public static final String DB_DEFAULT_FOLDER = "db";
    private final IgniteConfiguration cfg;
    private final IgniteLogger log;
    @Nullable
    private final Serializable consistentId;
    private final Function<File, L> tryLock;

    public PdsFolderResolver(IgniteConfiguration cfg, IgniteLogger log, @Nullable Serializable consistentId, Function<File, L> tryLock) {
        this.cfg = cfg;
        this.log = log;
        this.consistentId = consistentId;
        this.tryLock = tryLock;
    }

    private PdsFolderSettings<L> compatibleResolve(@Nullable File pstStoreBasePath, @NotNull Serializable consistentId) {
        if (this.cfg.getConsistentId() != null) {
            return new PdsFolderSettings(pstStoreBasePath, this.cfg.getConsistentId());
        }
        return new PdsFolderSettings(pstStoreBasePath, consistentId);
    }

    public PdsFolderSettings<L> resolve() throws IgniteCheckedException {
        String subFolder;
        FileLockHolder oldStyleFolderLockHolder;
        boolean clientMode = this.cfg.isClientMode() == Boolean.TRUE || this.cfg.isDaemon();
        File pstStoreBasePath = this.resolvePersistentStoreBasePath(clientMode);
        if (!CU.isPersistenceEnabled(this.cfg) && !CU.isCdcEnabled(this.cfg)) {
            return this.compatibleResolve(pstStoreBasePath, this.consistentId);
        }
        if (clientMode) {
            return new PdsFolderSettings(pstStoreBasePath, UUID.randomUUID());
        }
        if (IgniteSystemProperties.getBoolean("IGNITE_DATA_STORAGE_FOLDER_BY_CONSISTENT_ID", false)) {
            return this.compatibleResolve(pstStoreBasePath, this.consistentId);
        }
        if (this.cfg.getConsistentId() != null) {
            return new PdsFolderSettings(pstStoreBasePath, this.cfg.getConsistentId());
        }
        if (this.consistentId != null && (oldStyleFolderLockHolder = (FileLockHolder)this.tryLock.apply(new File(pstStoreBasePath, subFolder = U.maskForFileName(this.consistentId.toString())))) != null) {
            return new PdsFolderSettings<FileLockHolder>(pstStoreBasePath, subFolder, this.consistentId, oldStyleFolderLockHolder, true);
        }
        File[] oldStyleFolders = pstStoreBasePath.listFiles(DB_SUBFOLDERS_OLD_STYLE_FILTER);
        if (oldStyleFolders != null && oldStyleFolders.length != 0) {
            for (File folder : oldStyleFolders) {
                String path = this.getPathDisplayableInfo(folder);
                U.warn(this.log, "There is other non-empty storage folder under storage base directory [" + path + "]");
            }
        }
        for (FolderCandidate next : this.getNodeIndexSortedCandidates(pstStoreBasePath)) {
            FileLockHolder fileLockHolder = (FileLockHolder)this.tryLock.apply(next.subFolderFile());
            if (fileLockHolder == null) continue;
            if (this.log.isInfoEnabled()) {
                this.log.info("Successfully locked persistence storage folder [" + next.subFolderFile() + "]");
            }
            return new PdsFolderSettings<FileLockHolder>(pstStoreBasePath, next.subFolderFile().getName(), next.uuid(), fileLockHolder, false);
        }
        return null;
    }

    public PdsFolderSettings<L> generateNew() throws IgniteCheckedException {
        File pstStoreBasePath = this.resolvePersistentStoreBasePath(false);
        try (L rootDirLock = this.lockRootDirectory(pstStoreBasePath);){
            List<FolderCandidate> sortedCandidates = this.getNodeIndexSortedCandidates(pstStoreBasePath);
            int nodeIdx = sortedCandidates.isEmpty() ? 0 : sortedCandidates.get(sortedCandidates.size() - 1).nodeIndex() + 1;
            PdsFolderSettings<L> pdsFolderSettings = this.generateAndLockNewDbStorage(pstStoreBasePath, nodeIdx);
            return pdsFolderSettings;
        }
    }

    private static FolderParams folderSize(File dir) {
        FolderParams params = new FolderParams();
        PdsFolderResolver.visitFolder(dir, params);
        return params;
    }

    private static void visitFolder(File dir, FolderParams params) {
        for (File file : dir.listFiles()) {
            if (file.isDirectory()) {
                PdsFolderResolver.visitFolder(file, params);
                continue;
            }
            params.size = params.size + file.length();
            params.lastModified = Math.max(params.lastModified, dir.lastModified());
        }
    }

    @NotNull
    private String getPathDisplayableInfo(File folder) {
        SB res = new SB();
        res.a(this.getCanonicalPath(folder));
        res.a(", ");
        FolderParams params = PdsFolderResolver.folderSize(folder);
        res.a(params.size);
        res.a(" bytes, modified ");
        DateTimeFormatter formatter = DateTimeFormatter.ofPattern("MM/dd/yyyy hh:mm a").withZone(ZoneId.systemDefault());
        res.a(formatter.format(Instant.ofEpochMilli(params.lastModified)));
        res.a(" ");
        return res.toString();
    }

    @NotNull
    private String getCanonicalPath(File file) {
        try {
            return file.getCanonicalPath();
        }
        catch (IOException ignored) {
            return file.getAbsolutePath();
        }
    }

    private static String padStart(String str, int minLength, char padChar) {
        A.notNull(str, "String should not be empty");
        if (str.length() >= minLength) {
            return str;
        }
        SB sb = new SB(minLength);
        for (int i = str.length(); i < minLength; ++i) {
            sb.a(padChar);
        }
        sb.a(str);
        return sb.toString();
    }

    @NotNull
    private PdsFolderSettings<L> generateAndLockNewDbStorage(File pstStoreBasePath, int nodeIdx) throws IgniteCheckedException {
        UUID uuid = UUID.randomUUID();
        String consIdBasedFolder = PdsFolderResolver.genNewStyleSubfolderName(nodeIdx, uuid);
        File newRandomFolder = U.resolveWorkDirectory(pstStoreBasePath.getAbsolutePath(), consIdBasedFolder, false);
        FileLockHolder fileLockHolder = (FileLockHolder)this.tryLock.apply(newRandomFolder);
        if (fileLockHolder != null) {
            if (this.log.isInfoEnabled()) {
                this.log.info("Successfully created new persistent storage folder [" + newRandomFolder + "]");
            }
            return new PdsFolderSettings<FileLockHolder>(pstStoreBasePath, consIdBasedFolder, uuid, fileLockHolder, false);
        }
        throw new IgniteCheckedException("Unable to lock file generated randomly [" + newRandomFolder + "]");
    }

    @NotNull
    public static String genNewStyleSubfolderName(int nodeIdx, UUID uuid) {
        String uuidAsStr = uuid.toString();
        assert (uuidAsStr.matches(UUID_STR_PATTERN));
        String nodeIdxPadded = PdsFolderResolver.padStart(Integer.toString(nodeIdx), 2, '0');
        return DB_FOLDER_PREFIX + nodeIdxPadded + NODEIDX_UID_SEPARATOR + uuidAsStr;
    }

    @NotNull
    private L lockRootDirectory(File pstStoreBasePath) throws IgniteCheckedException {
        FileLockHolder rootDirLock;
        int retry = 0;
        while ((rootDirLock = (FileLockHolder)this.tryLock.apply(pstStoreBasePath)) == null) {
            if (retry > 600) {
                throw new IgniteCheckedException("Unable to start under DB storage path [" + pstStoreBasePath + "]. Lock is being held to root directory");
            }
            ++retry;
        }
        return (L)rootDirLock;
    }

    @Nullable
    private List<FolderCandidate> getNodeIndexSortedCandidates(File pstStoreBasePath) {
        File[] files = pstStoreBasePath.listFiles(DB_SUBFOLDERS_NEW_STYLE_FILTER);
        if (files == null) {
            return Collections.emptyList();
        }
        ArrayList<FolderCandidate> res = new ArrayList<FolderCandidate>();
        for (File file : files) {
            FolderCandidate candidate = this.parseFileName(file);
            if (candidate == null) continue;
            res.add(candidate);
        }
        Collections.sort(res, new Comparator<FolderCandidate>(){

            @Override
            public int compare(FolderCandidate c1, FolderCandidate c2) {
                return Integer.compare(c1.nodeIndex(), c2.nodeIndex());
            }
        });
        return res;
    }

    @Nullable
    private File resolvePersistentStoreBasePath(boolean clientMode) throws IgniteCheckedException {
        DataStorageConfiguration dsCfg = this.cfg.getDataStorageConfiguration();
        if (dsCfg == null) {
            return null;
        }
        String pstPath = clientMode ? null : dsCfg.getStoragePath();
        return U.resolveWorkDirectory(this.cfg.getWorkDirectory(), pstPath != null ? pstPath : DB_DEFAULT_FOLDER, false);
    }

    private FolderCandidate parseFileName(@NotNull File subFolderFile) {
        return PdsFolderResolver.parseSubFolderName(subFolderFile, this.log);
    }

    @Nullable
    public static FolderCandidate parseSubFolderName(@NotNull File subFolderFile, @NotNull IgniteLogger log) {
        String fileName = subFolderFile.getName();
        Matcher matcher = Pattern.compile(NODE_PATTERN).matcher(fileName);
        if (!matcher.find()) {
            return null;
        }
        int uidStart = matcher.end();
        try {
            String uid = fileName.substring(uidStart);
            UUID uuid = UUID.fromString(uid);
            String substring = fileName.substring(DB_FOLDER_PREFIX.length(), uidStart - NODEIDX_UID_SEPARATOR.length());
            int idx = Integer.parseInt(substring);
            return new FolderCandidate(subFolderFile, idx, uuid);
        }
        catch (Exception e) {
            U.warn(log, "Unable to parse new style file format from [" + subFolderFile.getAbsolutePath() + "]: " + e);
            return null;
        }
    }

    public static class FolderCandidate {
        private final File subFolderFile;
        private final int nodeIdx;
        private final UUID uuid;

        public FolderCandidate(File subFolderFile, int nodeIdx, UUID uuid) {
            this.subFolderFile = subFolderFile;
            this.nodeIdx = nodeIdx;
            this.uuid = uuid;
        }

        public int nodeIndex() {
            return this.nodeIdx;
        }

        public Serializable uuid() {
            return this.uuid;
        }

        public File subFolderFile() {
            return this.subFolderFile;
        }
    }

    private static class FolderParams {
        private long size;
        private long lastModified;

        private FolderParams() {
        }
    }
}

