/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.ignite.ml.math.distances;

import java.util.Arrays;
import java.util.Collection;
import org.apache.ignite.ml.math.primitives.vector.Vector;
import org.apache.ignite.ml.math.primitives.vector.impl.DenseVector;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import static org.junit.Assert.assertEquals;

/**
 * Evaluate JensenShannonDistance in multiple test datasets
 */
@RunWith(Parameterized.class)
public class JensenShannonDistanceTest {
  /** Precision. */
  private static final double PRECISION = 0.01;

  /** */
  @Parameterized.Parameters(name = "{0}")
  public static Collection<TestData> data() {
    return Arrays.asList(
        new TestData(
            new double[] {1.0, 0.0, 0.0},
            new double[] {0.0, 1.0, 0.0},
            2.0,
            1.0
        ),
        new TestData(
            new double[] {1.0, 0.0},
            new double[] {0.5, 0.5},
            Math.E,
            0.46
        ),
        new TestData(
            new double[] {1.0, 0.0, 0.0},
            new double[] {1.0, 0.5, 0.0},
            Math.E,
            0.36
        )
    );
  }

  /** */
  private final TestData testData;

  /** */
  public JensenShannonDistanceTest(TestData testData) {
    this.testData = testData;
  }

  /** */
  @Test
  public void testJensenShannonDistance() {
    DistanceMeasure distanceMeasure = new JensenShannonDistance(testData.base);

    assertEquals(testData.expRes,
        distanceMeasure.compute(testData.vectorA, testData.vectorB), PRECISION);
    assertEquals(testData.expRes,
        distanceMeasure.compute(testData.vectorA, testData.vectorB), PRECISION);
  }

  /** */
  private static class TestData {
    /** */
    public final Vector vectorA;

    /** */
    public final Vector vectorB;

    /** */
    public final Double expRes;

    /** */
    public final Double base;

    /** */
    private TestData(double[] vectorA, double[] vectorB, Double base, Double expRes) {
      this.vectorA = new DenseVector(vectorA);
      this.vectorB = new DenseVector(vectorB);
      this.base = base;
      this.expRes = expRes;
    }

    /** {@inheritDoc} */
    @Override public String toString() {
      return String.format("d(%s,%s;%s) = %s",
          Arrays.toString(vectorA.asArray()),
          Arrays.toString(vectorB.asArray()),
          base,
          expRes
      );
    }
  }
}
