/*
 * Decompiled with CFR 0.152.
 */
package org.apache.ignite.internal.processors.compress;

import com.github.luben.zstd.Zstd;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.file.Path;
import net.jpountz.lz4.LZ4Compressor;
import net.jpountz.lz4.LZ4Factory;
import net.jpountz.lz4.LZ4FastDecompressor;
import org.apache.ignite.IgniteCheckedException;
import org.apache.ignite.IgniteException;
import org.apache.ignite.configuration.DiskPageCompression;
import org.apache.ignite.internal.GridKernalContext;
import org.apache.ignite.internal.pagemem.PageUtils;
import org.apache.ignite.internal.processors.cache.persistence.tree.io.CompactablePageIO;
import org.apache.ignite.internal.processors.cache.persistence.tree.io.PageIO;
import org.apache.ignite.internal.processors.compress.CompressionProcessor;
import org.apache.ignite.internal.processors.compress.FileSystemUtils;
import org.apache.ignite.internal.util.GridUnsafe;
import org.apache.ignite.internal.util.typedef.internal.U;
import org.xerial.snappy.Snappy;

public class CompressionProcessorImpl
extends CompressionProcessor {
    private final ThreadLocalByteBuffer compactBuf = new ThreadLocalByteBuffer(16384);
    private final ThreadLocalByteBuffer compressBuf = new ThreadLocalByteBuffer(17408);

    public CompressionProcessorImpl(GridKernalContext ctx) {
        super(ctx);
    }

    static ByteBuffer allocateDirectBuffer(int cap) {
        return ByteBuffer.allocateDirect(cap).order(GridUnsafe.NATIVE_BYTE_ORDER);
    }

    public void checkPageCompressionSupported() throws IgniteCheckedException {
    }

    public void checkPageCompressionSupported(Path storagePath, int pageSize) throws IgniteCheckedException {
        if (!U.isLinux()) {
            throw new IgniteCheckedException("Currently page compression is supported only for Linux.");
        }
        FileSystemUtils.checkSupported();
        int fsBlockSize = FileSystemUtils.getFileSystemBlockSize((Path)storagePath);
        if (fsBlockSize <= 0) {
            throw new IgniteCheckedException("Failed to get file system block size: " + storagePath);
        }
        if (!U.isPow2((int)fsBlockSize)) {
            throw new IgniteCheckedException("Storage block size must be power of 2: " + fsBlockSize);
        }
        if (pageSize < fsBlockSize * 2) {
            throw new IgniteCheckedException("Page size (now configured to " + pageSize + " bytes) must be at least 2 times larger than the underlying storage block size (detected to be " + fsBlockSize + " bytes at '" + storagePath + "') for page compression.");
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public ByteBuffer compressPage(ByteBuffer page, int pageSize, int blockSize, DiskPageCompression compression, int compressLevel) throws IgniteCheckedException {
        assert (compression != null && compression != DiskPageCompression.DISABLED) : compression;
        assert (U.isPow2((int)blockSize)) : blockSize;
        assert (page.position() == 0 && page.limit() >= pageSize);
        int oldPageLimit = page.limit();
        try {
            page.limit(pageSize);
            ByteBuffer compactPage = this.doCompactPage(page, pageSize);
            int compactSize = compactPage.limit();
            assert (compactSize <= pageSize) : compactSize;
            if (compactSize < blockSize || compression == DiskPageCompression.SKIP_GARBAGE) {
                ByteBuffer byteBuffer = CompressionProcessorImpl.setCompactionInfo(compactPage, compactSize);
                return byteBuffer;
            }
            ByteBuffer compressedPage = this.doCompressPage(compression, compactPage, compactSize, compressLevel);
            assert (compressedPage.position() == 0);
            int freeCompactBlocks = (pageSize - compactSize) / blockSize;
            int compressedSize = compressedPage.limit();
            int freeCompressedBlocks = (pageSize - compressedSize) / blockSize;
            if (freeCompactBlocks >= freeCompressedBlocks) {
                if (freeCompactBlocks == 0) {
                    ByteBuffer byteBuffer = page;
                    return byteBuffer;
                }
                ByteBuffer byteBuffer = CompressionProcessorImpl.setCompactionInfo(compactPage, compactSize);
                return byteBuffer;
            }
            ByteBuffer byteBuffer = CompressionProcessorImpl.setCompressionInfo(compressedPage, compression, compressedSize, compactSize);
            return byteBuffer;
        }
        finally {
            page.limit(oldPageLimit);
        }
    }

    private ByteBuffer doCompactPage(ByteBuffer page, int pageSize) throws IgniteCheckedException {
        PageIO io = PageIO.getPageIO((ByteBuffer)page);
        ByteBuffer compactPage = this.compactBuf.get();
        if (io instanceof CompactablePageIO) {
            ((CompactablePageIO)io).compactPage(page, compactPage, pageSize);
        } else {
            if (page.isDirect()) {
                return page;
            }
            PageUtils.putBytes((long)GridUnsafe.bufferAddress((ByteBuffer)compactPage), (int)0, (byte[])page.array());
            compactPage.limit(pageSize);
        }
        return compactPage;
    }

    private static ByteBuffer setCompactionInfo(ByteBuffer page, int compactSize) {
        return CompressionProcessorImpl.setCompressionInfo(page, DiskPageCompression.SKIP_GARBAGE, compactSize, compactSize);
    }

    private static ByteBuffer setCompressionInfo(ByteBuffer page, DiskPageCompression compression, int compressedSize, int compactedSize) {
        assert (compressedSize >= 0 && compressedSize <= Short.MAX_VALUE) : compressedSize;
        assert (compactedSize >= 0 && compactedSize <= Short.MAX_VALUE) : compactedSize;
        PageIO.setCompressionType((ByteBuffer)page, (byte)CompressionProcessorImpl.getCompressionType(compression));
        PageIO.setCompressedSize((ByteBuffer)page, (short)((short)compressedSize));
        PageIO.setCompactedSize((ByteBuffer)page, (short)((short)compactedSize));
        return page;
    }

    private ByteBuffer doCompressPage(DiskPageCompression compression, ByteBuffer compactPage, int compactSize, int compressLevel) {
        switch (compression) {
            case ZSTD: {
                return this.compressPageZstd(compactPage, compactSize, compressLevel);
            }
            case LZ4: {
                return this.compressPageLz4(compactPage, compactSize, compressLevel);
            }
            case SNAPPY: {
                return this.compressPageSnappy(compactPage, compactSize);
            }
        }
        throw new IllegalStateException("Unsupported compression: " + compression);
    }

    private ByteBuffer compressPageLz4(ByteBuffer compactPage, int compactSize, int compressLevel) {
        LZ4Compressor compressor = Lz4.getCompressor(compressLevel);
        ByteBuffer compressedPage = this.compressBuf.get();
        CompressionProcessorImpl.copyPageHeader(compactPage, compressedPage, compactSize);
        compressor.compress(compactPage, compressedPage);
        compactPage.flip();
        compressedPage.flip();
        return compressedPage;
    }

    private ByteBuffer compressPageZstd(ByteBuffer compactPage, int compactSize, int compressLevel) {
        ByteBuffer compressedPage = this.compressBuf.get();
        CompressionProcessorImpl.copyPageHeader(compactPage, compressedPage, compactSize);
        Zstd.compress((ByteBuffer)compressedPage, (ByteBuffer)compactPage, (int)compressLevel);
        compactPage.flip();
        compressedPage.flip();
        return compressedPage;
    }

    private ByteBuffer compressPageSnappy(ByteBuffer compactPage, int compactSize) {
        ByteBuffer compressedPage = this.compressBuf.get();
        CompressionProcessorImpl.copyPageHeader(compactPage, compressedPage, compactSize);
        try {
            int compressedSize = Snappy.compress((ByteBuffer)compactPage, (ByteBuffer)compressedPage);
            assert (compressedPage.limit() == 40 + compressedSize);
        }
        catch (IOException e) {
            throw new IgniteException("Failed to compress page with Snappy.", (Throwable)e);
        }
        compactPage.position(0);
        compressedPage.position(0);
        return compressedPage;
    }

    private static void copyPageHeader(ByteBuffer compactPage, ByteBuffer compressedPage, int compactSize) {
        compactPage.limit(40);
        compressedPage.put(compactPage);
        compactPage.limit(compactSize);
    }

    private static byte getCompressionType(DiskPageCompression compression) {
        if (compression == DiskPageCompression.DISABLED) {
            return 0;
        }
        switch (compression) {
            case ZSTD: {
                return 2;
            }
            case LZ4: {
                return 3;
            }
            case SNAPPY: {
                return 4;
            }
            case SKIP_GARBAGE: {
                return 1;
            }
        }
        throw new IllegalStateException("Unexpected compression: " + compression);
    }

    public void decompressPage(ByteBuffer page, int pageSize) throws IgniteCheckedException {
        assert (page.capacity() >= pageSize) : "capacity=" + page.capacity() + ", pageSize=" + pageSize;
        byte compressType = PageIO.getCompressionType((ByteBuffer)page);
        if (compressType == 0) {
            return;
        }
        short compressedSize = PageIO.getCompressedSize((ByteBuffer)page);
        short compactSize = PageIO.getCompactedSize((ByteBuffer)page);
        assert (compactSize <= pageSize && compactSize >= compressedSize);
        if (compressType == 1) {
            page.position(0).limit(compactSize);
        } else {
            ByteBuffer dst = this.compressBuf.get();
            page.limit(compressedSize).position(40);
            dst.limit(compactSize - 40);
            switch (compressType) {
                case 2: {
                    Zstd.decompress((ByteBuffer)dst, (ByteBuffer)page);
                    dst.flip();
                    break;
                }
                case 3: {
                    Lz4.decompress(page, dst);
                    dst.flip();
                    break;
                }
                case 4: {
                    try {
                        Snappy.uncompress((ByteBuffer)page, (ByteBuffer)dst);
                        break;
                    }
                    catch (IOException e) {
                        throw new IgniteException((Throwable)e);
                    }
                }
                default: {
                    throw new IgniteException("Unknown compression: " + compressType);
                }
            }
            page.position(40).limit(compactSize);
            page.put(dst).flip();
            assert (page.limit() == compactSize);
        }
        PageIO io = PageIO.getPageIO((ByteBuffer)page);
        if (io instanceof CompactablePageIO) {
            ((CompactablePageIO)io).restorePage(page, pageSize);
        } else assert (compactSize == pageSize) : "Wrong compacted page size [compactSize=" + compactSize + ", pageSize=" + pageSize + ']';
        CompressionProcessorImpl.setCompressionInfo(page, DiskPageCompression.DISABLED, 0, 0);
    }

    static final class ThreadLocalByteBuffer
    extends ThreadLocal<ByteBuffer> {
        final int size;

        ThreadLocalByteBuffer(int size) {
            this.size = size;
        }

        @Override
        protected ByteBuffer initialValue() {
            return CompressionProcessorImpl.allocateDirectBuffer(this.size);
        }

        @Override
        public ByteBuffer get() {
            ByteBuffer buf = (ByteBuffer)super.get();
            buf.clear();
            return buf;
        }
    }

    static class Lz4 {
        static final LZ4Factory factory = LZ4Factory.fastestInstance();
        static final LZ4FastDecompressor decompressor = factory.fastDecompressor();
        static final LZ4Compressor fastCompressor = factory.fastCompressor();

        Lz4() {
        }

        static LZ4Compressor getCompressor(int level) {
            assert (level >= 0 && level <= 17) : level;
            return level == 0 ? fastCompressor : factory.highCompressor(level);
        }

        static void decompress(ByteBuffer page, ByteBuffer dst) {
            decompressor.decompress(page, dst);
        }
    }
}

