/*
 * ====================================================================
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 *
 */

package org.apache.hc.client5.http.psl;

import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.charset.StandardCharsets;
import java.util.List;

import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;

public class TestPublicSuffixMatcher {

    private static final String SOURCE_FILE = "suffixlistmatcher.txt";

    private PublicSuffixMatcher matcher;

    @Before
    public void setUp() throws Exception {
        final ClassLoader classLoader = getClass().getClassLoader();
        final InputStream in = classLoader.getResourceAsStream(SOURCE_FILE);
        Assert.assertNotNull(in);
        final List<PublicSuffixList> lists = new PublicSuffixListParser().parseByType(
                new InputStreamReader(in, StandardCharsets.UTF_8));
        matcher = new PublicSuffixMatcher(lists);
    }

    @Test
    public void testGetDomainRootAnyType() {
        // Private
        Assert.assertEquals("xx", matcher.getDomainRoot("example.XX"));
        Assert.assertEquals("xx", matcher.getDomainRoot("www.example.XX"));
        Assert.assertEquals("xx", matcher.getDomainRoot("www.blah.blah.example.XX"));
        Assert.assertEquals("appspot.com", matcher.getDomainRoot("example.appspot.com"));
        // Too short
        Assert.assertNull(matcher.getDomainRoot("jp"));
        Assert.assertNull(matcher.getDomainRoot("ac.jp"));
        Assert.assertNull(matcher.getDomainRoot("any.tokyo.jp"));
        // ICANN
        Assert.assertEquals("metro.tokyo.jp", matcher.getDomainRoot("metro.tokyo.jp"));
        Assert.assertEquals("blah.blah.tokyo.jp", matcher.getDomainRoot("blah.blah.tokyo.jp"));
        Assert.assertEquals("blah.ac.jp", matcher.getDomainRoot("blah.blah.ac.jp"));
        // Unknown
        Assert.assertEquals("garbage", matcher.getDomainRoot("garbage"));
        Assert.assertEquals("garbage", matcher.getDomainRoot("garbage.garbage"));
        Assert.assertEquals("garbage", matcher.getDomainRoot("*.garbage.garbage"));
        Assert.assertEquals("garbage", matcher.getDomainRoot("*.garbage.garbage.garbage"));
    }

    @Test
    public void testGetDomainRootOnlyPRIVATE() {
        // Private
        Assert.assertEquals("xx", matcher.getDomainRoot("example.XX", DomainType.PRIVATE));
        Assert.assertEquals("xx", matcher.getDomainRoot("www.example.XX", DomainType.PRIVATE));
        Assert.assertEquals("xx", matcher.getDomainRoot("www.blah.blah.example.XX", DomainType.PRIVATE));
        Assert.assertEquals("appspot.com", matcher.getDomainRoot("example.appspot.com"));
        // Too short
        Assert.assertNull(matcher.getDomainRoot("jp", DomainType.PRIVATE));
        Assert.assertNull(matcher.getDomainRoot("ac.jp", DomainType.PRIVATE));
        Assert.assertNull(matcher.getDomainRoot("any.tokyo.jp", DomainType.PRIVATE));
        // ICANN
        Assert.assertNull(matcher.getDomainRoot("metro.tokyo.jp", DomainType.PRIVATE));
        Assert.assertNull(matcher.getDomainRoot("blah.blah.tokyo.jp", DomainType.PRIVATE));
        Assert.assertNull(matcher.getDomainRoot("blah.blah.ac.jp", DomainType.PRIVATE));
        // Unknown
        Assert.assertNull(matcher.getDomainRoot("garbage", DomainType.PRIVATE));
        Assert.assertNull(matcher.getDomainRoot("garbage.garbage", DomainType.PRIVATE));
        Assert.assertNull(matcher.getDomainRoot("*.garbage.garbage", DomainType.PRIVATE));
        Assert.assertNull(matcher.getDomainRoot("*.garbage.garbage.garbage", DomainType.PRIVATE));
    }

    @Test
    public void testGetDomainRootOnlyICANN() {
        // Private
        Assert.assertNull(matcher.getDomainRoot("example.XX", DomainType.ICANN));
        Assert.assertNull(matcher.getDomainRoot("www.example.XX", DomainType.ICANN));
        Assert.assertNull(matcher.getDomainRoot("www.blah.blah.example.XX", DomainType.ICANN));
        // Too short
        Assert.assertNull(matcher.getDomainRoot("xx", DomainType.ICANN));
        Assert.assertNull(matcher.getDomainRoot("jp", DomainType.ICANN));
        Assert.assertNull(matcher.getDomainRoot("ac.jp", DomainType.ICANN));
        Assert.assertNull(matcher.getDomainRoot("any.tokyo.jp", DomainType.ICANN));
        // ICANN
        Assert.assertEquals("metro.tokyo.jp", matcher.getDomainRoot("metro.tokyo.jp", DomainType.ICANN));
        Assert.assertEquals("blah.blah.tokyo.jp", matcher.getDomainRoot("blah.blah.tokyo.jp", DomainType.ICANN));
        Assert.assertEquals("blah.ac.jp", matcher.getDomainRoot("blah.blah.ac.jp", DomainType.ICANN));
        // Unknown
        Assert.assertNull(matcher.getDomainRoot("garbage", DomainType.ICANN));
        Assert.assertNull(matcher.getDomainRoot("garbage.garbage", DomainType.ICANN));
        Assert.assertNull(matcher.getDomainRoot("*.garbage.garbage", DomainType.ICANN));
        Assert.assertNull(matcher.getDomainRoot("*.garbage.garbage.garbage", DomainType.ICANN));
    }


    @Test
    public void testMatch() {
        Assert.assertTrue(matcher.matches(".jp"));
        Assert.assertTrue(matcher.matches(".ac.jp"));
        Assert.assertTrue(matcher.matches(".any.tokyo.jp"));
        // exception
        Assert.assertFalse(matcher.matches(".metro.tokyo.jp"));
        Assert.assertFalse(matcher.matches(".xx"));
        Assert.assertFalse(matcher.matches(".appspot.com"));
    }

    @Test
    public void testMatchUnicode() {
        Assert.assertTrue(matcher.matches(".h\u00E5.no")); // \u00E5 is <aring>
        Assert.assertTrue(matcher.matches(".xn--h-2fa.no"));
        Assert.assertTrue(matcher.matches(".h\u00E5.no"));
        Assert.assertTrue(matcher.matches(".xn--h-2fa.no"));
    }

}
