/*
 * Licensed to the Apache Software Foundation (ASF) under one or more contributor license
 * agreements. See the NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The ASF licenses this file to You under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance with the License. You may obtain a
 * copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under the License
 * is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express
 * or implied. See the License for the specific language governing permissions and limitations under
 * the License.
 */
package org.apache.geode.cache.lucene.internal.cli.functions;

import static org.junit.Assert.*;
import static org.mockito.Mockito.*;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;

import org.apache.lucene.analysis.Analyzer;
import org.apache.lucene.analysis.core.KeywordAnalyzer;
import org.apache.lucene.analysis.standard.StandardAnalyzer;
import org.junit.Test;
import org.junit.experimental.categories.Category;
import org.mockito.ArgumentCaptor;
import org.mockito.Mockito;

import org.apache.geode.cache.execute.FunctionContext;
import org.apache.geode.cache.execute.ResultSender;
import org.apache.geode.cache.lucene.LuceneIndex;
import org.apache.geode.cache.lucene.internal.InternalLuceneService;
import org.apache.geode.cache.lucene.internal.LuceneIndexImpl;
import org.apache.geode.cache.lucene.internal.LuceneServiceImpl;
import org.apache.geode.cache.lucene.internal.cli.LuceneIndexDetails;
import org.apache.geode.internal.cache.GemFireCacheImpl;
import org.apache.geode.test.fake.Fakes;
import org.apache.geode.test.junit.categories.UnitTest;

@Category(UnitTest.class)

public class LuceneListIndexFunctionJUnitTest {

  @Test
  @SuppressWarnings("unchecked")
  public void testExecute() throws Throwable {
    GemFireCacheImpl cache = Fakes.cache();
    final String serverName = "mockServer";
    LuceneServiceImpl service = mock(LuceneServiceImpl.class);
    when(cache.getService(InternalLuceneService.class)).thenReturn(service);

    FunctionContext context = mock(FunctionContext.class);
    ResultSender resultSender = mock(ResultSender.class);
    when(context.getResultSender()).thenReturn(resultSender);
    when(context.getCache()).thenReturn(cache);

    LuceneIndexImpl index1 = getMockLuceneIndex("index1");
    LuceneIndexImpl index2 = getMockLuceneIndex("index2");

    TreeSet expectedResult = new TreeSet();
    expectedResult.add(new LuceneIndexDetails(index1, serverName));
    expectedResult.add(new LuceneIndexDetails(index2, serverName));

    ArrayList<LuceneIndex> allIndexes = new ArrayList();
    allIndexes.add(index1);
    allIndexes.add(index2);
    when(service.getAllIndexes()).thenReturn(allIndexes);

    LuceneListIndexFunction function = new LuceneListIndexFunction();
    function.execute(context);

    ArgumentCaptor<Set> resultCaptor = ArgumentCaptor.forClass(Set.class);
    verify(resultSender).lastResult(resultCaptor.capture());
    Set<String> result = resultCaptor.getValue();

    assertEquals(2, result.size());
    assertEquals(expectedResult, result);
  }

  private LuceneIndexImpl getMockLuceneIndex(final String indexName) {
    String[] searchableFields = {"field1", "field2"};
    Map<String, Analyzer> fieldAnalyzers = new HashMap<>();
    fieldAnalyzers.put("field1", new StandardAnalyzer());
    fieldAnalyzers.put("field2", new KeywordAnalyzer());

    LuceneIndexImpl index = mock(LuceneIndexImpl.class);
    when(index.getName()).thenReturn(indexName);
    when(index.getRegionPath()).thenReturn("/region");
    when(index.getFieldNames()).thenReturn(searchableFields);
    when(index.getFieldAnalyzers()).thenReturn(fieldAnalyzers);
    return index;
  }

}
