/*
 * Licensed to the Apache Software Foundation (ASF) under one or more contributor license
 * agreements. See the NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The ASF licenses this file to You under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance with the License. You may obtain a
 * copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under the License
 * is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express
 * or implied. See the License for the specific language governing permissions and limitations under
 * the License.
 */
package org.apache.geode.distributed.internal;

import static java.util.concurrent.TimeUnit.MINUTES;
import static org.apache.geode.distributed.ConfigurationProperties.MCAST_PORT;
import static org.apache.geode.distributed.internal.DefaultPropertiesGenerator.getDefaultFileName;
import static org.apache.geode.internal.lang.SystemUtils.getClassPath;
import static org.assertj.core.api.Assertions.assertThat;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.util.Properties;

import org.junit.After;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.experimental.categories.Category;
import org.junit.rules.TemporaryFolder;
import org.junit.rules.TestName;

import org.apache.geode.cache.Cache;
import org.apache.geode.cache.CacheFactory;
import org.apache.geode.test.junit.categories.IntegrationTest;

@Category(IntegrationTest.class)
public class DefaultPropertiesGeneratorIntegrationTest {

  private Process process;
  private String propertiesFile;
  private Cache cache;

  @Rule
  public TemporaryFolder temporaryFolder = new TemporaryFolder();

  @Rule
  public TestName testName = new TestName();

  @Before
  public void before() throws Exception {
    String tmp = this.temporaryFolder.getRoot().getAbsolutePath();
    this.propertiesFile = tmp + "gf" + System.nanoTime() + ".properties";
    DefaultPropertiesGenerator generator = new DefaultPropertiesGenerator();
    generator.generateDefaultPropertiesFile(this.propertiesFile);
    assertThat(new File(this.propertiesFile)).exists();
  }

  @After
  public void after() throws Exception {
    if (this.cache != null) {
      this.cache.close();
    }
    if (this.process != null) {
      this.process.destroyForcibly();
    }
  }

  @Test
  public void propertiesShouldNotBeEmpty() throws Exception {
    Properties properties = loadProperties();
    assertThat(properties).isNotEmpty();
  }

  /**
   * test that the gemfire.properties generated by default is able to start a server
   */
  @Test
  public void propertiesShouldCreateValidCache() throws Exception {
    Properties properties = loadProperties();
    CacheFactory cacheFactory = new CacheFactory(properties);
    this.cache = cacheFactory.create();
    assertThat(this.cache).isNotNull();
  }

  @Test
  public void shouldGeneratePropertiesFile() throws Exception {
    String[] command = {"java", "-cp", getClassPath(), DefaultPropertiesGenerator.class.getName()};
    ProcessBuilder launcher = new ProcessBuilder(command).directory(this.temporaryFolder.getRoot());
    this.process = launcher.start();
    this.process.waitFor(2, MINUTES);

    String tmp = this.temporaryFolder.getRoot().getAbsolutePath();
    File file = new File(tmp + File.separator + getDefaultFileName());
    assertThat(file).exists();

    Properties properties = loadProperties();
    assertThat(properties).isNotEmpty();
  }

  @Test
  public void shouldUseSpecifiedPropertiesFile() throws Exception {
    String targetFileName = "gf" + this.testName.getMethodName() + ".properties";
    String[] command =
        {"java", "-cp", getClassPath(), DefaultPropertiesGenerator.class.getName(), targetFileName};
    ProcessBuilder launcher = new ProcessBuilder(command).directory(this.temporaryFolder.getRoot());
    this.process = launcher.start();
    this.process.waitFor(2, MINUTES);

    String tmp = this.temporaryFolder.getRoot().getAbsolutePath();
    File file = new File(tmp + File.separator + targetFileName);
    assertThat(file).exists();

    Properties properties = loadProperties();
    assertThat(properties).isNotEmpty();
  }

  private Properties loadProperties() throws IOException {
    Properties properties = new Properties();
    properties.load(new FileInputStream(this.propertiesFile));
    assertThat(properties.getProperty(MCAST_PORT)).isEqualTo("0");
    properties.setProperty(MCAST_PORT, "0");
    return properties;
  }
}
