/*
 * Licensed to the Apache Software Foundation (ASF) under one or more contributor license
 * agreements. See the NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The ASF licenses this file to You under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance with the License. You may obtain a
 * copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under the License
 * is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express
 * or implied. See the License for the specific language governing permissions and limitations under
 * the License.
 */
package org.apache.geode.test.golden;

import static org.junit.Assert.*;

import org.junit.Test;
import org.junit.experimental.categories.Category;

import org.apache.geode.test.junit.categories.IntegrationTest;
import org.apache.geode.test.process.ProcessWrapper;

/**
 * Verifies that test output containing an unexpected extra line will fail with that line as the
 * failure message.
 *
 */
@Category(IntegrationTest.class)
public class FailWithExtraLineInOutputJUnitTest extends FailOutputTestCase {

  @Override
  String problem() {
    return "This is an extra line";
  }

  @Override
  void outputProblemInProcess(final String message) {
    System.out.println(message);
  }

  /**
   * Process output has an extra line and should fail
   */
  @Test
  public void testFailWithExtraLineInOutput() throws Exception {
    final String goldenString = "Begin " + name() + ".main" + "\n" + "Press Enter to continue."
        + "\n" + "End " + name() + ".main" + "\n";
    debug(goldenString, "GOLDEN");

    final ProcessWrapper process = createProcessWrapper(new ProcessWrapper.Builder(), getClass());
    process.execute(createProperties());
    process.waitForOutputToMatch("Begin " + name() + "\\.main");
    process.waitForOutputToMatch("Press Enter to continue\\.");
    process.sendInput();
    process.waitForOutputToMatch("End " + name() + "\\.main");
    process.waitFor();

    try {
      assertOutputMatchesGoldenFile(process.getOutput(), goldenString);
      fail("assertOutputMatchesGoldenFile should have failed due to " + problem());
    } catch (AssertionError expected) {
      assertTrue(expected.getMessage().contains(problem()));
    }
  }

  public static void main(final String[] args) throws Exception {
    new FailWithExtraLineInOutputJUnitTest().executeInProcess();
  }
}
