/*
 * Licensed to the Apache Software Foundation (ASF) under one or more contributor license
 * agreements. See the NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The ASF licenses this file to You under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance with the License. You may obtain a
 * copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under the License
 * is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express
 * or implied. See the License for the specific language governing permissions and limitations under
 * the License.
 */
package org.apache.geode.security;

import org.apache.commons.lang.StringUtils;
import org.apache.shiro.authz.permission.WildcardPermission;

import org.apache.geode.cache.Region;

/**
 * ResourcePermission defines the resource, the operation, the region and the key involved in the
 * action to be authorized.
 *
 * It is passed to the SecurityManager for the implementation to decide whether to grant a user this
 * permission or not.
 */
public class ResourcePermission extends WildcardPermission {

  public static String ALL = "*";

  /**
   * @deprecated use ALL
   */
  public static String ALL_REGIONS = "*";
  /**
   * @deprecated use All
   */
  public static String ALL_KEYS = "*";

  public enum Resource {
    NULL, CLUSTER, DATA
  }

  public enum Operation {
    NULL, MANAGE, WRITE, READ
  }

  // when ALL is specified, we need it to convert to string "*" instead of "ALL".
  public enum Target {
    ALL(ResourcePermission.ALL), DISK, GATEWAY, QUERY, DEPLOY;

    private String name;

    Target() {}

    Target(String name) {
      this.name = name;
    }

    public String getName() {
      if (name != null) {
        return name;
      }
      return name();
    }
  }

  // these default values are used when creating a lock around an operation
  private Resource resource = Resource.NULL;
  private Operation operation = Operation.NULL;
  private String target = ALL;
  private String key = ALL;

  public ResourcePermission() {
    this(Resource.NULL, Operation.NULL, ALL, ALL);
  }

  public ResourcePermission(String resource, String operation) {
    this(resource, operation, ALL, ALL);
  }

  public ResourcePermission(String resource, String operation, String target) {
    this(resource, operation, target, ALL);
  }

  public ResourcePermission(String resource, String operation, String target, String key) {
    this((resource == null) ? Resource.NULL : Resource.valueOf(resource.toUpperCase()),
        (operation == null) ? Operation.NULL : Operation.valueOf(operation.toUpperCase()), target,
        key);
  }

  public ResourcePermission(Resource resource, Operation operation) {
    this(resource, operation, ALL, ALL);
  }

  public ResourcePermission(Resource resource, Operation operation, String target) {
    this(resource, operation, target, ALL);
  }

  public ResourcePermission(Resource resource, Operation operation, Target target) {
    this(resource, operation, target, ALL);
  }

  public ResourcePermission(Resource resource, Operation operation, Target target,
      String targetKey) {
    this(resource, operation, target.getName(), targetKey);
  }

  public ResourcePermission(Resource resource, Operation operation, String target, String key) {
    if (resource != null) {
      this.resource = resource;
    }
    if (operation != null) {
      this.operation = operation;
    }
    if (target != null) {
      this.target = StringUtils.stripStart(target, Region.SEPARATOR);
    }
    if (key != null) {
      this.key = key;
    }

    setParts(this.resource + ":" + this.operation + ":" + this.target + ":" + this.key, true);
  }

  /**
   * Returns the resource, could be either DATA or CLUSTER
   */
  public Resource getResource() {
    return resource;
  }

  /**
   * Returns the operation, could be either MANAGE, WRITE or READ
   */
  public Operation getOperation() {
    return operation;
  }

  /**
   * returns the regionName, or cluster target, could be "*", meaning all regions or all targets
   */
  public String getTarget() {
    return target;
  }

  /**
   * @deprecated use getTarget()
   */
  public String getRegionName() {
    return getTarget();
  }

  /**
   * returns the key, could be "*" meaning all keys.
   */
  public String getKey() {
    return key;
  }

  @Override
  public String toString() {
    if (ALL.equals(target)) {
      return getResource() + ":" + getOperation();
    } else if (ALL.equals(key)) {
      return resource + ":" + operation + ":" + target;
    } else {
      return resource + ":" + operation + ":" + target + ":" + key;
    }
  }

}
