/*
 * Decompiled with CFR 0.152.
 */
package org.apache.commons.statistics.descriptive;

import java.math.BigInteger;
import java.util.Objects;
import java.util.Set;
import java.util.function.DoubleConsumer;
import java.util.function.Function;
import java.util.function.IntConsumer;
import org.apache.commons.statistics.descriptive.FirstMoment;
import org.apache.commons.statistics.descriptive.GeometricMean;
import org.apache.commons.statistics.descriptive.Int128;
import org.apache.commons.statistics.descriptive.IntMax;
import org.apache.commons.statistics.descriptive.IntMean;
import org.apache.commons.statistics.descriptive.IntMin;
import org.apache.commons.statistics.descriptive.IntSum;
import org.apache.commons.statistics.descriptive.IntSumOfSquares;
import org.apache.commons.statistics.descriptive.IntVariance;
import org.apache.commons.statistics.descriptive.Kurtosis;
import org.apache.commons.statistics.descriptive.Product;
import org.apache.commons.statistics.descriptive.Skewness;
import org.apache.commons.statistics.descriptive.Statistic;
import org.apache.commons.statistics.descriptive.StatisticResult;
import org.apache.commons.statistics.descriptive.Statistics;
import org.apache.commons.statistics.descriptive.StatisticsConfiguration;
import org.apache.commons.statistics.descriptive.SumOfCubedDeviations;
import org.apache.commons.statistics.descriptive.SumOfFourthDeviations;
import org.apache.commons.statistics.descriptive.SumOfLogs;
import org.apache.commons.statistics.descriptive.UInt128;

public final class IntStatistics
implements IntConsumer {
    private static final String NO_CONFIGURED_STATISTICS = "No configured statistics";
    private static final String UNSUPPORTED_STATISTIC = "Unsupported statistic: ";
    private long count;
    private final IntConsumer consumer;
    private final IntMin min;
    private final IntMax max;
    private final FirstMoment moment;
    private final IntSum sum;
    private final Product product;
    private final IntSumOfSquares sumOfSquares;
    private final SumOfLogs sumOfLogs;
    private StatisticsConfiguration config;

    IntStatistics(long count, IntMin min, IntMax max, FirstMoment moment, IntSum sum, Product product, IntSumOfSquares sumOfSquares, SumOfLogs sumOfLogs, StatisticsConfiguration config) {
        this.count = count;
        this.min = min;
        this.max = max;
        this.moment = moment;
        this.sum = sum;
        this.product = product;
        this.sumOfSquares = sumOfSquares;
        this.sumOfLogs = sumOfLogs;
        this.config = config;
        this.consumer = Statistics.compose(min, max, sum, sumOfSquares, IntStatistics.composeAsInt(moment, product, sumOfLogs));
    }

    private static IntConsumer composeAsInt(DoubleConsumer ... consumers) {
        DoubleConsumer c = Statistics.compose(consumers);
        if (c != null) {
            return c::accept;
        }
        return null;
    }

    public static IntStatistics of(Statistic ... statistics) {
        return IntStatistics.builder(statistics).build();
    }

    public static IntStatistics of(Set<Statistic> statistics, int ... values) {
        if (statistics.isEmpty()) {
            throw new IllegalArgumentException(NO_CONFIGURED_STATISTICS);
        }
        Builder b = new Builder();
        statistics.forEach(b::add);
        return b.build(values);
    }

    public static Builder builder(Statistic ... statistics) {
        if (statistics.length == 0) {
            throw new IllegalArgumentException(NO_CONFIGURED_STATISTICS);
        }
        Builder b = new Builder();
        for (Statistic s : statistics) {
            b.add(s);
        }
        return b;
    }

    @Override
    public void accept(int value) {
        ++this.count;
        this.consumer.accept(value);
    }

    public long getCount() {
        return this.count;
    }

    public boolean isSupported(Statistic statistic) {
        switch (statistic) {
            case GEOMETRIC_MEAN: 
            case SUM_OF_LOGS: {
                return this.sumOfLogs != null;
            }
            case KURTOSIS: {
                return this.moment instanceof SumOfFourthDeviations;
            }
            case MAX: {
                return this.max != null;
            }
            case MIN: {
                return this.min != null;
            }
            case PRODUCT: {
                return this.product != null;
            }
            case SKEWNESS: {
                return this.moment instanceof SumOfCubedDeviations;
            }
            case STANDARD_DEVIATION: 
            case VARIANCE: {
                return this.sum != null && this.sumOfSquares != null;
            }
            case MEAN: 
            case SUM: {
                return this.sum != null;
            }
            case SUM_OF_SQUARES: {
                return this.sumOfSquares != null;
            }
        }
        return false;
    }

    public double getAsDouble(Statistic statistic) {
        return this.getResult(statistic).getAsDouble();
    }

    public int getAsInt(Statistic statistic) {
        return this.getResult(statistic).getAsInt();
    }

    public long getAsLong(Statistic statistic) {
        return this.getResult(statistic).getAsLong();
    }

    public BigInteger getAsBigInteger(Statistic statistic) {
        return this.getResult(statistic).getAsBigInteger();
    }

    public StatisticResult getResult(Statistic statistic) {
        StatisticResult stat = null;
        switch (statistic) {
            case GEOMETRIC_MEAN: {
                stat = this.getGeometricMean();
                break;
            }
            case KURTOSIS: {
                stat = this.getKurtosis();
                break;
            }
            case MAX: {
                stat = Statistics.getResultAsIntOrNull(this.max);
                break;
            }
            case MEAN: {
                stat = this.getMean();
                break;
            }
            case MIN: {
                stat = Statistics.getResultAsIntOrNull(this.min);
                break;
            }
            case PRODUCT: {
                stat = Statistics.getResultAsDoubleOrNull(this.product);
                break;
            }
            case SKEWNESS: {
                stat = this.getSkewness();
                break;
            }
            case STANDARD_DEVIATION: {
                stat = this.getStandardDeviation();
                break;
            }
            case SUM: {
                stat = Statistics.getResultAsBigIntegerOrNull(this.sum);
                break;
            }
            case SUM_OF_LOGS: {
                stat = Statistics.getResultAsDoubleOrNull(this.sumOfLogs);
                break;
            }
            case SUM_OF_SQUARES: {
                stat = Statistics.getResultAsBigIntegerOrNull(this.sumOfSquares);
                break;
            }
            case VARIANCE: {
                stat = this.getVariance();
                break;
            }
        }
        if (stat != null) {
            return stat;
        }
        throw new IllegalArgumentException(UNSUPPORTED_STATISTIC + (Object)((Object)statistic));
    }

    private StatisticResult getGeometricMean() {
        if (this.sumOfLogs != null) {
            return () -> GeometricMean.computeGeometricMean(this.count, this.sumOfLogs);
        }
        return null;
    }

    private StatisticResult getKurtosis() {
        if (this.moment instanceof SumOfFourthDeviations) {
            return new Kurtosis((SumOfFourthDeviations)this.moment).setBiased(this.config.isBiased())::getAsDouble;
        }
        return null;
    }

    private StatisticResult getMean() {
        if (this.sum != null) {
            Int128 s = this.sum.getSum();
            return () -> IntMean.computeMean(s, this.count);
        }
        return null;
    }

    private StatisticResult getSkewness() {
        if (this.moment instanceof SumOfCubedDeviations) {
            return new Skewness((SumOfCubedDeviations)this.moment).setBiased(this.config.isBiased())::getAsDouble;
        }
        return null;
    }

    private StatisticResult getStandardDeviation() {
        return this.getVarianceOrStd(true);
    }

    private StatisticResult getVariance() {
        return this.getVarianceOrStd(false);
    }

    private StatisticResult getVarianceOrStd(boolean std) {
        if (this.sum != null && this.sumOfSquares != null) {
            Int128 s = this.sum.getSum();
            UInt128 ss = this.sumOfSquares.getSumOfSquares();
            boolean biased = this.config.isBiased();
            return () -> IntVariance.computeVarianceOrStd(ss, s, this.count, biased, std);
        }
        return null;
    }

    public IntStatistics combine(IntStatistics other) {
        Statistics.checkCombineCompatible(this.min, other.min);
        Statistics.checkCombineCompatible(this.max, other.max);
        Statistics.checkCombineCompatible(this.sum, other.sum);
        Statistics.checkCombineCompatible(this.product, other.product);
        Statistics.checkCombineCompatible(this.sumOfSquares, other.sumOfSquares);
        Statistics.checkCombineCompatible(this.sumOfLogs, other.sumOfLogs);
        Statistics.checkCombineAssignable(this.moment, other.moment);
        this.count += other.count;
        Statistics.combine(this.min, other.min);
        Statistics.combine(this.max, other.max);
        Statistics.combine(this.sum, other.sum);
        Statistics.combine(this.product, other.product);
        Statistics.combine(this.sumOfSquares, other.sumOfSquares);
        Statistics.combine(this.sumOfLogs, other.sumOfLogs);
        Statistics.combineMoment(this.moment, other.moment);
        return this;
    }

    public IntStatistics setConfiguration(StatisticsConfiguration v) {
        this.config = Objects.requireNonNull(v);
        return this;
    }

    public static final class Builder {
        private static final int[] NO_VALUES = new int[0];
        private Function<int[], IntMin> min;
        private Function<int[], IntMax> max;
        private Function<int[], FirstMoment> moment;
        private Function<int[], IntSum> sum;
        private Function<int[], Product> product;
        private Function<int[], IntSumOfSquares> sumOfSquares;
        private Function<int[], SumOfLogs> sumOfLogs;
        private int momentOrder;
        private StatisticsConfiguration config = StatisticsConfiguration.withDefaults();

        Builder() {
        }

        Builder add(Statistic statistic) {
            switch (statistic) {
                case GEOMETRIC_MEAN: 
                case SUM_OF_LOGS: {
                    this.sumOfLogs = SumOfLogs::of;
                    break;
                }
                case KURTOSIS: {
                    this.createMoment(4);
                    break;
                }
                case MAX: {
                    this.max = IntMax::of;
                    break;
                }
                case MIN: {
                    this.min = IntMin::of;
                    break;
                }
                case PRODUCT: {
                    this.product = Product::of;
                    break;
                }
                case SKEWNESS: {
                    this.createMoment(3);
                    break;
                }
                case STANDARD_DEVIATION: 
                case VARIANCE: {
                    this.sum = IntSum::of;
                    this.sumOfSquares = IntSumOfSquares::of;
                    break;
                }
                case MEAN: 
                case SUM: {
                    this.sum = IntSum::of;
                    break;
                }
                case SUM_OF_SQUARES: {
                    this.sumOfSquares = IntSumOfSquares::of;
                    break;
                }
                default: {
                    throw new IllegalArgumentException(IntStatistics.UNSUPPORTED_STATISTIC + (Object)((Object)statistic));
                }
            }
            return this;
        }

        private void createMoment(int order) {
            if (order > this.momentOrder) {
                this.momentOrder = order;
                this.moment = order == 4 ? SumOfFourthDeviations::of : SumOfCubedDeviations::of;
            }
        }

        public Builder setConfiguration(StatisticsConfiguration v) {
            this.config = Objects.requireNonNull(v);
            return this;
        }

        public IntStatistics build() {
            return this.build(NO_VALUES);
        }

        public IntStatistics build(int ... values) {
            Objects.requireNonNull(values, "values");
            return new IntStatistics(values.length, Builder.create(this.min, values), Builder.create(this.max, values), Builder.create(this.moment, values), Builder.create(this.sum, values), Builder.create(this.product, values), Builder.create(this.sumOfSquares, values), Builder.create(this.sumOfLogs, values), this.config);
        }

        private static <T> T create(Function<int[], T> constructor, int[] values) {
            if (constructor != null) {
                return constructor.apply(values);
            }
            return null;
        }
    }
}

