/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.clerezza.rdf.stable.serializer;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.util.Arrays;
import java.util.Collection;

import junit.framework.Assert;

import org.apache.clerezza.commons.rdf.ImmutableGraph;
import org.apache.clerezza.commons.rdf.Graph;
import org.apache.clerezza.commons.rdf.impl.utils.simple.SimpleGraph;
import org.apache.clerezza.rdf.core.serializedform.Parser;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

/**
 *
 * @author daniel
 */
@RunWith(Parameterized.class)
public class NTriplesSerializerTest {

    private String inputFileName;
    private String format;

    public NTriplesSerializerTest(String inputFileName, String format) {
        this.inputFileName = inputFileName;
        this.format = format;
    }

    @Parameterized.Parameters
    public static Collection<String[]> inputFileNames() {
        return Arrays.asList(new String[][] {
            {"amp-in-url-test001.rdf", "application/rdf+xml"},
            {"datatypes-test001.rdf", "application/rdf+xml"},
            {"datatypes-test002.rdf", "application/rdf+xml"},
            {"rdf-charmod-literals-test001.rdf", "application/rdf+xml"},
            {"rdf-charmod-uris-test001.rdf", "application/rdf+xml"},
            {"rdf-charmod-uris-test002.rdf", "application/rdf+xml"},
            {"xml-canon-test001.rdf", "application/rdf+xml"},
            {"css3deps.rdf", "application/rdf+xml"},
            {"agenda_62.rdf", "application/rdf+xml"},
            {"Talks.rdf", "application/rdf+xml"},
            {"elvisimp.rdf", "application/rdf+xml"},
            //{"images.xrdf", "application/rdf+xml"}, //large
            {"libby.foaf", "application/rdf+xml"},
            {"documentation-example.nt", "application/n-triples"}
        });
    }

    @Test
    public void RDFTestCases() {
        NTriplesSerializer nts = new NTriplesSerializer();

        Parser parser = Parser.getInstance();
        ImmutableGraph deserializedGraphOld = parser.parse(
                getClass().getResourceAsStream(inputFileName), format);

        Graph tc = new SimpleGraph();
        tc.addAll(deserializedGraphOld);

        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        nts.serialize(baos, tc);

        ByteArrayInputStream bais = new ByteArrayInputStream(baos.toByteArray());

        ImmutableGraph deserializedGraphNew = parser.parse(bais, "application/n-triples");

        Assert.assertEquals(deserializedGraphNew, deserializedGraphOld);
    }

}
