/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.camel.model.language;

import javax.xml.bind.annotation.XmlAccessType;
import javax.xml.bind.annotation.XmlAccessorType;
import javax.xml.bind.annotation.XmlAttribute;
import javax.xml.bind.annotation.XmlRootElement;
import javax.xml.bind.annotation.XmlTransient;

import org.apache.camel.spi.Metadata;

/**
 * Evaluate an XQuery expressions against an XML payload.
 */
@Metadata(firstVersion = "1.0.0", label = "language,xml", title = "XQuery")
@XmlRootElement(name = "xquery")
@XmlAccessorType(XmlAccessType.FIELD)
public class XQueryExpression extends NamespaceAwareExpression {
    @XmlAttribute
    private String type;
    @XmlTransient
    private Class<?> resultType;
    @XmlAttribute
    private String headerName;
    @XmlAttribute
    @Metadata(label = "advanced")
    private String configurationRef;
    @XmlTransient
    private Object configuration;

    public XQueryExpression() {
    }

    public XQueryExpression(String expression) {
        super(expression);
    }

    @Override
    public String getLanguage() {
        return "xquery";
    }

    public String getType() {
        return type;
    }

    /**
     * Sets the class name of the result type (type from output)
     * <p/>
     * The default result type is NodeSet
     */
    public void setType(String type) {
        this.type = type;
    }

    public Class<?> getResultType() {
        return resultType;
    }

    /**
     * Sets the class of the result type (type from output).
     * <p/>
     * The default result type is NodeSet
     */
    public void setResultType(Class<?> resultType) {
        this.resultType = resultType;
    }

    public String getHeaderName() {
        return headerName;
    }

    /**
     * Name of header to use as input, instead of the message body
     */
    public void setHeaderName(String headerName) {
        this.headerName = headerName;
    }

    public String getConfigurationRef() {
        return configurationRef;
    }

    /**
     * Reference to a saxon configuration instance in the registry to use for xquery (requires camel-saxon). This may be
     * needed to add custom functions to a saxon configuration, so these custom functions can be used in xquery
     * expressions.
     */
    public void setConfigurationRef(String configurationRef) {
        this.configurationRef = configurationRef;
    }

    public Object getConfiguration() {
        return configuration;
    }

    /**
     * Custom saxon configuration (requires camel-saxon). This may be needed to add custom functions to a saxon
     * configuration, so these custom functions can be used in xquery expressions.
     */
    public void setConfiguration(Object configuration) {
        this.configuration = configuration;
    }
}
