/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.camel.component.aws.swf;

import java.util.Arrays;

import com.amazonaws.services.simpleworkflow.flow.worker.GenericActivityWorker;
import org.apache.camel.Exchange;
import org.apache.camel.Processor;
import org.apache.camel.support.DefaultConsumer;
import org.apache.camel.util.URISupport;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SWFActivityConsumer extends DefaultConsumer {
    private static final transient Logger LOGGER = LoggerFactory.getLogger(SWFWorkflowProducer.class);
    private SWFEndpoint endpoint;
    private final SWFConfiguration configuration;
    private GenericActivityWorker genericWorker;

    private transient String swfActivityConsumerToString;

    public SWFActivityConsumer(SWFEndpoint endpoint, Processor processor, SWFConfiguration configuration) {
        super(endpoint, processor);
        this.endpoint = endpoint;
        this.configuration = configuration;
    }

    public Object processActivity(Object[] inputParameters, String taskToken) throws Exception {
        LOGGER.debug("Processing activity task: " + Arrays.toString(inputParameters));

        Exchange exchange = endpoint.createExchange(inputParameters, SWFConstants.EXECUTE_ACTION);
        exchange.getIn().setHeader(SWFConstants.TASK_TOKEN, taskToken);

        getProcessor().process(exchange);
        return endpoint.getResult(exchange);
    }

    @Override
    protected void doStart() throws Exception {
        CamelActivityImplementationFactory factory = new CamelActivityImplementationFactory(this, configuration);
        genericWorker = new GenericActivityWorker(
                endpoint.getSWClient(), configuration.getDomainName(), configuration.getActivityList());
        genericWorker.setActivityImplementationFactory(factory);
        genericWorker.setTaskExecutorThreadPoolSize(configuration.getActivityThreadPoolSize());
        genericWorker.start();
        super.doStart();
    }

    @Override
    protected void doStop() throws Exception {
        genericWorker.setDisableServiceShutdownOnStop(true);
        genericWorker.shutdownNow();
        super.doStop();
    }

    @Override
    public String toString() {
        if (swfActivityConsumerToString == null) {
            swfActivityConsumerToString = "SWFActivityConsumer[" + URISupport.sanitizeUri(getEndpoint().getEndpointUri()) + "]";
        }
        return swfActivityConsumerToString;
    }
}
