/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.camel.component.atomix.client.value;

import java.util.Collections;
import java.util.Map;
import java.util.UUID;

import io.atomix.variables.DistributedValue;
import org.apache.camel.Component;
import org.apache.camel.EndpointInject;
import org.apache.camel.FluentProducerTemplate;
import org.apache.camel.Message;
import org.apache.camel.RoutesBuilder;
import org.apache.camel.builder.RouteBuilder;
import org.apache.camel.component.atomix.client.AtomixClientConstants;
import org.apache.camel.component.atomix.client.AtomixClientTestSupport;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.Test;

import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertFalse;
import static org.junit.jupiter.api.Assertions.assertTrue;

public class AtomixValueProducerTest extends AtomixClientTestSupport {
    private static final String VALUE_NAME = UUID.randomUUID().toString();
    private DistributedValue<Object> value;

    @EndpointInject("direct:start")
    private FluentProducerTemplate fluent;

    // ************************************
    // Setup
    // ************************************

    @Override
    protected Map<String, Component> createComponents() {
        AtomixValueComponent component = new AtomixValueComponent();
        component.setNodes(Collections.singletonList(getReplicaAddress()));

        return Collections.singletonMap("atomix-value", component);
    }

    @Override
    protected void doPreSetup() throws Exception {
        super.doPreSetup();

        value = getClient().getValue(VALUE_NAME).join();
    }

    @Override
    @AfterEach
    public void tearDown() throws Exception {
        value.close();

        super.tearDown();
    }

    // ************************************
    // Test
    // ************************************

    @Test
    void test() {
        final String val1 = context().getUuidGenerator().generateUuid();
        final String val2 = context().getUuidGenerator().generateUuid();

        Message result;

        result = fluent.clearAll()
                .withHeader(AtomixClientConstants.RESOURCE_ACTION, AtomixValue.Action.SET)
                .withBody(val1)
                .request(Message.class);

        assertFalse(result.getHeader(AtomixClientConstants.RESOURCE_ACTION_HAS_RESULT, Boolean.class));
        assertEquals(val1, value.get().join());

        result = fluent.clearAll()
                .withHeader(AtomixClientConstants.RESOURCE_ACTION, AtomixValue.Action.COMPARE_AND_SET)
                .withHeader(AtomixClientConstants.RESOURCE_OLD_VALUE, val1)
                .withBody(val2)
                .request(Message.class);

        assertTrue(result.getHeader(AtomixClientConstants.RESOURCE_ACTION_HAS_RESULT, Boolean.class));
        assertEquals(val2, value.get().join());

        result = fluent.clearAll()
                .withHeader(AtomixClientConstants.RESOURCE_ACTION, AtomixValue.Action.GET)
                .request(Message.class);

        assertTrue(result.getHeader(AtomixClientConstants.RESOURCE_ACTION_HAS_RESULT, Boolean.class));
        assertEquals(val2, result.getBody());
        assertEquals(val2, value.get().join());

        result = fluent.clearAll()
                .withHeader(AtomixClientConstants.RESOURCE_ACTION, AtomixValue.Action.GET_AND_SET)
                .withBody(val1)
                .request(Message.class);

        assertTrue(result.getHeader(AtomixClientConstants.RESOURCE_ACTION_HAS_RESULT, Boolean.class));
        assertEquals(val2, result.getBody());
        assertEquals(val1, value.get().join());
    }

    // ************************************
    // Routes
    // ************************************

    @Override
    protected RoutesBuilder createRouteBuilder() {
        return new RouteBuilder() {
            public void configure() {
                from("direct:start")
                        .toF("atomix-value:%s", VALUE_NAME);
            }
        };
    }
}
