/*
 * Decompiled with CFR 0.152.
 */
package org.apache.camel.component.hdfs;

import java.io.Closeable;
import java.io.IOException;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.atomic.AtomicLong;
import org.apache.camel.RuntimeCamelException;
import org.apache.camel.component.hdfs.HdfsConfiguration;
import org.apache.camel.component.hdfs.HdfsFileType;
import org.apache.camel.component.hdfs.HdfsInfo;
import org.apache.camel.component.hdfs.HdfsInfoFactory;
import org.apache.camel.component.hdfs.Holder;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.io.IOUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class HdfsInputStream
implements Closeable {
    private static final Logger LOG = LoggerFactory.getLogger(HdfsInputStream.class);
    private HdfsFileType fileType;
    private HdfsInfo info;
    private String actualPath;
    private String suffixedPath;
    private String suffixedReadPath;
    private Closeable in;
    private boolean opened;
    private int chunkSize;
    private final AtomicLong numOfReadBytes = new AtomicLong(0L);
    private final AtomicLong numOfReadMessages = new AtomicLong(0L);
    private boolean streamDownload;
    private EntryHolder cachedNextEntry;

    protected HdfsInputStream() {
    }

    public static HdfsInputStream createInputStream(String hdfsPath, HdfsInfoFactory hdfsInfoFactory) {
        HdfsConfiguration endpointConfig = hdfsInfoFactory.getEndpointConfig();
        HdfsInputStream iStream = new HdfsInputStream();
        iStream.fileType = endpointConfig.getFileType();
        iStream.actualPath = hdfsPath;
        iStream.suffixedPath = iStream.actualPath + '.' + endpointConfig.getOpenedSuffix();
        iStream.suffixedReadPath = iStream.actualPath + '.' + endpointConfig.getReadSuffix();
        iStream.chunkSize = endpointConfig.getChunkSize();
        iStream.streamDownload = endpointConfig.isStreamDownload();
        try {
            iStream.info = hdfsInfoFactory.newHdfsInfo(iStream.actualPath);
            if (iStream.info.getFileSystem().rename(new Path(iStream.actualPath), new Path(iStream.suffixedPath))) {
                iStream.in = iStream.fileType.createInputStream(iStream.suffixedPath, hdfsInfoFactory);
                iStream.opened = true;
            } else {
                LOG.debug("Failed to open file [{}] because it doesn't exist", (Object)hdfsPath);
                iStream = null;
            }
        }
        catch (IOException e) {
            throw new RuntimeCamelException((Throwable)e);
        }
        return iStream;
    }

    @Override
    public final void close() throws IOException {
        if (this.opened) {
            IOUtils.closeStream((Closeable)this.in);
            this.info.getFileSystem().rename(new Path(this.suffixedPath), new Path(this.suffixedReadPath));
            this.opened = false;
        }
    }

    public final long next(Holder<Object> key, Holder<Object> value) {
        EntryHolder nextEntry = Optional.ofNullable(this.cachedNextEntry).orElseGet(() -> this.getNextFromStream(key, value));
        this.cachedNextEntry = null;
        key.setValue(nextEntry.getKey().getValue());
        value.setValue(nextEntry.getValue().getValue());
        return nextEntry.getByteCount();
    }

    private EntryHolder getNextFromStream(Holder<Object> key, Holder<Object> value) {
        long nb = this.fileType.next(this, key, value);
        if (nb == 0L && this.numOfReadMessages.get() > 0L || Objects.isNull(value.getValue())) {
            nb = -1L;
        } else {
            this.numOfReadBytes.addAndGet(nb);
            this.numOfReadMessages.incrementAndGet();
        }
        return new EntryHolder(key, value, nb);
    }

    public final boolean hasNext() {
        if (Objects.isNull(this.cachedNextEntry)) {
            Holder<Object> nextKey = new Holder<Object>();
            Holder<Object> nextValue = new Holder<Object>();
            long nextByteCount = this.next(nextKey, nextValue);
            this.cachedNextEntry = new EntryHolder(nextKey, nextValue, nextByteCount);
        }
        return this.cachedNextEntry.hasNext();
    }

    public final long getNumOfReadBytes() {
        return this.numOfReadBytes.longValue();
    }

    public final long getNumOfReadMessages() {
        return this.numOfReadMessages.longValue();
    }

    public final String getActualPath() {
        return this.actualPath;
    }

    public final int getChunkSize() {
        return this.chunkSize;
    }

    public final Closeable getIn() {
        return this.in;
    }

    public boolean isOpened() {
        return this.opened;
    }

    public boolean isStreamDownload() {
        return this.streamDownload;
    }

    private static class EntryHolder {
        private long byteCount;
        private Holder<Object> key;
        private Holder<Object> value;

        public EntryHolder(Holder<Object> key, Holder<Object> value, long byteCount) {
            this.key = key;
            this.value = value;
            this.byteCount = byteCount;
        }

        public Holder<Object> getKey() {
            return this.key;
        }

        public Holder<Object> getValue() {
            return this.value;
        }

        public Boolean hasNext() {
            return this.byteCount >= 0L;
        }

        public long getByteCount() {
            return this.byteCount;
        }
    }
}

