/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.camel.component.jms;

import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.io.Reader;
import java.io.StringReader;
import java.nio.ByteBuffer;

import org.apache.camel.builder.RouteBuilder;
import org.apache.camel.component.mock.MockEndpoint;
import org.junit.jupiter.api.Test;

/**
 * Unit test that we send payload as byte[] for certain types
 */
public class PayloadByteArrayJmsTest extends AbstractJMSTest {

    protected final String componentName = "activemq";

    @Test
    public void testReaderShouldBeByteArray() throws Exception {
        MockEndpoint mock = getMockEndpoint("mock:result");
        mock.expectedMessageCount(1);
        mock.message(0).body().isInstanceOf(byte[].class);
        mock.message(0).body(String.class).isEqualTo("Hello World");

        Reader body = new StringReader("Hello World");

        template.sendBody("activemq:queue:PayloadByteArrayJmsTest", body);

        assertMockEndpointsSatisfied();
    }

    @Test
    public void testInputStreamShouldBeByteArray() throws Exception {
        MockEndpoint mock = getMockEndpoint("mock:result");
        mock.expectedMessageCount(1);
        mock.message(0).body().isInstanceOf(byte[].class);
        mock.message(0).body(String.class).isEqualTo("Hello World");

        InputStream body = new ByteArrayInputStream("Hello World".getBytes());

        template.sendBody("activemq:queue:PayloadByteArrayJmsTest", body);

        assertMockEndpointsSatisfied();
    }

    @Test
    public void testByteBufferShouldBeByteArray() throws Exception {
        MockEndpoint mock = getMockEndpoint("mock:result");
        mock.expectedMessageCount(1);
        mock.message(0).body().isInstanceOf(byte[].class);
        mock.message(0).body(String.class).isEqualTo("Hello World");

        ByteBuffer body = ByteBuffer.wrap("Hello World".getBytes());

        template.sendBody("activemq:queue:PayloadByteArrayJmsTest", body);

        assertMockEndpointsSatisfied();
    }

    @Override
    public String getComponentName() {
        return componentName;
    }

    @Override
    protected RouteBuilder createRouteBuilder() {
        return new RouteBuilder() {
            public void configure() {
                from("activemq:queue:PayloadByteArrayJmsTest").to("mock:result");
            }
        };
    }
}
